/**
 * @file      tripapi.h
 * @brief     C prototype header for the TRIPsystem kernel API
 * @copyright DVG Operations GmbH 2024 - all rights reserved
 */

#ifndef __TRIPAPI_PROTOTYPES__
#define __TRIPAPI_PROTOTYPES__

#include <stdio.h>
#include <time.h>

/* Macros for compiler-specific deprecation signaling */
#ifdef __cplusplus
#if __cplusplus >= 201300
#define TDB_DEPRECATED(msg) [[deprecated(msg)]]
#endif
#endif
#ifndef TDB_DEPRECATED
#ifdef _MSC_VER
#define TDB_DEPRECATED(msg) __declspec(deprecated(msg))
#elif defined(__GNUC__) || defined(__clang__)
#define TDB_DEPRECATED(msg) __attribute__((deprecated))
#else
#define TDB_DEPRECATED(msg)
#endif
#endif

/* Include the kernel structure and constant definitions required */
#include "tripapi_defs.h"
#include "tripapi_types.h"

#if defined(__cplusplus)
extern "C" {
#endif

   /**
    * @defgroup session Session Commands
    * @brief Manipulation of session state
    *
    * The session commands is a group of TRIP functions that operate on the
    * session level. These are used to establish and tear down sessions and
    * query and modify session-related information.
    */


   /**
    * @brief Enables or disables API logging for the current session.
    * @param enable True to enable and false to disable
    * @ingroup session
    *
    * #### Description
    * API logging can be useful to analyze the behavior of applications in
    * situations where hard-to-locate errors occur.
    *
    * API logs enabled this way are automatically removed when the process
    * exits and do not show in the file system. If this log data should be
    * retained, either call the TdbApiLogRead() function to read the log
    * from your application, or define the \p TDBS_APILOG_RETAIN
    * variable in tdbs.conf or the environment before you start the TRIP
    * process.
    *
    * #### Return Codes
    * On success, this function returns a success value of 1, and on failure
    * it returns 0 (zero).
    *
    * @see TdbApiLogCheckpoint()
    * @see TdbApiLogRead()
    */
   int TdbApiLogEnable(boolean enable);

   /**
    * @brief Sets an API log checkpoint position.
    * @param id ID of checkpoint to set; valid ID numbers are 1 to 8.
    * @ingroup session
    *
    * #### Description
    * API log checkpoints are used to set a position from which calls to
    * the TdbApiLogRead() function will start to read the log. Use this
    * behavior when you wish to get the API log for a specific part of
    * your application only.
    *
    * #### Return Codes
    * On success, this function returns a success value of 1, and on failure
    * it returns 0 (zero).
    *
    * @see TdbApiLogEnable()
    * @see TdbApiLogRead()
    */
   int TdbApiLogCheckpoint(int id);

   /**
    * @brief Read data from the API log
    * @param id ID of checkpoint to read from
    * @param position [IN/OUT] Pointer to long int to hold read position; set to zero for first call.
    * @param buffer [OUT] Pointer to memory buffer to receieve API log data.
    * @param bufsize [IN] The size of the memory area that the \p buffer parameter points to.
    * @param readcount [OUT] Receieves the number of bytes copied to the \p buffer
    * @ingroup session
    *
    * #### Description
    * The API log is, if enabled via the TdbApiLogEnable() function, by
    * default not visible in the file system. In order to read data from
    * it, the application should call this function.
    *
    * This function reads the API log from a position associated with a
    * checkpoint, set by calling the TdbApiLogCheckpoint() function. A
    * special checkpoint with value 0 (zero) can also be used in order to
    * read data from the start of the log.
    *
    * #### Return Codes
    * On success, this function returns a success value of 1, and on failure
    * it returns 0 (zero).
    */
   int TdbApiLogRead(int checkpoint_id, long* position, char* buffer,
                     size_t bufsize, size_t* readcount);

   /**
    * @brief Set verbosity level for the API log
    * @param level Verbosity level to set
    * @ingroup session
    *
    * #### Description
    * The API log verbosity can be set by calling this method. Verbosity
    * levels control what API functions are included the log. The default
    * level is used for most API calls, and while labeled minimal, it is
    * sufficient for most needs. Functions whose impact on the log size
    * is significant and/or whose inclusion are rarely required are
    * emitted at higher verbosity levels.
    *
    * levels are:
    *
    * | Value | Description |
    * | ----_ | ----------- |
    * | 0 | Minimal verbosity (default) |
    * | 1 | Low verbosity |
    * | 2 | Average verbosity |
    * | 3 | High verbosity |
    * | 4 | Max verbosity |
    *
    * #### Return Codes
    * On success, this function returns a success value of 1, and on failure
    * it returns 0 (zero).
    */
   int TdbApiLogSetVerbosity(int level);
   
   /**
    * @brief Checks that a supplied date is valid and transforms date strings
    *        to integers and vice versa
    * @param dateform Date format
    * @param datestring Date string
    * @param dateint Datee numeric value.
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The TdbCheckDate() function performs a number of operations upon TRIP
    * dates depending on the value of the parameters passed to it.
    *
    * If the \p dateform parameter value is positive (between 1 and 17), the
    * \p datestring parameter is not an empty string, and the \p dateint
    * parameter is 0 (zero), this function will check that the \p datestring
    * parameter value is valid in that format.
    *
    * If the \p dateform parameter value is positive (between 1 and 17), the
    * \p datestring parameter is an empty string and the \p dateint parameter
    * is not 0 (zero), this function transforms the value of the \p dateint
    * parameter from a \c long representation of a date to a string form in the
    * specified dateform.
    *
    * The change to the date format lasts only for the duration of the current
    * call to the TdbCheckDate() function. However, if the value of the
    * \p dateform parameter is negative, the function performs all actions
    * described above but will also alter the date format until the calling
    * application terminates.
    *
    * #### Parameter Values
    * The \p dateform parameter is passed an \c int value for the dateform
    * that is to be checked or set. Valid values are:
    *
    * | Value | Date format | Example for May 1st 1988 |
    * | ---- | ---- | ---- |
    * | 1 | YYYY-MM-DD | 1988-05-01 |
    * | 2 | YYYY-mM-dD | 1988-5-1 |
    * | 3 | yyYY-mM-dD | 88-5-1 |
    * | 4 | YYYY-mm-dD | 1988-May-1 |
    * | 5 | yyYY-mm-dD | 88-May-1 |
    * | 6 | MM-DD-YYYY | 05-01-1988 |
    * | 7 | mM-dD-YYYY | 5-1-1988 |
    * | 8 | mM-dD-yyYY | 5-1-88 |
    * | 9 | mm-dD-YYYY | May-1-1988 |
    * | 10 | mm-dD-yyYY | May-1-88 |
    * | 11 | DD-MM-YYYY | 01-05-1988 |
    * | 12 | dD-mM-YYYY | 1-5-1988 |
    * | 13 | dD-mM-yyYY | 1-5-88 |
    * | 14 | dD-mm-YYYY | 1-May-1988 |
    * | 15 | dD-mm-yyYY | 1-May-88 |
    * | 16 | YYYYMMDD | 19880501 |
    * | 17 | yyYYMMDD | 880501 |
    *
    * The \p datestring parameter is passed a character string as the date in
    * string form. Whether this date string is to checked or set depends upon
    * the value passed in the \p dateint parameter.
    *
    * The \p dateint parameter is passed an integer representation of the
    * date. As with the \p parameter whether this value is checked or set
    * depends upon the \p dateform parameter.
    *
    * #### Return Codes
    * On success, this function returns a success value of 1, and on failure
    * it returns 0 (zero).
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | The date transformation failed. |
    *
    * #### Examples
    * Format a specified date according in the current date format:
    *
    * \code{.cpp}
    char formattedDate[64] = "";
    int year = 2004;
    int month = 9;
    int day = 29;
    
    // Create an integer representation of the date.
    int ymd = year << 9;
    ymd += month << 5;
    ymd += day;
   
    // Convert to string and display.
    TdbCheckDate(0,formattedDate,&ymd);
    printf("%s\n",formattedDate);
    * \endcode
    * Check if a specified string is a valid date in the current date format:
    *
    * \code{.cpp}
    int ymd = 0;
    char dateString[64] = "2004-09-29";
   
    // Check if the date string is a valid date.
    if (IsError(TdbCheckDate(1,dateString,&ymd))
        printf("String not recognized as a date in the current date format.\n");
    else
        printf("Date is ok. Integer representation of the date is %d\n",ymd);
    * \endcode
    */
   int TdbCheckDate(int dateform, char* datestring, int* dateint);

   
   /**
    * @brief Perform a login operation to the TRIPkernel by validating both
    *        username and password.
    * @param user Name of user. Recieves the normalized user name on return.
    * @param password Password
    * @param FM Receieves file manager privilege
    * @param UM Receieves user manager privilege
    * @param SM Receieves system manager privilege
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The function TdbCheckUser() must be called to gain access to TRIP. All
    * logins are validated upon the contents of the \p username and \p password
    * parameters.
    *
    * @note A call to this function is now included in the TdbStartTrip()
    * function except when TdbStartTrip() is called with an empty string as
    * username and as password. In this case the application must call
    * TdbCheckUser() afterwards.
    *
    * If the user's profile record has the \c os_entry flag set then the
    * password can be ignored if the operating system username is the same as
    * the TRIP username. When the two usernames are the same the string in the
    * \p password parameter can be left blank or set to a random value. Note
    * that this only applies for the server-side version of this function.
    *
    * Before calling TdbCheckUser() the calling process must have called the
    * function TdbBeginTdbs() to initialize the TRIP kernel. Until a successful
    * call of TdbCheckUser() the security subsystem will not allow any access
    * to database functions.
    *
    * This function can only be successfully invoked once per TRIP session; it
    * can, however, be invoked any number of times unsuccessfully prior to that
    * successful invocation.
    *
    * On successful return, the three boolean flags FM , UM , and SM reflect
    * the file, user and system management status of the calling process. These
    * flags are returned in the parameters with the same names.
    *
    * The SIF file for the current TRIP application session is created upon
    * successful checking of the CONTROL file for the user/password
    * combination.
    *
    * #### Parameter Values
    * The \p user parameter is passed a character string as the name of the
    * user being checked. Upon return from TdbCheckUser() the normalized form
    * of the user name is returned in this parameter.
    *
    * The \p password is passed a character string as the password for the user
    * being checked.
    *
    * The \p FM parameter is passed a char as a flag indicating whether the
    * named user has FM rights. If the lowest bit of the char is set upon
    * return the user has file manager privileges and the calling process may
    *  perform file management functions such as creating databases, clusters
    * and thesauri, as well as granting existing users and groups access to
    * those objects.
    *
    * The \p UM parameter is passed a char as a flag indicating whether the
    * named user has UM rights. If the lowest bit of the char is set upon
    *  return the user has user manager privileges and the calling process can
    * perform user management functions such as creation and deletion of users,
    * groups, and group-wide procedures.
    *
    * The \p SM parameter is passed a char as a flag indicating whether the
    * named user has SM rights. If the lowest bit of the char is set upon
    * return the user is acting as the system manager, user SYSTEM and the
    * calling process can perform system management functions such as changing
    * other user's passwords, and transferring ownership of objects from other
    * managers to itself.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 322 | Cannot open *.SIF, disk quota exceeded or no write access. |  |
    * | 2434 | Restart file 1 not found. |  |
    * | 2466 | Password invalid for restart file. |  |
    * | 9634 | No such user as 2 |  |
    * | 9666 | User id 2 is a group id. |  |
    * | 10306 | Invalid password - Please try again. |  |
    * | 18594 | Old version of restart file (SIF, PRT, RIF), aborting... |  |
    * | 21602 | Existing restart file belongs to another user - try again. |  |
    *
    * @see TdbCreateUser()
    * @see TdbChangePassword()
    * @see TdbSetPassword()
    * @see TdbStartTrip()
    * @see TdbBeginTdbs()
    */
   int TdbCheckUser(const char* user, const char* password, char* FM, char* UM, char* SM);

   /**
    * @brief Change the currently logged on user, or temporarily logs out the current user.
    * @param user Name of new user to log on as
    * @param password The password for the TRIP user
    * @param mode Bitmask controlling how existing session state is to be handled
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * This function is used to switch the current session from one user to another. 
    * The \p mode parameter controls how to handle the current session state; to
    * clear it or to keep it. See the table below for details.
    *
    * If the username and password are both specified as empty strings, the current
    * user will be logged off. The session will then be left in an intermediary 
    * state in which the only valid operation is a call to TdbChangeUser() to 
    * log in a new user into the session, or to call TdbStopTrip() to terminate the
    * session altogether.
    *
    * If the username is empty and the password is specified as a valid login ticket,
    * the ticket owner will be assigned the current user of the session.
    *
    * The \p mode parameter can have the following values:
    * | Symbol | Description |
    * | ------ | ----------- |
    * | CHANGEUSER_KEEPALL | All state is kept. Cannot be combined with any other value. |
    * | CHANGEUSER_CLEAR_SETS | Delete all search sets. This also removes all held PRINT jobs. |
    * | CHANGEUSER_CLOSE_DATABASES | Close all databases. |
    * | CHANGEUSER_RESTORE_DEFAULTS | Restore DEFINEs to their default values. |
    * | CHANGEUSER_CLEARALL | Combines CHANGEUSER_CLEAR_SETS, CHANGEUSER_CLOSE_DATABASES and CHANGEUSER_RESTORE_DEFAULTS. |
    *
    * The TdbSessionInfo() function can be used to query the current state of the session.
    * Use the SESSIONINFO_SESSION_STATUS mode parameter. If the integer output parameter
    * contains 4 or 5, the current user has been temporarily logged out.
    * 
    * #### Since
    * This function was introduced in TRIP version 8.0-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 322 | Cannot open *.SIF, disk quota exceeded or no write access. |  |
    * | 9634 | No such user as 2 |  |
    * | 9666 | User id 2 is a group id. |  |
    * | 10306 | Invalid password - Please try again. |  |
    *
    * @see TdbStartTrip()
    * @see TdbStopTrip()
    * @see TdbSessionInfo()
    */
   int TdbChangeUser(const char* user, const char* password, int mode);

   /**
    * @brief Change the session character set
    * @param charset Character set to use
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The valid values for the \p charset parameter for this function are:
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | CHSET_NONE | 0 | Default character set. |
    * | CHSET_LA1 | 1 | Latin 1 |
    * | CHSET_UTF8 | 16 | Unicode |
    *
    * If \p charset is set to CHSET_NONE, the value will be taken from the
    * TDBS_CHARS logical name in tdbs.conf. Regardless of how the \p charset
    * is specified (explicitly or implicitly via CHSET_NONE/TDBS_CHARS), the
    * only valid values for the new character set are LATIN-1 and UTF-8.
    *
    * #### Since
    * This function was introduced in TRIP version 8.2-0.
    */
   int TdbSetUICharset(int charset);

   /**
    * @brief Create a login ticket for the currently logged in user.
    * @param ticket Buffer to receive the login ticket
    * @param maxlen Allocated size of buffer (should be at least 64 bytes)
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    * 
    * #### Description
    * A login ticket is a temporary access token that can be used to authenticate
    * as the user that created the ticket. Logging in using a login ticket is
    * done using the normal APIs for login, save for that the ticket is passed 
    * as the password and the username as an empty string.
    * 
    * The login ticket is valid for the duration specified by the 
    * \c TDBS_TICKET_TIMEOUT logical name in the tdbs.conf configuration
    * file, or until the TRIP daemon (tripd) is restarted. The ticket can also
    * be revoked by calling TdbDeleteTicket() as the user to which the ticket
    * applies, or as SYSTEM.
    * 
    * @note Login tickets, if used, must be stored in a secure manner by the 
    * application. 
    * 
    * @note Login tickets are not persistently stored. They will automatically
    * be revoked if TRIPsystem is restarted. Use token pairs instead if you 
    * need your tokens to survive system restarts.
    * 
    * @see TdbDeleteTicket()
    * @see TdbCreateTokenPair()
    */
   int TdbCreateTicket(char* ticket, int maxlen);

   /**
    * @brief Delete login ticket(s) for the currently logged in user, 
    * @param ticket Ticket to delete or an empty string for the current user's ticket
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * If the \p ticket parameter specifies a login ticket for another user, the 
    * currently logged on user must have SYSTEM privileges.
    * 
    * @see TdbCreateTicket()
    */
   int TdbDeleteTicket(const char* ticket);
   

    /**
     * \brief Create an access/refresh token pair for the specified or currently logged on user
     * \param apikey A valid API key, or NULL / empty string if an API key is not required for token creation 
     * \param user Name of user to create token for, or NULL/empty for the current user
     * \param tag ID / purpose for the token pair
     * \param access_token Recieves pointer to the new access token
     * \param access_token_length Recieves the length of the access token
     * \param refresh_token Recieves pointer to the new refresh token
     * \param refresh_token_length Recieves the length of the refresh token
     * \param access_expires_at Receives the access token expiration timestamp
     * \param refresh_expires_at Receives the refresh token expiration timestamp
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * This function creates a new access/refresh token pair for a named user
     * or the the currently logged on user. If the named user is other than 
     * the current user, the logged on user must be its user mangager (UM) or
     * SYSTEM with SUPERMAN privileges.
     * 
     * The access token can be used to authenticate as the user for whom the
     * token was created. Logging in using this access token is done using the
     * normal APIs for login, save for that the token is passed as the
     * password and the username as an empty string.
     * 
     * The refresh token can be used to refresh the token pair, extending
     * the validity of use. See TdbRefreshTokenPair() for more information.
     * 
     * The \p tag parameter must, if provided, be unique for the user for
     * which the token pair is to be created. Refreshing a token (using
     * the TdbRefreshToken() function) retains the tag value.
     * 
     * The pointers returned in \p access_token and \p refresh_token must be 
     * deallocated by call to the TdbDeallocate() function.
     * 
     * #### Return Codes
     * 
     * This function returns an odd value on success and an even value
     * on failure.
     * 
     * @see TdbCreateApiKey()
     * @see TdbRefreshTokenPair()
     * @see TdbPurgeTokens()
     * @see TdbPurgeUserTokens()
     * @see TdbRevokeTokens()
     * @see TdbEnumTokens()
     * @see TdbEnumUserTokens()
     * @see TdbDeallocate()
     */
    int TdbCreateTokenPair(const char* apikey,
        const char* user, const char* tag,
        char** access_token, int* access_token_length,
        char** refresh_token, int* refresh_token_length,
        time_t* access_expires_at, time_t* refresh_expires_at);

    /**
     * \brief Create an access/refresh token pair using a refresh token
     * \param apikey A valid API key, or NULL / empty string if an API key is not required for token refresh 
     * \param current_refresh_token Valid one-time use refresh token
     * \param access_token Recieves pointer to the new access token
     * \param access_token_length Recieves the length of the access token
     * \param refresh_token Recieves pointer to the new refresh token
     * \param refresh_token_length Recieves the length of the refresh token
     * \param access_expires_at Receives the access token expiration timestamp
     * \param refresh_expires_at Receives the refresh token expiration timestamp
     * \param username Receives the name of the user to whom the token belongs
     * \param username_size The size of the username buffer
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * If supplied with a valid, existing one-time use refresh token, this 
     * function creates a new access/refresh token pair to replace the token
     * pair to which the \p current_refresh_token belongs.
     * 
     * This function can be used by anyone if the session is initialized but
     * no user is logged on. If a user is logged on, the provided refresh
     * token must either belong to the logged on user, or the logged on user
     * must have user manager (UM) privileges.
     * 
     * If successful, a new access/refresh token pair is created and returned.
     * The access token can be used to authenticate as the 
     * user for whom the token was created. Logging in using this access token
     * is done using the normal APIs for login, save for that the token is 
     * passed as the password and the username as an empty string.
     * 
     * The pointers returned in \p access_token and \p refresh_token must be 
     * deallocated by call to the TdbDeallocate() function.
     * 
     * #### Return Codes
     * 
     * This function returns an odd value on success and an even value
     * on failure.
     * 
     * @see TdbCreateApiKey()
     * @see TdbCreateTokenPair()
     * @see TdbPurgeTokens()
     * @see TdbPurgeUserTokens()
     * @see TdbRevokeTokens()
     * @see TdbEnumTokens()
     * @see TdbEnumUserTokens()
     * @see TdbDeallocate()
     */
    int TdbRefreshTokenPair(const char* apikey,
        const char* current_refresh_token,
        char** access_token, int* access_token_length,
        char** refresh_token, int* refresh_token_length,
        time_t* access_expires_at, time_t* refresh_expires_at,
        char* username, size_t username_size);

    /**
     * \brief Purge expired access/refresh token pairs for a user
     * \param mode One of the TOKEN_PURGE_* values (see table below)
     * \param username The name of the user for whom to purge tokens, or NULL for the current user
     * \param from_timestamp Date/time from which to purge (depending on \p mode) 
     * \param to_timestamp Date/time until which to purge (depending on \p mode) 
     * \param purgedcount Recieves pointer the number of purged token pairs
     * \param callback Optional callback function to receive verbose purge information
     * \param userdata Pointer to user data to pass along to the \p callback funciton
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * If called without a \p username this function will purge tokens from 
     * the currently logged on user. Otherwise, the \p username must be the
     * name of a user owned by the currently logged on user, or the currently
     * logged on user must have SYSTEM privileges.
     * 
     * The number of tokens purged is output in the \p purgedcount parameter.
     * For more verbose information on the purged tokens (creation time and
     * expiration timestamps), a \p callback function can be provided, which
     * gets called for each token matching the purge condition given by the
     * \p mode parameter. Note that this verbose output can be emitted even
     * if the function ultimately fails.
     * 
     * The valid values of \p mode are:
     * 
     * | Symbolic Name | Constant Value | Description |
     * | ------------- | -------------- | ------- |
     * | TOKEN_PURGE_ALL | 0 | Purge all token pairs, expired or not |
     * | TOKEN_PURGE_REFRESH | 1 | Purge all token pairs whose refresh token has expired |
     * | TOKEN_PURGE_ACCESS | 2 | Purge all token pairs whose access token has expired |
     * | TOKEN_PURGE_REFRESH_BY_TIMERANGE | 129 | Purge all token pairs whose refresh token has expired within the time range specified by \p fromTimestamp and \p toTimestamp |
     * | TOKEN_PURGE_ACCESS_BY_TIMERANGE | 130 | Purge all token pairs whose access token has expired within the time range specified by \p fromTimestamp and \p toTimestamp |
     * | TOKEN_PURGE_CREATED_BY_TIMERANGE | 132 | Purge all token pairs created within the time range specified by \p fromTimestamp and \p toTimestamp (regardless of if the tokens have expired or not) |
     * 
     * #### Return Codes
     * 
     * This function returns an odd value on success and an even value
     * on failure.
     * 
     * @see TdbCreateTokenPair()
     * @see TdbRefreshTokenPair()
     * @see TdbRevokeTokens()
     * @see TdbEnumTokens()
     * @see TdbEnumUserTokens()
     * @see TdbPurgeTokens()
     */
    int TdbPurgeUserTokens(int mode, const char* username, 
        time_t from_fimestamp, time_t to_timestamp, int* purgedcount,
        FP_TdbPurgeTokensCallback callback, void* userdata);

    /**
     * \brief Purge expired access/refresh token pairs
     * \param mode One of the TOKEN_PURGE_* values (see table below)
     * \param from_timestamp Date/time from which to purge (depending on \p mode) 
     * \param to_timestamp Date/time until which to purge (depending on \p mode) 
     * \param purgedcount Recieves pointer the number of purged token pairs
     * \param callback Optional callback function to receive verbose purge information
     * \param userdata Pointer to user data to pass along to the \p callback funciton
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * This function removes expires tokens (depending in the \p mode 
     * parameter) from either the current user (if logged on without
     * UM or SM privileges), from the owned users (if logged on as UM),
     * or from all users (if logged as SYSTEM with SUPERMAN privileges).
     * 
     * The number of tokens purged is output in the \p purgedcount parameter.
     * For more verbose information on the purged tokens (creation time and
     * expiration timestamps), a \p callback function can be provided, which
     * gets called for each token matching the purge condition given by the
     * \p mode parameter. Note that this verbose output can be emitted in part
     * even if the function ultimately fails.
     * 
     * The valid values of \p mode are:
     * 
     * | Symbolic Name | Constant Value | Description |
     * | ------------- | -------------- | ------- |
     * | TOKEN_PURGE_ALL | 0 | Purge all token pairs, expired or not |
     * | TOKEN_PURGE_REFRESH | 1 | Purge all token pairs whose refresh token has expired |
     * | TOKEN_PURGE_ACCESS | 2 | Purge all token pairs whose access token has expired |
     * | TOKEN_PURGE_REFRESH_BY_TIMERANGE | 129 | Purge all token pairs whose refresh token has expired within the time range specified by \p fromTimestamp and \p toTimestamp |
     * | TOKEN_PURGE_ACCESS_BY_TIMERANGE | 130 | Purge all token pairs whose access token has expired within the time range specified by \p fromTimestamp and \p toTimestamp |
     * | TOKEN_PURGE_CREATED_BY_TIMERANGE | 132 | Purge all token pairs created within the time range specified by \p fromTimestamp and \p toTimestamp (regardless of if the tokens have expired or not) |
     * 
     * #### Return Codes
     * 
     * This function returns an odd value on success and an even value
     * on failure.
     * 
     * @see TdbCreateTokenPair()
     * @see TdbRefreshTokenPair()
     * @see TdbRevokeTokens()
     * @see TdbEnumTokens()
     * @see TdbEnumUserTokens()
     * @see TdbPurgeUserTokens
     */
    int TdbPurgeTokens(int mode, 
        time_t from_timestamp, time_t to_timestamp, int* purgedcount,
        FP_TdbPurgeTokensCallback callback, void* userdata);

    /**
     * \brief Revoke a specific token or all tokens for a particular user.
     * \param username The name of the user for whom to revoke token(s)
     * \param token Access or refresh token for the token pair to revoke
     * \param tag Tag of token pair to revoke. Used instead of \p token but requires \p username
     * \param revokecount Recieves pointer the number of revoked token pairs
     * \param callback Optional callback function to receive verbose revocation information
     * \param userdata Pointer to user data to pass along to the \p callback funciton
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * If called without either \p token or \p tag this function will revoke
     * all tokens from the named user. If called with a \p tag and a
     * \p username the user's tokens with identical tags will be revoked.
     * If called both a \p token and a \p username the token must belong to
     * the named user. If the named user is not the currently logged on user,
     * the currently logged on user must be its owner or have SYSTEM privileges.
     * 
     * The number of revoked tokens is returned in the \p callback parameter.
     * For more verbose information on the revoked tokens (creation time and
     * expiration timestamps), a \p callback function can be provided, which
     * gets called for each revoked token. Note that this verbose output can
     * be emitted even if the function ultimately fails.
     * 
     * #### Return Codes
     * 
     * This function returns an odd value on success and an even value
     * on failure.
     * 
     * @see TdbCreateTokenPair()
     * @see TdbRefreshTokenPair()
     * @see TdbEnumTokens()
     * @see TdbEnumUserTokens()
     * @see TdbPurgeTokens()
     * @see TdbPurgeUserTokens()
     */
    int TdbRevokeTokens(const char* username, const char* token, const char* tag,
        int* revokecount, FP_TdbPurgeTokensCallback callback, void* userdata);

    /**
     * \brief Enumerate access/refresh token pairs for a user
     * \param username The name of the user for whom to enumerate tokens, or NULL for the current user
     * \param handle Pointer to a void pointer that holds the enumeration state. 
     * \param tag Buffer to receive the tag for the token pair
     * \param tag_size The allocated size of the 'tag' buffer
     * \param created Receives the date/time at which the token pair was created
     * \param access_expires_at Receives the date/time at which the access token expires
     * \param refresh_expires_at Receives the date/time at which the refresh token expires
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * The first call to this function requires that the \p handle points 
     * to a void pointer set to NULL. Unless the enumeration is for the
     * currently logged on user, the first call must also specify the 
     * \p username parameter.
     * 
     * If first called without a \p username this function will enumerate 
     * tokens for the currently logged on user. Otherwise, the \p username 
     * must on first call be the name of a user owned by the currently logged
     * on user, or the currently logged on user must have SYSTEM privileges.
     * 
     * The once called, the function MUST be called until it returns zero
     * or fails. Aborting the numeration call sequence without waiting for
     * a zero or failure exit code will result in memory leaks.
     * 
     * #### Return Codes
     * 
     * This function returns 1 on success, 0 if there are no more tokens,
     * and an even value on failure.
     * 
     * @see TdbCreateTokenPair()
     * @see TdbRefreshTokenPair()
     * @see TdbPurgeTokens()
     * @see TdbPurgeUserTokens()
     * @see TdbRevokeTokens()
     * @see TdbEnumTokens()
     * @see TdbEnumTokens()
     */
    int TdbEnumUserTokens(const char* username, void** handle, 
        char* tag, size_t tag_size,
        time_t* created, time_t* access_expires_at, 
        time_t* refresh_expires_at);

    /**
     * \brief Enumerate access/refresh token pairs for all users
     * \param handle Pointer to a void pointer that holds the enumeration state. 
     * \param user Buffer that receives name of the user owning the token pair 
     * \param user_size Size of the 'buffer' buffer
     * \param tag Buffer that receives the tag for the token pair 
     * \param tag_size Size of the 'tag' buffer
     * \param created Receives the date/time at which the token pair was created
     * \param access_expires_at Receives the date/time at which the access token expires
     * \param refresh_expires_at Receives the date/time at which the refres htoken expires
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * The first call to this function requires that the \p handle points 
     * to a void pointer set to NULL.
     * 
     * The function returns, for each call, information about a token pair;
     * the owning user, the access token expiration timestamp, the refresh
     * token expiration timestamp and the creation timestamp. The data is
     * returned for all owned users if the user has UM (user manager) 
     * privileges, all users if the user is SYSTEM and has SUPERMAN
     * privileges, or otherwise only for the current user if the user.
     * 
     * The once called, the function MUST be called until it returns zero
     * or fails. Aborting the numeration call sequence without waiting for
     * a zero or failure exit code will result in memory leaks.
     * 
     * #### Return Codes
     * 
     * This function returns 1 on success, 0 if there are no more tokens,
     * and an even value on failure.
     * 
     * @see TdbCreateTokenPair()
     * @see TdbRefreshTokenPair()
     * @see TdbPurgeTokens()
     * @see TdbPurgeUserTokens()
     * @see TdbRevokeTokens()
     * @see TdbEnumTokens()
     */
    int TdbEnumTokens(void** handle,  char* user, size_t user_size, 
        char* tag, size_t tag_size, time_t* created,
        time_t* access_expires_at, time_t* refresh_expires_at);

    /**
     * @brief Create an API key for use with the token functions
     * @param keyname The name of the API key to create (max 128 bytes)
     * @param comment A brief description of the API key
     * @param replace True to replace an existing key with the same name
     * @param keybuf Receives pointer to buffer containing the created API key
     * @param keybuf_size Receives the length of the data returned in 'keybuf'
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Descrition
     * 
     * A valid API key may be needed in order to create or refresh a token
     * pair, as well as to log in using an access token. This requirement may
     * be controlled using the tdbs.conf variable TDBS_REQUIRE_APIKEY (refer
     * to the TRIPsystem Environment reference manual for details).
     * 
     * The returned key must be kept safe by the calling application. If this
     * key is compromised, for whatever reason, it must be destroyed and a new
     * one created in its stead.
     * 
     * #### Return Codes
     * 
     * This function returns 1 on success, 0 if there are no more tokens,
     * and an even value on failure.
     * 
     * @see TdbRevokeApiKey()
     * @see TdbApiKeyInfo()
     * @see TdbCreateTokenPair()
     * @see TdbRefreshTokenPair()
     * @see TdbRevokeTokens()
     */
    int TdbCreateApiKey(const char* keyname, const char* comment,
        boolean replace, char** keybuf, int* keybuf_size);

    /**
     * @brief Revoke an API key
     * @param keyname The name of the API key to revoke
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * This function deletes an existing API key. All token based access
     * using the API key will immediately cease to work.
     * 
     * #### Return Codes
     * 
     * This function returns 1 on success, 0 if there are no more tokens,
     * and an even value on failure.
     * 
     * @see TdbCreateApiKey()
     * @see TdbApiKeyInfo()
     * @see TdbCreateTokenPair()
     * @see TdbRefreshTokenPair()
     * @see TdbRevokeTokens()
     */
    int TdbRevokeApiKey(const char* keyname);

    /**
     * @brief Get information about a named API key
     * @param keyname Name of the API key to get information on
     * @param mode The type of information to retrieve as a APIKEYINFO_* value
     * @param outstr Buffer to receive a string output (depends on \p mode)
     * @param outstrsz The allocated size of the \p outstr buffer
     * @param outint Buffer to receive an int output (depends on \p mode)
     * @return 1 or other odd value on success, even value on error
     * @ingroup session
     * 
     * #### Description
     * 
     * This function retrieves information on an existing API key.
     * 
     * The \p mode parameter can have the following values:
     * 
     * | Symbolic Name | Output Parameter | Purpose |
     * | ------------- | -------------- | --------- |
     * | APIKEYINFO_COMMENT | outstr | The \p outstr parameter receives the comment assigned to the API key |
     * | APIKEYINFO_OWNER | outstr | The \p outstr parameter receives the name of the TRIP user that created the key |
     * | APIKEYINFO_CREATED | outstr | The \p outstr parameter receives the date and time for when the key was created |
     * | APIKEYINFO_CREATED_DATE | outstr | The \p outstr parameter receives the date at which the key was created |
     * | APIKEYINFO_CREATED_TIME | outstr | The \p outstr parameter receives the tme at which the key was created |
     * 
     * If the funciton is passed a \p mode value of 0 (zero), it will act 
     * as an existence check only and not return any values in either the 
     * \p outstr or \p outint parameters. In this case, a successful return
     * code indicates that the key exists.
     * 
     * #### Return Codes
     * 
     * This function returns 1 on success, 0 if there are no more tokens,
     * and an even value on failure.
     * 
     * @see TdbCreateApiKey()
     * @see TdbRevokeApiKey()
     * @see TdbCreateTokenPair()
     * @see TdbRefreshTokenPair()
     * @see TdbRevokeTokens()
     */
    int TdbApiKeyInfo(const char* keyname, int mode, 
        char* outstr, size_t outstrsz, int* outint);

    /**
     * @brief Check if an API key is valid.
     * @param apikey The API key to check for validity
     * @return 1 or other odd value on success, or an even value on error
     * @ingroup session
     */
    int TdbCheckApiKey(const char* apikey);

    /**
     * @brief Enumerate available API keys
     * @param keyname Buffer to receieve the name of an API key
     * @param keyname_size The allocated size of the \p keyname buffer
     * @param comment Buffer to receieve the comment for an API key
     * @param comment_size The allocated size of the \p comment buffer
     * @param creator Buffer to receieve the name of the creator of an API key
     * @param creator_size The allocated size of the \p creator buffer
     * @param credate Buffer to receieve the creation date for an API key
     * @param credate_size The allocated size of the \p credate buffer
     * @param cretime Buffer to receieve the creation time for an API key
     * @param cretime_size The allocated size of the \p cretime buffer
     * @return 1 on success, 0 if there are no more keys, or a non-zero even value on error
     * @ingroup session
     */
    int TdbEnumApiKeys(void** handle, 
        char* keyname, size_t keyname_size,
        char* comment, size_t comment_size,
        char* creator, size_t creator_size,
        char* credate, size_t credate_size,
        char* cretime, size_t cretime_size);

   /**
    * @brief Retrieve the text of a TRIP message indicated by number, or
    *        display a message
    * @param messageNumber Message number
    * @param message Text of message
    * @param length Length of message string
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The function TdbMessage() retrieves the help or error message associated
    * with the specified message number. The number should be a value
    * previously returned by another TRIPapi routine. Return values smaller
    * than the predefined constant \c MSG_HAS_TEXT are however not associated
    * with a message, but simply indicate the success or failure of the routine
    * called.
    *
    * In ASE routines, TdbMessage() can be used to display a message supplied
    * by the ASE routine itself. The message will appear on the TRIPclassic
    * status bar when the ASE returns control to TRIPclassic. To specify the
    * message text as an input argument, the message number must be set to
    * \c MSG_SET_ERROR or \c MSG_SET_INFO . Note that for some types of ASEs
    * (such as data entry ASEs) the display of the message is also dependent
    * upon the code returned by the ASE.
    *
    * #### Parameter Values
    * The \p messageNumber parameter is passed an \c int as the message
    * number for which the associated text is to be retrieved. The acceptable
    * values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | MSG_HAS_TEXT | >= 32 | Retrieve the text of the associated message |
    * | MSG_SET_ERROR | -1 | Display the message argument as a user-supplied error message in the TRIPclassic status line |
    * | MSG_SET_INFO | -2 | Display the message argument as a user-supplied information message in the TRIPclassic status line. |
    *
    * The \p message parameter is passed a character string as the text of the
    * message. When \p messageNumber is either MSG_SET_ERROR or MSG_SET_INFO
    * then the string passed in used otherwise the message text associated
    * with \p messageNumber is returned in this string.
    *
    * The \p length parameter is passed the length of the message string. When
    * \p messageNumber is either MSG_SET_ERROR or MSG_SET_INFO then the
    * argument is taken to be the length of the string being passed to TRIP
    * otherwise it is returned as the length of the message text returned in
    * the \p message string.
   *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbRetCode()
    */
   int TdbMessage(int messageNumber, char* message, int* length);

   
   /**
    * @brief Return the TRIP message number
    * @return The return value returned by the previous TRIP function call.
    * @ingroup session
    *
    * #### Description
    * The function TdbRetCode() returns the current TRIP message number set by
    * TRIP upon completion of other function calls.
    *
    * #### Return Codes
    * This function can return any TRIPsystem code.
    *
    * @see TdbMessage()
    */
   int TdbRetCode(void);


   /**
    * @brief Check if current user has SUPERMAN privileges
    * @return Non-zero if user is SUPERMAN, and zero otherwise
    * @ingroup session
    *
    * #### Description
    * The tdbs.conf privileged configuration setting TDBS_SUPERMAN will, if
    * defined, give the SYSTEM user extended privileges such that it will be
    * allowed to access CONTROL objects (databases, clusters, etc) owned by
    * other users. This function returns a non-zero value if the current user
    * is SYSTEM and has been granted SUPERMAN privileges.
    */
   int TdbIsSuperUser(void);


   /**
    * @brief Get or set a TRIP date format.
    * @param dateform Date form number
    * @param datestring date format delimiter characters on inut, and an example formatted date on output.
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The function TdbShellDefDateForm() sets the date format to one of the
    * 17 date formats supported by TRIP.
    *
    * #### Parameter Values
    * The \p dateform parameter is passed an integer representing the format
    * of the date. Pass the number as a positive value to retrieve an example date
	* formatted in the specified format. Pass the number as a negative value
	* to also change the current session's date format to the specified one.
	* Acceptable values are:
    *
    * | Value | Date format | Example for May 1st 1988 |
    * | ---- | ---- | ---- |
    * | 1 | YYYY-MM-DD | 1988-05-01 |
    * | 2 | YYYY-mM-dD | 1988-5-1 |
    * | 3 | yyYY-mM-dD | 88-5-1 |
    * | 4 | YYYY-mm-dD | 1988-May-1 |
    * | 5 | yyYY-mm-dD | 88-May-1 |
    * | 6 | MM-DD-YYYY | 05-01-1988 |
    * | 7 | mM-dD-YYYY | 5-1-1988 |
    * | 8 | mM-dD-yyYY | 5-1-88 |
    * | 9 | mm-dD-YYYY | May-1-1988 |
    * | 10 | mm-dD-yyYY | May-1-88 |
    * | 11 | DD-MM-YYYY | 01-05-1988 |
    * | 12 | dD-mM-YYYY | 1-5-1988 |
    * | 13 | dD-mM-yyYY | 1-5-88 |
    * | 14 | dD-mm-YYYY | 1-May-1988 |
    * | 15 | dD-mm-yyYY | 1-May-88 |
    * | 16 | YYYYMMDD | 19880501 |
    * | 17 | yyYYMMDD | 880501 |
    *
    * The \p datestring parameter is passed a two-character string with the
    * delimiter characters to use in the date string, e.g. "--" for the
    * examples above. It receives an example of a date formatted in the
    * specified form, which is why the buffer that \p datestring refers to
    * must be large enough to accommodate the formatted example date.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Could not set the date form. |
    *
    * @see TdbCheckDate()
    * @see TdbShellModifyCcl()
    */
   int TdbShellDefDateform(int dateform, char* datestring);

   
   /**
    * @brief Modify the CCL locale language being used
    * @param language CCL locale language to use
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The function TdbShellModifyCcl() can be used to change the CCL language
    * that is being used in the normal TRIP search mode.
    *
    * #### Parameter Values
    * The \p language parameter is passed an \c int as the indicator for the
    * CCL locale language to use. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | LANGUAGE_DEFAULT | 1 | Use the system default language |
    * | LANGUAGE_ENGLISH | 9 | Use English dialect |
    * | LANGUAGE_SWEDISH | 10 | Use Swedish dialect |
    * | LANGUAGE_CHINESE | 11 | Use Chinese dialect |
    * | LANGUAGE_GERMAN | 12 | Use German dialect |
    * | LANGUAGE_FRENCH | 13 | Use French dialect |
    * | LANGUAGE_FINNISH | 14 | Use Finnish dialect |
    * | LANGUAGE_NORWEGIAN | 15 | Use Norwegian dialect |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbStartTrip()
    * @see TdbShellDefDateForm()
    * @see TdbCheckDate()
    */
   int TdbShellModifyCcl(int language);

   
   /**
    * @brief Check a field type on its name or number
    * @param fieldtypeName Name of field type
    * @param fieldtypeNumber Field type number
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The function TdbShellType() returns information about a field type. If
    * the field type is specified by a number, the routine will return the
    * name in the user's current CCL dialect. The type number can also be set
    * to zero, in which case the field type name (or the first part of the
    * name) must be specified in order to obtain the number of the type.
    *
    * #### Parameter Values
    * The \p fieldtypeName parameter is passed a character string as the name
    * of the field type. The string has to be at least 16 characters long. If
    * the type name is only partially specified, the name will be expanded on
    * return.
    *
    * The \p fieldtypeNumber parameter is passed a pointer to an integer to
    * receive the field type number. Acceptable values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * |  | 0 | The type number can also be set to zero, in which case the field type name (or the first part of the name) must be specified in order to obtain the number of the type. |
    * | FIELD_TEXT | 1 | TExt type field |
    * | FIELD_PHRASE | 3 | PHRase type field |
    * | FIELD_INTEGER | 9 | INteger type field |
    * | FIELD_NUMBER | 10 | NUMber type field |
    * | FIELD_DATE | 11 | DAte type field |
    * | FIELD_TIME | 12 | TIme type field |
    * | FIELD_STRING | 14 | STring type field |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Field check failed. |
    *
    * @see TdbCheckField()
    */
   int TdbShellType(char* fieldtypeName, int* fieldtypeNumber);

   
   /**
    * @brief Initializes the client application for a new TRIP session.
    * @param username Name of user or empty string.
    * @param password Password of user or empty string
    * @param language CCL locale language
    * @param mode TRIP initialization options. Pass 0 if not used.
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The TdbStartTrip() function initializes TRIP for use by the calling
    * process. It performs consistency checks upon the operating system
    * environment, TRIPsystem installation, licence conditions, and initialises
    * essential status variables and reporting mechanisms in TRIP. If the site
    * running the application is controlled by a concurrent user license, the
    * TRIPdaemon service is consulted to gain a session token, if possible. It
    * checks for the existence and accessibility of the data dictionary (the
    * CONTROL database) and of the error message store (the TDBSERR database)
    * that the kernel needs during execution. If the site running the
    * application is controlled by a concurrent user license, TRIPdaemon is
    * consulted to gain a session token, if possible.
    * 
    * The TdbStartTrip() function is the first function that should be called
    * by an application program. Upon return, and assuming a success return
    * code, the session will have been initialised. However, this function
    * should never be called by an application software exit (ASE) as all
    * required initialisation will have been successfully completed prior to
    * the ASE being activated.
    *
    * The TdbStartTrip() function replaces TdbBeginTdbs(), TdbShellInit() and
    * optionally TdbCheckUser() .
    * 
    * If login tickets are enabled and ticket based login is desired, the 
    * \p password parameter is passed a string with the login ticket, and the
    * \p username parameter left as an empty string. If the \p username
    * (parameter is left empty (and also \p password if login tickets are 
    * enabled), TdbStartTrip() will work as specified, except that it does
    * not call TdbCheckUser(), which instead must be called afterwards by
    * the application. 
    *
    * #### Parameter Values
    * The \p username parameter is passed a character string as the name of the
    * user being checked or an empty string.
    *
    * The \p password parameter is passed a character string as the password
    * for the user being checked or an empty string.
    *
    * The \p language parameter is passed an integer that indicates the CCL
    * locale language to use. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | LANGUAGE_DEFAULT | 1 | Use the system default language |
    * | LANGUAGE_ENGLISH | 9 | Use English dialect |
    * | LANGUAGE_SWEDISH | 10 | Use Swedish dialect |
    * | LANGUAGE_CHINESE | 11 | Use Chinese dialect |
    * | LANGUAGE_GERMAN | 12 | Use German dialect |
    * | LANGUAGE_FRENCH | 13 | Use French dialect |
    * | LANGUAGE_FINNISH | 14 | Use Finnish dialect |
    * | LANGUAGE_NORWEGIAN | 15 | Use Norwegian dialect |
    *
    * The \p mode parameter is passed an integer that indicates the TRIP
    * initialization options to use. The \p mode parameter is a bit mask. 
    * This means that the various option values can be easily combined, 
    * e.g. "16 | CHARSET_MODE(CHSET_UTF)". However, only one character set
    * option may be assigned.
    *
    * The \p mode parameter can also be used to indicate which character set
    * to use for the session. This overrides the TDBS_CHARS value as
    * specified in the TRIPrcs file. This is also the only way by which
    * unicode (in the UTF-8 encoding) can be chosen as character set.
    *
    * Valid character set values for the \p mode parameter:
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | CHARSET_MODE(CHSET_NONE) | 0 | Default character set. |
    * | CHARSET_MODE(CHSET_LA1) | 256 | Latin 1 |
    * | CHARSET_MODE(CHSET_LA2) | 512 | Latin 2 |
    * | CHARSET_MODE(CHSET_LA3) | 768 | Latin 3 |
    * | CHARSET_MODE(CHSET_UTF8) | 4096 | Unicode |
    * | CHARSET_MODE(CHSET_CHI) | 4352 | Chinese |
    * | CHARSET_MODE(CHSET_GBK) | 4608 | Chinese GBK |
    * | CHARSET_MODE(CHSET_EUC) | 4864 | EUC |
    * | CHARSET_MODE(CHSET_SJIS) | 5120 | Shift-JIS |
    *
    * Note that the character set value to pass to the \p mode parameter is
    * shifted 8 bits in order to be able to fit into this bit mask. This
    * shifting is done using the CHARSET_MODE macro, as illustrated in the
    * table above. Using one of the character set contstants (e.g. CHSET_UTF8)
    * directly will not work.
    * 
    * Miscellaneous other options for the \p mode parameter:
    * | Constant Value | Purpose |
    * | -------------- | ------- |
    * | 16 | Restart a former session. |
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * @note If the TdbStartTrip() function fails and haven't previously been 
    * successfully called with a blank username and password, the error message
    * cannot be fetched by calling TdbMessage() since no session will yet exist.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 9634 | No such user as username | The supplied user name could not be identified as a TRIP user. |
    * | 9666 | User id username is a group id | The supplied user name was identified as the name of a TRIP group. |
    * | 10306 | Invalid password - Please try again. | The supplied password was not valid for the specified user. |
    * | 15840 | Concurrent user quota (N) exceeded | The maximum number of licenced concurrent TRIP users has been reached. When this error code is returned the accompanying error text will include the maximum number of concurrent users licenced for the machine. |
    * | 16512 | Invalid TRIPdaemon process name | TRIPsystem cannot locate the process for the TRIP daemon. |
    *
    * @see TdbStopTrip()
    * @see TdbSessionInfo()
    * @see TdbChangeUser()
    * @see TdbShellDefDateForm()
    */
   int TdbStartTrip(char* username, char* password, int language, int mode);
   
   
   
   /**
    * @brief Terminates a TRIP session started with TdbStartTrip.
    * @param restart Mode of session reinitialization. Use value 1 to restart
    *        session from saved SIF file, and 0 (zero) to reinitialize a new
    *        session. When using the client version of this API, only pass 0
    *        (zero).
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    *
    * #### Description
    * The TdbStopTrip() function will end a TRIP session that has been started
    * with TdbStartTrip .
    *
    * The TdbStopTrip() function replaces TdbEndTdbs().
    *
    * When a server-side TRIP application exits, an option exists to save the
    * current SIF file for future use. The \p restart parameter is given value
    * 1 to save the SIF file, and 0 (zero) to delete it.
    *
    * This function must be the last TRIP toolkit routine called by the
    * application. The server-side version must be called only once per image activation.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 10434 | Background printing still in progress. | This may occur when a PRINT command is still executing on the server. |
    *
    * @see TdbStartTrip()
    */
   int TdbStopTrip(boolean restart);

   /**
    * @brief Get a configuration symbol value 
    * @param symbolName Name of configuration symbol to get
    * @param section Id of tdbs.conf section to read value from
    * @param valueBuffer Buffer that recives a copy of the value, or NULL to return the requried buffer size
    * @param bufferSize Allocated size of the \p valueBuffer
    * @param defaultValue Default value to assign if not found or NULL
    * @return 1 on success, 0 if not found, or 2 if the specified bufferSize is too small
    * @ingroup session
    *
    * #### Description
    * Retrieves the value of a configuration symbol, as read from either
    * the tdbs.conf file.
    *
    * Valid values for the \p section parameter:
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | CONF_PRIVILEGED | 1 | Look for value in the the Privileged section of tdbs.conf |
    * | CONF_GROUP | 2 | Look for value in the the user group specific section of tdbs.conf |
    * | CONF_NONPRIVILEGED | 4 | Look for value in the the NonPrivileged section of tdbs.conf |
    * | CONF_ALL | 7 | Look for value in any section of tdbs.conf |
    */
   int TdbGetConf(const char* symbolName, int section, 
       char* valueBuffer, size_t bufferSize,
       const char* defaultValue);

   /**
    * Assign a configuration value to use for the remainder of the current TRIP session.
    * @param symbolName Name of configuration symbol to assign
    * @param symbolValue Value of the configuration symbol
    * @return 1 or other odd value on success, even value on error
    * @ingroup session
    */
   int TdbSetConf(const char* symbolName, const char* symbolValue);


   /**
    * Resolve database file name containing logical names from tdbs.conf into a file systenm path.
    * @param symbolPath Database (BAF, BIF, VIF) file name as specified in the DB design
    * @param resolved Buffer to receieve the resolved path
    * @param bufsize The allocated size of the resolved buffer
    * @return 1 on successs and 0 if path could not be resolved.
    * @ingroup session
    *
    * The successful resolution of the symbol path does not guarantee that the
    * path actually exists on the local file system. If resolution fails, the
    * resolved buffer is left unmodified.
    */
   int TdbResolvePath(const char* symbolPath, char* resolved, size_t bufsize);

   
   // ------------------------------------------------------------------------

   /**
    * @defgroup cmdsearcexxref Command and Search Execution
    * @brief Query related operations
    *
    * This group of functions cover execution of CCL commands, term list
    * operations and querying status on conducted searches.
    */

   
   /** @name Indexing and batch Jobs
    *  Functions for the execution of database indexing and other
    *  bactch-oriented jobs.
    *  @{
    */

   /**
    * @brief Submit a batch job to load data into a database or index a database
    * @param mode Type of operation to be performed.
    * @param base Name of database.
    * @param tformFile Name of TForm file.
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * The TdbBatchLoadIndex() function can load data from a file, index a
    * database, or both. To load data (which must be in TForm format), set the
    * mode argument to LOAD and specify both a database name and a filename. To
    * index a database, set the mode argument to INDEX and specify just the
    * database name. To perform both operations, set the mode argument to
    * LOADIX and specify both the database name and the filename.
    *
    * The name of the file in this setting can be the full RMS file
    * specification, including any logical names which will be recognised by
    * the batch process.
    *
    * #### Parameter Values
    * The \p mode parameter is passed an integer value to select the type of
    * operation to be performed on the database. Valid values are:
    *
    * | Symbolic Constant | Constant Value | Purpose |
    * | ----------------- | -------------- | ------- |
    * | LOADIX | 1 | Load the TForm file and then index the database |
    * | INDEX | 2 | Index the database |
    * | LOAD | 3 | Load the TForm file |
    *
    * The \p base parameter is passed a character string containing the name
    * of the database to be modified.
    *
    * The \p tformFile parameter is passed a character string containing the
    * name of the TFORM file which is to be loaded into the database. The file
    * must reside on the TRIP server machine.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 514 | TFORM input file not found. | The TFORM file passed to the function cannot be found. |
    * | 9314 | Subprocess quota reached, try later (OS status: 2). | If the process quota for maximum number of subprocesses is reached during a call to this function the TRIPkernel will return this error. The error message text will include the status code from the underlying operating system. |
    * | 15010 | 1 is a database cluster. | The database name passed to the function refers to a database cluster and not a single database. |
    * | 16610 | Cannot submit job (OS status: 2). | An operating system error has prevented the batch job from being submitted. The error message text will include the operating system specific status code. |
    * | 17762 | TFORM file not specified. |  |
    *
    * @see TdbPutRecordBuffer()
    */
   int TdbBatchLoadIndex(int mode, const char* base, const char* tformFile);

   
   /**
    * @brief Index a specified record in a database.
    * @param base The name of the database that contains the record to index.
    * @param recordNumber Record id of record to index, or INDEX_ALL to index all new and modified records
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * The TdbIndex() function is used to index a single specified record in a
    * database in a synchronous fashion. Alternatively, pass SYNCH_INDEX_ALL 
    * \p recordNumber instead of a record number to index all new and modified 
    * records in the database, or SYNCH_REINDEX_ALL to reindex the entire database
    * synchronously.
    *
    * @note With TRIPsystem versions older than 7.0 there is a defect in this
    * function that renders it unusable. If you are using an older version of
    * TRIPsystem, please use TdbBatchLoadIndex() instead and poll the indexed
    * date timestamp for changes to emulate synchronous indexing.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0, but requires version
    * 7.0-0 or later for it to work correctly.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 3106 | Old version of BIF/VIF files. | The index files have been created by an older version of TRIP. The database will have to be reindexed. |
    * | 11362 | Max 250 databases can be opened during a TRIP session.. | Too many databases are open. Close a few and try again. |
    * | 2530 | No access to database name. | The logged on user does not have any access to the database. |
    * | 9538 | No write access to the database name. | The logged on user does not have write access to the database. |
    * | 15010 | Name is a database cluster. | A database cluster cannot be indexed. |
    * | 962 | File name error, check logical names for database. | The BAF/BIF/VIF file names are not valid. |
    * | 0000 | Database name is OS read protected. | File system protection for the database files prohibits read access. |
    * | 22594 | The type file is not found or not available. | The specified file for the database could not be located. |
    * | 22626 | No disk write access to the type file. | The BAF, BIF or VID file for the database is write protected. |
    * | 22434 | The type file is currently locked by another user. | Another user is locking the database or its index. Try again later. |
    * | 22690 | The database name is empty. | There is no data in the database. |
    * | 17218 | The BAF file of database is no BAF file. | The file specified as the BAF file for the database in the database definition could not be recognized as a BAF file. |
    * | 22722 | Old version of the type file, reindex database. | The database BAF, BIF or VIF file is of an older version. Reindexing is required. |
    * | 15202 | The BAF file of database needs reorganization (See PACKIT). | Run PACKIT on the TRIPsystem server machine to correct this problem. |
    * | 22498 | No disk write access to directory for the type file. | The directory in which the database file of the stated type is specified to be located cannot be written to. Check file system protection for the directory. |
    * | 22562 | Cannot create the type file. | The specified file could not be created. |
    * | 22754 | The DB indices are being updated. |  |
    * | 22786 | Invalid rescan, database contains unindexed records. |  |
    * | 22530 | Not enough disk space for the type file. | Out of disk space! |
    * | 17858 | Chinese vocabulary files not found. |  |
    * | 22818 | Not enough memory for indexing. |  |
    *
    * @see TdbBatchLoadIndex()
    */
   int TdbIndex(const char* base, int recordNumber);
   

   /** @} */         // END OF BATCH COMMANDS BLOCK

   /** @name CCL Execution
    *  Functions for the execution of CCL commands
    *  @{
    */


   /**
    * @brief Performs syntax analysis of a CCL order.
    * @param cclorder CCL order string
    * @param status CCL order identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * Checks the syntax of a CCL order. The status argument returns the command
    * number as listed in the include file tripbase.h or tdbdef.h
    * (macro constants in the form CMD_xxx, e.g "CMD_FIND").
    *
    * #### Parameter Values
    * The \p cclorder parameter is passed a character string as the CCL order
    * to parse.
    *
    * The \p status parameter receives the command number corresponding to the
    * CCL command contained in \p cclorder as listed in the include file
    * tripapi_defs.h. See TdbExecuteCcl() for a list of the command numbers.
    *
    * #### Return Codes
    * The return codes from this function depends on the nature of the CCL
    * command given. Since most of TRIP operations can be executed via CCL,
    *  this function may return almost any return code.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * @see TdbExecuteCcl()
    */
   int TdbCheckCcl(char* cclorder, int* status);
   

   /**
    * @brief Execute a CCL order
    * @param cclorder CCL command string
    * @param status CCL order identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Parameter Values
    * The \p status parameter receive the command number as listed in the
    * include files tripbase.h (server version) and tdbdef.h (client version).
    * See the list below for command numbers:
    *
    * | Number | Symbol |
    * | ------ | ------ |
    * | 0 | CMD_ERROR |
    * | 1 | CMD_CUT |
    * | 2 | CMD_SELECT |
    * | 3 | CMD_FIND |
    * | 4 | CMD_SHOW |
    * | 5 | CMD_MORE |
    * | 6 | CMD_BASE |
    * | 7 | CMD_CALL |
    * | 8 | CMD_DISPLAY |
    * | 9 | CMD_IMPORT |
    * | 10 | CMD_EXPORT |
    * | 11 | CMD_MACRO |
    * | 12 | CMD_STOP |
    * | 13 | CMD_DEFINE |
    * | 14 | CMD_LIST |
    * | 15 | CMD_NEXT |
    * | 15 | CMD_PREV |
    * | 16 | CMD_HELP |
    * | 17 | CMD_EXPAND |
    * | 18 | CMD_PRINT |
    * | 19 | CMD_RUN |
    * | 20 | CMD_SAVE |
    * | 21 | CMD_DELETE |
    * | 22 | CMD_TRACE |
    * | 23 | CMD_CONTINUE |
    * | 24 | CMD_STATUS |
    * | 25 | CMD_PRINT_LOCAL |
    * | 26 | CMD_EDIT |
    * | 27 | CMD_INSERT |
    * | 28 | CMD_UPDATE |
    * | 29 | CMD_BACK |
    * | 30 | CMD_TOP |
    * | 31 | CMD_MEASURE |
    * | 32 | CMD_FREQ |
    * | 33 | CMD_INDEX |
    * | 34 | CMD_CONT_EDIT |
    * | 35 | CMD_CLOSE |
    * | 36 | CMD_HIDE |
    * | 37 | CMD_REVEAL |
    * | 38 | CMD_CONT_SHOW |
    * | 39 | CMD_FUZZ |
    * | 40 | CMD_SFORM |
    * | 41 | CMD_LEAVE |
    * | 42 | CMD_LOAD |
    * | 43 | CMD_RENUM |
    * | 44 | CMD_FIND_SAVE |
    * | 45 | CMD_DELETE_SEARCH |
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * The return codes from this function depends on the nature of the CCL
    * command given. Since most of TRIP operations can be executed via CCL,
    * this function may return almost any return code.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbSearchInfo()
    * @see TdbGetParsedCcl()
    * @see TdbCheckCcl()
    */
   int TdbExecuteCcl(char* cclorder, int* status);

   
   /**
    * @brief Retrieve an alternate search expression that may yield better results
    * @param buffer Receives a alternate CCL expression on success
    * @param length [OUT] Receives the length of the data in \p buffer on success
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * After performing a search using the \c FUZZ command, it is possible that
    * there may be an alternate search expression that can yield better results.
    * To find out, call the TdbDidYouMean() function. If it succeeds, the
    * \p buffer parameter will contain a new search expression that can be
    * passed as input to TdbExecuteCcl().
    *
    * #### Since
    * This function was introduced in TRIP version 6.2-1
    *
    * @see TdbExecuteCcl()
    */
   int TdbDidYouMean(char* buffer, int* length);
   

   /**
    * @brief Get parsed version of the latest executed CCL order
    * @param mode Pass 1 to retrieve a normalized version of the original CCL order. Pass 0 (zero) to retrieve the order in DAO format.
    * @param parsed_order String buffer that receives the latest CCL order in parsed form.
    * @param length Receives the length of the value returned in the parsed_order parameter.
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * Returns, depending on \p mode , the parsed CCL or DAO form of a
    * previously entered CCL order.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbExecuteCcl()
    */
   int TdbGetParsedCcl(int mode, char** parsed_order, int* length);

   
   
   /** @} */     // END OF CCL COMMANDS BLOCK
   
   
   /** @name Search Sets
    *  Functions for the manipulation of search results
    *  @{
    */


   /**
    * @brief Check that a search with the given number exists
    * @param search Search number
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * The TdbCheckSearch() verifies that the given search exists, when
    * identified by number. For instance, to verify that S=5 is in the
    * calling process' history, use 5 as the search number.
    *
    * If a search number of zero is passed via the \p search parameter then
    * upon return from the function this parameter will contain the number of
    * the latest search.
    *
    * #### Parameter Values
    * The \p search parameter is passed the number of the search whose
    * existence is to be verified.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | The specified search set does not exist. |
    *
    * @see TdbSearchInfo()
    */
   int TdbCheckSearch(int* search);
   

   /**
    * @brief Retrieve information about a search set
    * @param search_number Set set number to query, or 0 for the latest one
    * @param mode The ID of the information to obtain
    * @param integer_value A pointer to a 64-bit signed integer that
    *        receives information represented as integer values.
    * @param string_buffer A pointer to a character string buffer that
    *        receives information represented as charcter data.
    * @param string_buffer_size The allocated size of the
    *        \p string_buffer parameter
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * This function returns information on search sets, and replaces the older
    * functions TdbGetSearch() and TdbSearchResult().
    *
    * The \p mode parameter is passed a constant that identifies what
    * information to retrieve about the search set identified by the
    * \p search_number parameter. Valid values are:
    *
    * | Symbolic Name | Constant Value | Description |
    * | ------------- | -------------- | ----------- |
    * | SEARCHINFO_SET_NUMBER | 1 | Search set ID, if 0 is passed to \p search_number |
    * | SEARCHINFO_RECORD_COUNT | 2 | Number of records in the search set |
    * | SEARCHINFO_HIT_COUNT | 3 | Number of hits in the search set |
    * | SEARCHINFO_COMMAND | 4 | The CCL command used to generate the set |
    * | SEARCHINFO_DATABASE | 5 | The name of the database or cluster searched |
    * | SEARCHINFO_DATABASES | 6 | A comma separated list of the names of the database(s) with hits in the search set |
    * 
    * If the \p mode parameter is SEARCHINFO_DATABASES, the caller should first
    * invoke this function with \p string_buffer set to NULL. This returns the
    * required minimum size of the buffer in the \p integer_value parameter.
    * The application should allocate a buffer of at least this size and then
    * call the function again with the new buffer passed in \p string_buffer
    * and its allocated size in the \p string_buffer_size parameter.
    * 
    * #### Since
    * This function was introduced in TRIP version 8.0-0.
    */
   int TdbSearchInfo(int search_number, int mode, int64_t* integer_value,
                     char* string_buffer, size_t string_buffer_size);
   

   /** @} */         // END OF SEARCH SETS BLOCK
   

   /**
    * @brief Retrieve the name of the currently open database
    * @param base Name of database or of (predefined) database cluster
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * The function TdbCurrentBase() returns the name of the most recently
    * opened database, cluster or multi-base view. It does not provide any
    * information about which databases make up either a cluster or a
    * multi-file view, To obtain such information call TdbGetClusterDesign()
    * for a database cluster or TdbSearchInfo() for a multi-file view.
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string to receive the name
    * of the database or the database cluster that is currently open. The
    * character buffer must be at least 65 bytes in length so that it can 
    * accommodate a database/cluster name of maximum size.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | There is no database currently open. |
    *
    * @see TdbSearchInfo()
    * @see TdbGetClusterDesign
    */
   int TdbCurrentBase(char* base);

   
   /** @name Term Lists, Sets and Trees
    *  Functions for the manipulation of DISPLAY order results
    *  @{
    */
   
   /**
    * @brief Create a term set based on the most recent DISPLAY result
    * @param termsetId Output parameter that receives the identity of the created term set upon success.
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * Create a term set based on the most recent DISPLAY result.
    *
    * A term set is a display list generated by a DISPLAY order but possible
    * to use also after a new DISPLAY-order has been executed.
    *
    * #### Parameter Values
    * The \p termsetId parameter receive the term set number, by which the
    * term set is identified to the TdbGetTermSetTerm() and TdbDeleteTermSet()
    * functions.
    *
    * #### Since
    * This function was introduced in TRIP version 7.0-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | No term list is available. |
    *
    * @see TdbGetTermSetTerm()
    * @see TdbDeleteTermSet()
    */
   int TdbCreateTermSet(int* termsetId);
   
   /**
    * @brief Deletes a term set
    * @param termsetId The identity of the created term set to delete
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * Deletes a term set created using the TdbCreateTermSet() function.
    *
    * #### Parameter Values
    * The \p termsetId parameter is the term set number as returned by the
    * TdbCreateTermSet() function.
    *
    * #### Since
    * This function was introduced in TRIP version 7.0-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Term set does not exist. |
    *
    * @see TdbCreateTermSet()
    * @see TdbGetTermSetTerm()
    */
   int TdbDeleteTermSet(int termsetId);

   /**
    * Create a termset with a baseline-oriented term list.
    * @param termsetId Receives the number of the created termset on success.
    * @param shin A term list CCL order without the DISPLAY command and sort options.
    * @param mode Sorting mode (TERM_SORT_FREQ or TERM_SORT_TEXT, optionally
    *             with TERM_SORT_BL_ATEND added to the bitmask).
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * A baseline termset is a termset that can be further refined without 
    * losing any terms. Updates to a baseline termset is done using the 
    * TdbUpdateTermSet() function. Any term not present in the update will
    * be retained, but with an update count of zero. The original count 
    * will also be availanble. Use the TdbTermSetCount() function to 
    * obtain the current record counts for the termset, and TdbGetTermSetTermEx()
    * to get the baseline and update counts for individual terms.
    *
    * The \p mode parameter is a bitmask used to indicate the sorting of the baseline
    * termset. Sorting can either be by term frequency (TERM_SORT_FREQ) or 
    * alphabetically (TERM_SORT_TEXT). Adding the TERM_SORT_BL_ATEND value to the
    * bitmask indicates that terms that after update have a count of zero should
    * be sorted at the end.
    *
    * @see TdbUpdateTermSet()
    * @see TdbTermSetCount()
    * @see TdbGetTermSetTermEx
    * @see TdbDeleteTermSet()
    */
   int TdbCreateBaselineTermSet(int *termsetId, const char *shin, int mode);

   /**
    * Update termset list with restricting display command.
    * @param termsetId Receives the number of the created termset on success.
    * @param shin A term list CCL order without the DISPLAY command and sort options.
    * @param mode Sorting mode (TERM_SORT_FREQ or TERM_SORT_TEXT, optionally
    *             with TERM_SORT_BL_ATEND added to the bitmask).
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * The \p shin parameter is expected to hold the conditional part of
    * a DISPLAY command that is assumed to refine the DISPLAY command supplied
    * to the TdbCreateBaselineTermSet() function, resulting in a subset of
    * terms.
    *
    * If \p shin is NULL or an empty string, the no update/refine
    * operation will be performed. Instead, the baseline is reset to reflect
    * the record counts from the last call to TdbUpdateTermSet(), replacing
    * the old baseline.
    * 
    * @see TdbCreateBaselineTermSet()
    * @see TdbTermSetCount()
    * @see TdbGetTermSetTermEx()
    * @see TdbDeleteTermSet()
    */
   int TdbUpdateTermSet(int termsetId, const char *shin, int mode);


   /**
    * Receives the record counts for a termset
    * @param termsetId ID of termset
    * @param recordCount Receives the (baseline) record count of the termset.
    * @param updateCount Receives the baseline update record count (not relevant for regular termsets).
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    * 
    * @see TdbCreateBaselineTermSet()
    * @see TdbCreateTermSet()
    * @see TdbUpdateTermSet()
    */
   int TdbTermSetCount(int termsetId, int64_t* recordCount, int64_t* updateCount);
   
   
   /**
    * @brief Retrieve information about the current display list
    * @param termCount Number of (main) terms retrieved
    * @param termType Type of CCL display command.
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * The function TdbGetDisplay() obtains information about the results of
    * the most recent CCL  Display order. It should be called immediately
    * after a call to the function TdbExecuteCcl() which returns a
    * \c DAO_DISPLAY status code, indicating that a Display command has been
    * executed.
    *
    * The terms in the display list may be obtained by calling the
    * TdbGetDisplayTerm() .
    *
    * #### Parameter Values
    * The \p termCount parameter recieves tbe number of terms retrieved by the
    * previous CCL  display command. For a thesaurus display, it is the number
    * of main terms only.
    *
    * The \p termType parameter receives a code for the type of CCL display
    * command actually executed. Valid values returned are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | DISPLAY_NORMAL | 1 | A standard CCL display order was executed. |
    * | DISPLAY_THES | 2 | A thesaurus specific CCL display order was executed. |
    * | DISPLAY_INDIRECT | 3 | An indirect CCL display command was executed. |
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbGetDisplayTerm()
    * @see TdbExecuteCcl()
    */
   int TdbGetDisplay(int* termCount, int* termType);

   
   /**
    * @brief Retrieve a term from the current display list
    * @param termNumber Ordinal number of term in list, starting with 1
    * @param term Term string
    * @param termLength Length of term
    * @param recordCount Number of record occurrences.
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * The function TdbGetDisplayTerm() is used to obtain a term and information
    * about it from the current display list. A display list is generated by
    * the CCL Display command.
    *
    * If the current display list is a thesaurus display list the function
    * TdbGetRecordInThes() must be called before TdbGetDisplayTerm() .
    *
    * #### Parameter Values
    * The \p termNumber parameter is passed the 1-based ordinal number of the
    * term to be extracted from the term list .
    *
    * The \p term parameter receives the term to be returned.
    *
    * The \p termLength parameter receives the length of the term extracted.
    *
    * The \p recordCount parameter receives the count of the number of records
    * in which the term occurs. If the record count for a term exceeds
    * 2147483647 (2^31 - 1), this function will retrun 2147483647 as the count.
    * To get the correct count for such terms, use the TdbGetDisplayTermEx()
    * function instead.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Display term could not be retrieved. |
    * @see TdbGetRecordInThes()
    * @see TdbGetDisplay()
    */
   int TdbGetDisplayTerm(int termNumber, char* term, int* termLength,
                         int* recordCount);

   
   /**
    * @brief Retrieve a term from the current display list
    * @param termNumber Ordinal number of term in list, starting with 1
    * @param term Term string, receives the term to be returned.
    * @param termLength Receives the length of the term extracted
    * @param recordCount Receives the count of the number of records
    *        in which the term occurs.
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * This function is used to obtain a term and information about it from the
    * current display list. It differs from the TdbGetDisplayTerm() function
    * in that this function is able to return a term record count larger than
    * 2147483647 (2^31 - 1). A display list is generated by the CCL Display
    * command.
    *
    * If the current display list is a thesaurus display list the function
    * TdbGetRecordInThes() must be called before TdbGetDisplayTermEx() .
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Display term could not be retrieved. |
    *
    * @see TdbGetRecordInThes()
    * @see TdbGetDisplay()
    */
   int TdbGetDisplayTermEx(int termNumber, char* term, int* termLength,
                           int64_t* recordCount);

   
   /**
    * @brief Retrieves a term from a term set
    * @param termsetId The identity of the created term set from which to retrieve a term
    * @param termNumber Ordinal number of the term to retrieve from the term set
    * @param termValue Buffer at least 256 bytes long to receive the term value
    * @param length Size of 'termValue' buffer in input, and length of the term returned in 'ternValue' on output.
    * @param recordCount Number of record occurrences for the term.
    * @return 1 or other odd value on success, even value on error
    * @ingroup cmdsearcexxref
    *
    * #### Description
    * The function TdbGetTermSet() is used to obtain a term and information
    *  about it from a term set. A term set is generated by declaring a term
    * list generated by the CCL Display command as a term set using the
    * TdbCreateTermSet() function.
    *
    * #### Parameter Values
    * The \p termsetId parameter is the term set number as returned by the
    * TdbCreateTermSet() function.
    *
    * The \p termNumber parameter is passed the ordinal number of the term to
    * be extracted from the term set.
    *
    * The \p termValue parameter receives the term to be returned.
    *
    * The \p length parameter is assigned the size of the 'termValue' buffer
    * and on return receives the count of the number of bytes returned in it.
    *
    * The \p recordCount parameter receives the count of the number of records
    * in which the term occurs. If the record count for a term exceeds 
	* 2147483648 (2^31), this function will return 2147483648 as the count. To
	* get the correct count for such terms, use the TdbGetTermSetTermEx() 
	* function instead.
    *
    * #### Since
    * This function was introduced in TRIP version 7.0-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Term could not be retrieved. |
    *
    * @see TdbCreateTermSet()
    * @see TdbDeleteTermSet()
	* @see TdbGetTermSetTermEx()
    */
   int TdbGetTermSetTerm(int termsetId, int termNumber, char* termValue,
                         int* length, int* recordCount);


   /**
	* @brief Retrieves a term from a term set
	* @param termsetId The identity of the created term set from which to retrieve a term
	* @param termNumber Ordinal number of the term to retrieve from the term set
	* @param termValue Buffer at least 256 bytes long to receive the term value
	* @param length Size of 'termValue' buffer in input, and length of the term returned in 'ternValue' on output.
	* @param recordCount Number of record occurrences for the term.
	* @param updateCount Number of record occurrences for the term after TdbUpdateTermSet()
	* @return 1 or other odd value on success, even value on error
	* @ingroup cmdsearcexxref
	*
	* #### Description
	* This function is used to obtain a term and information
	* about it from a term set. A term set is generated by declaring a term
	* list generated by the CCL Display command as a term set using the 
	* TdbCreateTermSet() function or the TdbCreateBaselineTermSet() function. 
	*
	* #### Parameter Values
	* The \p termsetId parameter is the term set number as returned by the
	* TdbCreateTermSet() function.
	*
	* The \p termNumber parameter is passed the ordinal number of the term to
	* be extracted from the term set.
	*
	* The \p termValue parameter receives the term to be returned.
	*
	* The \p length parameter is assigned the size of the 'termValue' buffer
	* and on return receives the count of the number of bytes returned in it.
	*
    * The \p recordCount parameter receives the count of the number of
    * records in which the term occurs. If the term set is created using the
    * TdbCreateBaselineTermSet() function, the record count returned here is
    * always the baseline count.
    
	* The \p updateCount parameter receives the count of the number of records
	* in which the term occurs. It is primarily used with term sets that have
    * been created using the TdbCreateBaselineTermSet(). The value returned in
    * this parameter is in such cases the count for the term after the most
    * recent call to TdbUpdateTermSet().
	*
	* #### Since
	* This function was introduced in TRIP version 8.0-0.
	*
	* #### Return Codes
	* | Code | Description | Explanation |
	* | ---- | ----------- | ----------- |
	* | 1 | Operation completed successfully | The function completed its operations without error. |
	* | 0 |  | Term could not be retrieved. |
	*
	* @see TdbCreateTermSet()
   * @see TdbCreateBaselineTermSet()
   * @see TdbDeleteTermSet()
   * @see TdbUpdateTermSet()
	*/
   int TdbGetTermSetTermEx(int termsetId, int termNumber, char* termValue,
                           int* length, int64_t* recordCount,
                           int64_t* updateCount);
   
   
   /** @} */

   
   // ------------------------------------------------------------------------

   /**
    * @defgroup winmanipxref Window Manipulation
    * @brief Manipulation of TRIP kernel windows.
    *
    * This group of TRIPtoolkit functions cover manipulation of TRIP kernel
    * windows. These are required to access TRIP features such as output
    * formats via the SHOW command.
    */
   
   
   /**
    * @brief Retrieve the identifier of a kernel output window
    * @param windowType Type of window to be retrieved
    * @param windowId Window identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Description
    * The TdbCheckScreenWindow() can be used to gain the identifier of the
    * output window associated with a particular type of output. This is
    * especially useful when processing scrolling commands, such as More and
    * Back, when it is important to know the identifier of the current window
    * being scrolled. In this case, use the special constant \p CURRENT_WINDOW
    * as the type argument. A window can be made current with a call to the
    * TdbSetScreenWindow() function.
    *
    * The window of the specified type must have been previously created by a
    * call to the TdbDefineWindow() function.
    *
    * #### Parameter Values
    * The \p windowType parameter is passed an \c int as the type of the window for
    * which the identifier is to be retrieved. Valid values are:
    *
    * | Symbolic Constant | Value of Constant | Window Usage |
    * | ----------------- | ----------------- | ------------ |
    * | CURRENT_WINDOW | 0 | Current window |
    * | SHOW_WINDOW | 1 | Show window |
    * | EXPAND_WINDOW | 2 | Expand window |
    * | DISPLAY_WINDOW | 3 | Display window |
    * | HISTORY_WINDOW | 4 | History window (Base, Find, List, etc) |
    * | SYSINFO_WINDOW | 5 | System information window (Help, STatus, TRace, etc) |
    * | PRINT_WINDOW | 6 | Print Local window |
    *
    * The \p windowId parameter is passed an integer as the actual identifier
    * associated with the particular type of output. The identifier is by
    * default equal to the type number.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 22690 | Argument wintypenr - value out of range. | The specified window type number was out of range. |
    *
    * @see TdbDefineWindow()
    * @see TdbSetScreenWindow()
    */
   int TdbCheckScreenWindow(int windowType, int* windowId);

   
   /**
    * @brief Define a kernel window buffer for formatted output
    * @param windowType Window type
    * @param height Number of rows
    * @param width Number of columns
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Description
    * The function TdbDefineWindow() defines a kernel memory buffer for holding
    * the results of commands which produce output, such as Show and STatus.
    * The value passed in the \p type parameter specifies what type of output
    * this window is to be used for.
    *
    * Most applications will simply define the six windows during program
    * initialization, however if memory is a significant resource, then a
    * program can choose to define only those windows that it will require.
    *
    * The maximum window size (rows*cols) is 64KB.
    *
    * #### Parameter Values
    * The \p windowType parameter is passed an \c int as the type of window buffer
    * to be defined. The available window types are:
    *
    * | Window Name | Constant Value | Window Purpose |
    * | ----------- | -------------- | -------------- |
    * | SHOW_WINDOW | 1 | Output from the Show order |
    * | EXPAND_WINDOW | 2 | Output from the Expand order |
    * | DISPLAY_WINDOW | 3 | Output from the Display order |
    * | HISTORY_WINDOW | 4 | Output from the History order |
    * | SYSINFO_WINDOW | 5 | Output from Help, Status, or Trace orders |
    * | PRINT_WINDOW | 6 | Output from Print Local order |
    *
    * The \p height parameter is passed an integer as the number of rows in
    * the window.
    *
    * The \p width parameter is passed an integer as the number of columns in
    * the window.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 3618 | Pagebuffersize size beyond limits. | There is an upper limit of 64KB in how large a window can be. The specified value (rows*cols) exeeded that limit. |
    *
    * @see TdbCheckScreenWindow()
    * @see TdbSetScreenWindow()
    */
   int TdbDefineWindow(int windowType, int height, int width);

   
   /**
    * @brief Deletes the specified TRIP kernel window.
    * @param windowNumber Window number
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Parameter Values
    * The \p windowNumber parameter is passed the number of the TRIP window to
    * be deleted. The window number is retrieved using TdbCheckScreenWindow().
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbCheckScreenWindow()
    */
   int TdbDeleteWindow(int windowNumber);

   
   /**
    * @brief Retrieve the next hit word in a formatted line
    * @param windowType Window type
    * @param linePosition Position of hit word
    * @param length Length of hit word
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Description
    * The function TdbGetWindowHit() returns the position of the next hit
    * word in a formatted line within a TRIP window. It can only be called
    * after first calling TdbGetWindowLine() . Successive hits on the the
    * same line can be retrieved by calling the function TdbGetWindowHit()
    * multiple times until a status code of 0 (zero) is returned.
    *
    * #### Parameter Values
    * The \p windowType parameter is passed a the window type for which
    * highlighting information is returned. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | SHOW_WINDOW | 1 | CCL Show window |
    * | EXPAND_WINDOW | 2 | CCL Expand window |
    * | DISPLAY_WINDOW | 3 | CCL Display window |
    * | HISTORY_WINDOW | 4 | CCL History window |
    * | SYSINFO_WINDOW | 5 | CCL Help, Status, Trace window |
    * | PRINT_WINDOW | 6 | CCL Print Local window |
    *
    * The \p linePosition parameter receives the position of the hit word
    * within the current line.
    *
    * The \p length parameter receives the length of the hit word.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | No more hits on the current TRIP window line. |
    *
    * @see TdbGetWindowLine()
    */
   int TdbGetWindowHit(int windowType, int* linePosition, int* length);

   

   /**
    * @brief Return hit words in a TRIP window
    * @param windowType Window type
    * @param hitCount Number of hits in window
    * @param windowHits Hit word locations
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Description
    * The function TdbGetWindowHits() returns an array containing all hits in
    * a TRIP window. It is called after a previous call to the
    * TdbGetWindowInfo() function.
    *
    * #### Parameter Values
    * The \p windowType parameter is passed the window type for which
    * highlighting information is returned. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | SHOW_WINDOW | 1 | CCL Show window |
    * | EXPAND_WINDOW | 2 | CCL Expand window |
    * | DISPLAY_WINDOW | 3 | CCL Display window |
    * | HISTORY_WINDOW | 4 | CCL History window |
    * | SYSINFO_WINDOW | 5 | CCL Help, Status, Trace window |
    * | PRINT_WINDOW | 6 | CCL Print Local window |
    *
    * The \p hitCount parameter receives the number of hits in the window.
    *
    * The \p windowHits parameter receives the address of an integer array
    * with hit information. The size of the array is three times the number of
    * hits (3 * \p hitCount ). Each hit is represented as a triplet where:
    *
    * | Nr in triplet | Description |
    * | ------------- | ----------- |
    * | 1 | Row number the hit is on. |
    * | 2 | Column number the hit is on. |
    * | 3 | Length of hit item. |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | The TRIP window did not contain any hits. |
    *
    * @see TdbGetWindowInfo()
    */
   int TdbGetWindowHits(int windowType, int* hitCount, int** windowHits);
   

   /**
    * @brief Load all lines from a TRIP window
    * @param windowType Window type
    * @param windowBuffer Buffer to recieve information in window
    * @param rowSize Number of rows in buffer
    * @param colSize Number of columns in buffer
    * @param rowCount Count of rows filled
    * @param flags Position in output
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Description
    * The function TdbGetWindowInfo() returns the whole contents of a TRIP window in one call.
    *
    *
    * #### Parameter Values
    * The \p windowType parameter is passed an \c int as the window type for
    * which content is returned. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | SHOW_WINDOW | 1 | CCL Show window |
    * | EXPAND_WINDOW | 2 | CCL Expand window |
    * | DISPLAY_WINDOW | 3 | CCL Display window |
    * | HISTORY_WINDOW | 4 | CCL History window |
    * | SYSINFO_WINDOW | 5 | CCL Help, Status, Trace window |
    * | PRINT_WINDOW | 6 | CCL Print Local window |
    *
    * The \p windowBuffer parameter a string buffer that will recieve
    * information in the window.
    *
    * The \p rowCount parameter receives the count of how many rows of
    * \p windowBuffer have been used. For example, if the output buffer is not
    * completely filled by the data contained in the window because you are at
    * the end of the output then the \p rowCount argument will be set to the
    * number of lines actually used in the buffer. Otherwise, the value of
    * the \p rowCount argument will be equal to the \p rowSize argument.
    *
    * The \p colSize parameter receives the number of columns in the
    * \p windowBuffer buffer.
    *
    * The \p rowSize parameter receives the number of rows in \p windowBuffer
    * buffer.
    *
    * The \p flags parameter receives a bit vector indicating where in the
    * output the window is positioned. Valid values are:
    *
    * | Bit Number | Purpose |
    * | ---------- | ------- |
    * | 0 | When bit 0 is set the window is covering the bottom of the output. |
    * | 1 | When bit 1 is set the window is covering the top of the output. |
    * | 2 | When bit 2 is set the window is subject to paged output (i.e., the output format being used has ended with a &lt;FF> filter or has a page header/trailer defined). |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | The TRIP window did not contain any hits. |
    */
   int TdbGetWindowInfo(int windowType, char** windowBuffer, int* rowSize,
                        int* colSize, int* rowCount, int* flags);

   
   /**
    * @brief Load a formatted line from a TRIP window
    * @param windowType Window type
    * @param line Window line.
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Description
    * The function TdbGetWindowLine() returns a line formatted for output in a
    * TRIP kernel window. Repeated calls to this routine until it returns zero
    * will retrieve each line in the window.
    *
    * The window receives its contents when a CCL order which generates output
    * for the window type in question is issued (using the TdbExecuteCcl()
    * function). Prior to this, the window must have been defined by a call to
    * the TdbDefineWindow() function.
    *
    * If the window line contains hit words, these can be obtained by calling
    * the TdbGetWindowHit() function.
    *
    * #### Parameter Values
    * The \p windowType parameter is passed an \c int as the window type for
    * which content is returned. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | SHOW_WINDOW | 1 | CCL Show window |
    * | EXPAND_WINDOW | 2 | CCL Expand window |
    * | DISPLAY_WINDOW | 3 | CCL Display window |
    * | HISTORY_WINDOW | 4 | CCL History window |
    * | SYSINFO_WINDOW | 5 | CCL Help, Status, Trace window |
    * | PRINT_WINDOW | 6 | CCL Print Local window |
    *
    * The \p line parameter receives the content of the current window line.
    * Its minimum allocated size must be the window's defined row size plus
    * one.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | No more lines are available. |
    *
    * @see TdbExecuteCcl()
    * @see TdbDefineWindow()
    * @see TdbGetWindowHit()
    */
   int TdbGetWindowLine(int windowType, char* line);

   
   /**
    * @brief Returns the window type for a window.
    * @param windowId The number of the window for which to request the type.
    * @param windowType The window type of the specified window.
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Description
    * The TdbGetWindowType() function returns the window type for the window
    * with the specified number.
    *
    * The \p windowId parameter is passed a window number, or DEFAULT_WINDOW
    * for the current window. This is a read-only value.
    *
    * The \p windowType parameter recieves the type of the specified window.
    * Window types are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | SHOW_WINDOW | 1 | CCL Show window |
    * | EXPAND_WINDOW | 2 | CCL Expand window |
    * | DISPLAY_WINDOW | 3 | CCL Display window |
    * | HISTORY_WINDOW | 4 | CCL History window |
    * | SYSINFO_WINDOW | 5 | CCL Help, Status, Trace window |
    * | PRINT_WINDOW | 6 | CCL Print Local window |
    *
    * @see TdbStartTrip()
    * @see TdbDefineWindow()
    * @see TdbSetScreenWindow()
    * @see TdbCheckScreenWindow()
    */
   int TdbGetWindowType(int windowId, int* windowType);

   
   /**
    * @brief Set the identifier of a kernel output window
    * @param windowType Window type number
    * @param windowNumber Window identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup winmanipxref
    *
    * #### Description
    * The function TdbSetScreenWindow() connects an identifier to a specified
    * output window.
    *
    * This routine should only be used to set the ID of the current window
    * (\c CURRENT_WINDOW ), as the windows automatically receive an ID equal
    * to the type when they are created by the TdbDefineWindow() function.
    *
    * A window is set to be the current window when it receives content from
    * an output-generating CCL command. A Show command, e.g., makes the SHOW
    * window the current window and subsequent scroll commands, such as MORE
    * and BACK, will operate on that window until another CCL command changes
    * the status. Thus TdbSetScreenWindow() can be used to direct scrolling
    * orders to a window other than that which most recently received output.
    *
    * #### Parameter Values
    * The \p windowType parameter is passed an \c int as the window type
    * number.
    *
    * | Symbolic Constant | Value of Constant | Window Usage |
    * | ----------------- | ----------------- | ------------ |
    * | CURRENT_WINDOW | 0 | Current window |
    * | SHOW_WINDOW | 1 | Show window |
    * | EXPAND_WINDOW | 2 | Expand window |
    * | DISPLAY_WINDOW | 3 | Display window |
    * | HISTORY_WINDOW | 4 | History window (Base, Find, List, etc) |
    * | SYSINFO_WINDOW | 5 | System information window (Help, STatus, TRace, etc) |
    * | PRINT_WINDOW | 6 | Print Local window |
    *
    * The \p windowNumber parameter is passed an \c int as the window identifier.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbDefineWindow()
    */
   int TdbSetScreenWindow(int windowType, int windowNumber);

   
   // ------------------------------------------------------------------------

   /**
    * @defgroup procmgmtxref Procedure Management
    * @brief Managementr of TRIP stored procedures
    *
    * Functions in the procedure management group provides access to TRIP
    * procedure manipulation and execution. A TRIP procedure is a named
    * sequence of stored CCL commands, callable with the CCL RUN command.
    */
   

   /**
    * @brief Clear the stack of running procedures, effectively doing a jump
    *        from any nesting level back to the top.
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * The function TdbClearRunStack() is used to terminate the execution of a
    * procedure when an error occurs. This would normally be used in
    * conjunction with the TdbContinue() function to ensure that the procedure
    * is only terminated if the calling process has not specified DEfine
    * CONTinue.
    *
    * @note If the calling process is operating in interactive mode, the stack
    * is cleared completely; if the calling process is operating in batch mode,
    * the stack is unwound only one level.
    *
    * @see TdbContinue()
    */
   int TdbClearRunStack(void);

   
   /**
    * @brief Check if continuation on error has been defined.
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * The function TdbContinue() is used to check if execution of commands in
    * a procedure is to continue when an error is detected. If this routine
    * returns a success code, then the procedure should continue; otherwise,
    *  the calling process should call the function TdbClearRunStack() to
    * return to the top level of processing.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 |  | Procedure execution may continue. |
    * | 0 |  | Do not continue to execute the procedure. |
    *
    * @see TdbClearRunStack()
    */
   int TdbContinue(void);

   

   /**
    * @brief Create a copy of a procedure
    * @param oldProcedure Name of source procedure
    * @param newProcedure Name of new procedure
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * The function TdbCopyProcedure() starts the copying of a procedure from
    * one name to another or from one user to a different user or group, if the
    * security subsystem allows it. The calling process must call
    * TdbPutProcedure() to commit the new procedure to the CONTROL database.
    *
    *
    * #### Parameter Values
    * The \p oldProcedure parameter is passed the name of the procedure to be
    * copied.
    *
    * The \p newProcedure parameter is passed the name the new procedure. The
    * maximum length of a procedure name is 16 characters. However, it may be
    * qualified with the name of the user or group that is to stand as the
    * owner of the procedure. This means that both "VERYLONGPROCNAME" and
    * "VERYLONGUSERNAME.VERYLONGPROCNAME" are valid.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 2402 | Attempt to create an already existing procedure. | A procedure with the target name already exists. |
    *
    * @see TdbPutProcedure()
    */
   int TdbCopyProcedure(char* oldProcedure, char* newProcedure);

   
   /**
    * @brief Delete a TRIP macro/procedure
    * @param procedure Name of procedure
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * The function TdbDeleteProcedure() deletes the specified TRIP
    * macro/procedure from the CONTROL file. The TRIP user ID for the current
    * session must be the creator of the macro/procedure to be deleted.
    *
    * #### Parameter Values
    * The \p procedure parameter is passed a character string as the name of
    * the procedure to be deleted.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 32899 | Procedure id.procname deleted. | The procedure was successfully deleted. |
    * | 9026 | Procedure/group name cannot be more than 16 characters each. |  |
    * | 12706 | Missing or invalid procedure name. | An empty string or one with only white space is not valid as a name. |
    * | 15618 | Only alphanumerics/underscores allowed in names. |  |
    * | 2370 | Attempt to delete a non existent procedure. |  |
    * | 9698 | No such group as group. | The current user is not the owner of the procedure, and the group that is the owner of the procedure could not be found. |
    * | 10466 | Group is not a group id. | The current user is not the owner of the procedure, and the registered procedure owner is not a group. |
    * | 9730 | You are not the creator of the group group. | If the procedure is owned by a group, only the user which is the creator of the group may delete the procedure. |
    */
   int TdbDeleteProcedure(char* procedure);

   
   /**
    * @brief Returns the next line to be executed out of a procedure.
    * @param line Procedure or macro line.
    * @param length Length of returned line parameter.
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * Returns the next line to be executed out of a procedure.
    *
    * This function should be called subsequent to the exection of a CCL RUN
    * command. It differs from the old TRIPapi function, TdbNextRunCommand()
    * in that it takes other arguments allowing for an unlimited length of a
    * CCL order.
    *
    * #### Parameter Values
    * The \p line parameter receives the next macro or procedure line to
    * execute.
    *
    * The \p length parameter recieves the length of the returned line.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    */
   int TdbGetNextProcLine(char** line, int* length);

   

   /**
    * @brief Load a macro/procedure line
    * @param line Procedure line
    * @param lineLength Length of procedure line
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * The function TdbGetProcLine() loads a single line from a macro/procedure
    * into the \p line parameter. The macro/procedure must have been previously
    * loaded into the internal buffer by using the TdbGetProcedure(). Repeated
    * calls to TdbGetProcLine() until a status code of zero is returned allows
    * processing of the complete macro/procedure.
    *
    * If a line exceeds the maximum length passed via the \p lineLength
    * parameter, then it will be truncated and a hyphen appended to it. The
    * next call to TdbGetProcLine() retrieves the rest of the line again
    * truncating it as required. Note that the hyphen is added onto the
    * maximum-length string, so the string may actually be one character
    * longer than what was specified by the length argument on input.
    *
    * #### Parameter Values
    * The \p line parameter receives the procedure line to be retrieved.
    *
    * The \p lineLength parameter in the server version of this function
    * receives the length of the returned procedure line. Before calling the
    * function the \p lineLength argument must specify the maximum length of
    * the string that is to be copied into the line buffer, including the
    * terminating null-byte.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Could not get procedure line. No more lines. |
    *
    * @see TdbGetProcedure()
    */
   int TdbGetProcLine(char* line, int* lineLength);

   

   /**
    * @brief Load a TRIP macro or procedure
    * @param procedure Procedure or macro name
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * The function TdbGetProcedure() loads a TRIPsystem macro or procedure into
    * an internal buffer for further processing. A procedure can only be loaded
    * by its owner.
    *
    * This function is intended primarily for use within the TRIPclassic
    * environment.
    *
    * #### Parameter Values
    * The \p procedure parameter is passed a character string as the name of a
    * procedure name. The name may alternatively identify a macro.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 9698 | No such group as group. | The current user is not the owner of the procedure, and the group that is the owner of the procedure could not be found. |
    * | 10466 | Group is not a group id. | The current user is not the owner of the procedure, and the registered procedure owner is not a group. |
    * | 9730 | You are not the creator of the group group. | If the procedure is owned by a group, only the user which is the creator of the group may load the procedure. |
    * | 2338 | Procedure name not found. | The procedure did not exist. |
    *
    * @see TdbGetProcLine()
    * @see TdbPutProcLine()
    * @see TdbPutProcedure()
    */
   int TdbGetProcedure(char* procedure);

   

   /**
    * @brief Load a macro/procedure line into an internal buffer.
    * @param line Line to add to procedure text
    * @param lineLength Length of line
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * The function TdbPutProcLine() loads a single macro/procedure line into an
    * internal buffer. The macro/procedure must have been loaded into the
    * buffer by a call to either the TdbGetProcedure() or TdbCopyProcedure()
    * function.
    *
    * If the macro/procedure is not new, all existing lines must first be read
    * by repeated calls to the TdbGetProcLine() function. The initial call to
    * TdbPutProcLine() will then empty the buffer and make the supplied line
    * the first in the macro/procedure, while subsequent calls append lines to
    ' the macro/procedure. It is for the application program to provide i
    * ntermediate storage for the lines.
    *
    * In order to store the macro/procedure the TdbPutProcedure() function has
    * to be called.
    *
    * #### Parameter Values
    * The \p line parameter is passed a character string as a line to add in
    * the procedure's body.
    *
    * The \p lineLength parameter is passed an integer as the length of the
    * line being added.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 17122 | CCL command line no number in command is too long. |  |
    *
    * @see TdbCopyProcedure()
    * @see TdbGetProcLine()
    * @see TdbGetProcedure()
    * @see TdbPutProcedure()
    */
   int TdbPutProcLine(char* line, int lineLength);
   

   /**
    * @brief Store a TRIP macro or procedure
    * @return 1 or other odd value on success, even value on error
    * @ingroup procmgmtxref
    *
    * #### Description
    * The function TdbPutProcedure() stores a TRIPsystem macro or procedure in
    * the CONTROL file. The call must have been preceded by a call to one of
    * the TdbGetProcedure() or TdbCopyProcedure() functions.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 32835 | Procedure id.name altered. | An existing procedure was successfully updated. |
    * | 32867 | Procedure id.name created. | A new procedure was successfully created. |
    * | 9538 | No write access to the database CONTROL. | The current user does not have write access to the CONTROL database. |
    * | 9538 | No write access to the database CONTROL. | The current user does not have write access to the CONTROL database. |
    *
    * @see TdbCopyProcedure()
    * @see TdbGetProcedure()
    */
   int TdbPutProcedure(void);

   
   // ------------------------------------------------------------------------
   
   /**
    * @defgroup grusadmxref Group and User Administration
    * @brief Administration of users and groups
    *
    * The group and user administration functions allows creation of group
    * and user maintenance rouintes.
    */


	/**
	 * @brief Inserts a TRIP username into the membership list of a group. 
	 * @param group Name of group to which the user is to be added
	 * @param user Name of user to add to the group
	 * @return 1 or other odd value on success, even value on error
	 * @ingroup grusadmxref
	 *
	 * #### Description
	 * The user acquires access to any database previously granted to the
	 * group and will also be able to run procedures created for the group
	 * by the group's manager. A return code of 33091 will be
	 * given upon successful completion of the function.
	 *
	 * Both the group and the user specified in the call must have been 
	 * created previously using the functions TdbCreateGroup() and
	 * TdbCreateUser(). A return code of 9698 or 9634 will be given if 
	 * the group or user has not been created.
	 *
	 * All TRIP users are members of the special user group PUBLIC and so
	 * cannot be added to that group. A return code of 10178 will be given
	 * if an attempt is made to add a user name to the PUBLIC group.
    *
    * #### Parameter Values
    * The \p group parameter is passed a string of characters as the name of
    * the group to update.
    *
    * The \p user parameter is passed a string of characters as the name of
    * the user to be added to the group.
    *
    * #### Privileges Required
    * The calling process must have user manager privilege and, further, must
    * be the user manager of the group to which the user is being added. The
    * calling process need not, however, be the manager of the user in question.
	 *
	 * #### Return Codes
	 * | Code | Description | Explanation |
	 * | ---- | ----------- | ----------- |
	 * | 1 | Operation completed successfully | The function completed its operations without error. |
	 * | 9634 | No such user as {username} | The user name passed in the user argument is unknown to the TRIP system. |
	 * | 9698 | No such group as {groupname} | The group name passed in the group argument is unknown to the TRIPsystem. |
	 * | 9730 | You are not the creator of the group {group}. | The filemanager calling this function is not the creator of the group named in the group argument. |
	 * | 9826 | You are not a user manager  | |
	 * | 10178 | All users are members of the group | All user ids are automatically made members of the group PUBLIC when they are created. This error is returned if a deliberate attempt is made to add a user id to the PUBLIC group with a call to this function. |
	 * | 10530 | {user} is already a member of {group}. | This error is returned when a subsequent attempt is made to add the named user to a group. |
	 * | 33091 | User {user} is now a member of group {group} | This code is returned upon successful completion of this function (ADD MEMBER). |
	 *
	 * @see TdbCreateUser()
	 * @see TdbCreateGroup()
	 * @see TdbDeleteMember()
	 * @see TdbDeleteGroup()
	 * @see TdbDeleteUser()
	 */
	int TdbAddMember(const char* group, const char* user);


	/**
	 * @brief Transfers ownership of a database from one user to another.
	 * @param old_fm Name of existing owner (file manager) of the database.
	 * @param new_fm Name of new owner (file manager) of the database.
	 * @param base Name of the database whose owner is being changed.
	 * @return 1 or other odd value on success, even value on error
	 * @ingroup grusadmxref
	 *
	 * #### Description
	 * The user SYSTEM can transfer ownership of any database from one 
	 * manager to another. Any other manager can only transfer the
	 * ownership of the databases that they own.
	 *
	 * If the base argument is set to a single asterisk character (*),
	 * then all of the databases that belong to the old file manager will
	 * be transferred to the new owner.
    *
    * #### Parameter Values
    * The \p old_fm parameter is passed a character string as the name of the
    * current file manager for the database.
    *
    * The \p new_fm parameter is passed a character string as the name of the
    * new file manager for the database.
    *
    * The \p base parameter is passed a character string as the name of the
    * database whose owner is being changed.
	 *
	 * #### Privileges Required
	 * The current TRIP user must possess the file manager (FM) privilege
	 * to execute this function.
	 *
	 * #### Return Codes
	 * | Code | Description | Explanation |
	 * | ---- | ----------- | ----------- |
	 * | 1 | Operation completed successfully | The function completed its operations without error. |
	 * | 608 | Database {name} not found | The database named in the base parameter cannot be located for this user. |
	 * | 9440 | You are not a SYSTEM manager | The TRIP username invoking this function is not a system manager. |
	 * | 10400 | No such user as {name} | There is no user with the name in the string passed via the newFM parameter. |
	 * | 10432 | There are no databases to transfer from {name} | There are no databases to transfer from the user named in the string passed via the oldFM parameter. |
	 * | 16672 | The current and the new manager name are the same | The user names passed in the strings via the oldFM and newFM parameters identify the same user. |
	 * | 17792 | You cannot change FM to the same user | You cannot change FM to the same user. |
	 * | 21568 | All databases moved from {username1} to {username2}. | All the databases owned |
	 *
	 * @see TdbSystemMode()
	 */
	int TdbChangeFm(const char* old_fm, const char* new_fm, const char* base);

   
	/**
	 * @brief Change password for the current user.
	 * @param oldpass Old password
	 * @param newpass New password
	 * @return 1 or other odd value on success, even value on error
	 * @ingroup grusadmxref
	 *
	 * #### Description
	 * This function allows the calling process to change the password of the
	 * current user. Only the SYSTEM user can change another user's password
	 * and to do so must call must call the TdbSetPassword() function.
	 *
	 * #### Return Codes
	 * | Code | Description | Explanation |
	 * | ---- | ----------- | ----------- |
	 * | 1 | Operation completed successfully | The function completed its operations without error.
	 * | 6240 | Password must contain alphanumeric characters | The password string passed via new password contains characters other than the alphanumerics.
	 * | 7680 | Old password validation error | The password string passed via old password failed validation by the TRIPkernel.
	 * | 18720 | Password is changed | The requested password change has been completed successfully.
	 *
	 * @see TdbCreateUser()
	 * @see TdbSetPassword()
 	 */
	int TdbChangePassword(const char* oldpass, const char* newpass);


	/**
	 * @brief Change the user manager for a TRIP user or group.
	 * @param old_um The name of the existing user manager responsible for the user or group.
	 * @param new_um The name of the new user manager responsible for the user or group.
	 * @param user_group User or group name
	 * @return 1 or other odd value on success, even value on error
	 * @ingroup grusadmxref
	 *
	 * #### Description
	 * This function directs the transfer of an existing TRIP user or group from
	 * one user manager to another. Once the transfer is effected, the new user
	 * manager gains control of the user or group, and in the case of a group,
	 * any group procedures that were created before the transfer.
	 *
	 * The user SYSTEM can transfer any user or group from one user manager to
	 * another. Any other user manager can transfer only those users or groups
	 * for which they are currently the manager.
	 *
	 * If the \p name argument is set to a single asterisk character (*), then
	 * all of the users that belong to the old manager will be transferred to the
	 * new manager.
	 * 
	 * The user SYSTEM, who owns all file and user managers, cannot transfer
	 * ownership for said users unless their file and/or user manager rights are
	 * first revoked. Consequently, only users without manager rights are 
	 * transferred if an asterisk is specified as the object to be transferred 
	 * from SYSTEM.
	 *
	 * #### Privileges Required
	 * The current TRIP user must possess the UM privilege to execute this function.
	 *
	 * #### Return Codes
	 * | Code | Description | Explanation |
	 * | ---- | ----------- | ----------- |
	 * | 1 | Operation completed successfully | The function completed its operations without error.
	 * | 327*32 | {Name} is not a user manager | The user named in the string passed via newUM is not a user manager.
	 * | 10496 | UM {name} has not created any users | The user manager named in the string passed via the oldUM has not created any users.
	 * | 16448 | Ownership of the group PUBLIC cannot be changed | The ownership of the group PUBLIC cannot be changed.
	 * | 16544 | FM/UM managers cannot be transfered | FM/UM managers cannot be transfered.
	 * | 17760 | You cannot change UM to the same user | The username in the string passed via the newUM parameter is identical to the original user manager.
	 * | 22752 | UM changed to {username1} for {username2}. | The user manager for the requested user (passed via the name has been changed to the user named in the string passed via the newUM parameter.
	 * | 22784 | All users moved from {username1} to {username2} | The all users managed by the user named in the string passed via oldUM have been moved to the user named in the string passed via newUM. Names of the user managers are included in the error message returned.
	 * | 23872 | {number} FM/UM manager(s) have not been transfered | One or more users could not be transfered to the new user manager because they are themselves file- or user-managers. The error message text will include a count of the number of users affected.
	 */
	int TdbChangeUm(const char* old_um, const char* new_um, const char* user_group);


   /**
    * @brief Check a user for membership of a group
    * @param group Group name string
    * @param user User name string
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbCheckMember() checks whether or not the specified user is
    * a member of the specified group.
    *
    * @note All users are members of the special group PUBLIC.
    *
    *
    * #### Parameter Values
    * The \p group parameter is passed a character string as the name of the
    * group of which membership is being checked.
    *
    * The \p user parameter is passed a character string as the name of the
    * user whose membership is being checked.
    *
    * #### Privileges Required
    * All users can check their own group memberships, but to check the
    * membership of other users, one has to be logged on as a user with UM
    * (user manager) privileges.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 9698 | No such group as group. |  |
    * | 9824 | You are not a user manager |  |
    * | 10498 | user is not a member of group. |  |
    *
    * @see TdbAddMember()
    * @see TdbCreateGroup()
    * @see TdbDeleteGroup()
    * @see TdbDeleteMember()
    */
   int TdbCheckMember(const char* group, const char* user);
   

   /**
    * @brief Create a user group
    * @param group Group name string
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbCreateGroup() creates a new user group with the name
    * supplied, and adds an entry for this group to the data dictionary. The
    * newly created group has no members and no access rights to databases.
    * Use the function TdbAddMember() to add members to the group.
    *
    * #### Parameter Values
    * The \p group parameter is passed a character string as the name of the
    * group to be created.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the user manager (UM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 33059 | Group name created. | Command completed successfully. |
    * | 9824 | You are not a user manager |  |
    * | 8770 | Username must be alphanumeric. |  |
    * | 10146 | Already existing group. |  |
    * | 10114 | Already existing user. | Attempt to create a group with the name like that of an existing user. |
    *
    * @see TdbAddMember()
    * @see TdbDeleteGroup()
    * @see TdbDeleteMember()
    */
   int TdbCreateGroup(const char* group);
   
   
	/**
	 * @brief Create a new user.
	 * @param username Name of user to be created
	 * @param password Password string
	 * @return 1 or other odd value on success, even value on error
	 * @ingroup grusadmxref
	 *
	 * #### Description
	 * This function creates a new user and adds an appropriate entry
	 * to the CONTROL file. The username and password assigned to the
	 * user are taken from the username and password parameters 
	 * respectively. The newly created user has no manager rights or 
	 * access to any databases. However, they will be able to use all
	 * PUBLIC procedures and those databases with access previously
	 * granted to the PUBLIC group. 
	 *
	 * #### Return Codes
	 * | Code | Description | Explanation |
	 * | ---- | ----------- | ----------- |
	 * | 32963 | User {name} has access to the system. | Command completed successfully. |
	 * | 9824  | You are not a user manager. | Only users who are user managers can create other users. |
	 * | 16386 | Registered user quota ({maxusers}) exceeded. | The installed TRIPsystem licence does not allow any more users to be created. |
	 * | 8770 | Username must be alphanumeric. | |
	 * | 6722 | Password must contain alphanumeric characters. | |
	 * | 10146 | Already existing group. | |
	 * | 10114 | Already existing user. | Attempt to create a user with the name like that of an existing user or group |
    *
    * @see TdbChangePassword()
    * @see TdbDeleteUser()
    * @see TdbSetPassword()
	 */
	int TdbCreateUser(const char* username, const char* password);
	
	
   /**
    * @brief Delete a user group
    * @param group Name of group to be deleted
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbDeleteGroup() removes the specified group from CONTROL if
    * the calling process is registered as the owner of the group in question.
    * The call also deletes any procedures and database access rights
    * associated with the group.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the user manager (UM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 33155 | Group name is deleted. |  |
    * | 9824 | You are not a user manager |  |
    * | 15170 | The group PUBLIC cannot be deleted. |  |
    * | 9698 | No such group as group. |  |
    * | 9730 | You are not the creator of the group group. |  |
    */
   int TdbDeleteGroup(const char* group);


   /**
    * @brief Delete a user from a group
    * @param group Group name string
    * @param user User name string
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbDeleteMember() deleted the username as passed via the
    * \p user parameter from the group named in the \p group parameter. The
    * TRIPsystem user ID for the current session must be the user manager of
    * the group from which the user is being deleted, but need not be the
    * creator of the user. All TRIP users are members of the group PUBLIC, and
    * so cannot be added to, or removed from that group. The routine removes
    * the specified TRIP user ID from the member list for the specified TRIP
    * user group. The user ceases to be a member of the group, and loses any
    * database or procedure access which he or she enjoyed exclusively because
    * of his or her membership of the group.
    *
    * #### Parameter Values
    * The \p group parameter is passed a character string as the name of the
    * group from which the user is to be deleted.
    *
    * The \p user parameter is passed a character string as the name of the
    * user to be deleted from the group.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the user manager (UM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 33184 | User username is no longer a member of group groupname. | Operation completed successfully. |
    * | 9824 | You are not a user manager |  |
    * | 9698 | No such group as group. |  |
    * | 9730 | You are not the creator of the group group. |  |
    * | 9634 | No such user as username. |  |
    * | 10178 | All users are members of the group PUBLIC |  |
    * | 10210 | Members of the group PUBLIC cannot be deleted. |  |
    * | 10498 | user is not a member of group. |  |
    */
   int TdbDeleteMember(const char* group, const char* user);

   

   /**
    * @brief Delete a TRIP user ID
    * @param user Name of user to delete
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbDeleteUser() deletes the specified user from the CONTROL
    * file. It also deletes any procedures owned by the user. The TRIPsystem
    * user ID for the current session must be the owner of the TRIP user ID to
    * be deleted.
    *
    * The user ID to be deleted must not be the current manager for either any
    * existing TRIP users or databases. Thus if the user to be deleted has user
    * manager rights within the TRIP system, any objects created by the user
    * must first be transferred to some other user manager. Similarly, the
    * ownership of any databases currently managed by the user must be
    * transferred to another file manager. The CONTROL file records for the
    * groups in which the user was formerly a member are updated to remove the
    * user membership link.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the user manager (UM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 32931 | User name has lost access to the system. | Operation completed successfully. |
    * | 9826 | You are not a user manager. | The logged on user must have user manager privileges in order to delete users. |
    * | 9634 | No such user as user. | Attempt to delete a user that does not exist. |
    * | 10274 | You have not created this user. | A user can only be deleted by its creator. |
    * | 8930 | User manager with existing users/groups cannot be deleted. | A user who is a user manager can not be deleted while there remains user created by this user manager. |
    * | 8962 | File manager with existing databases cannot be deleted. | The owner of existing databases cannot be deleted. |
    */
   int TdbDeleteUser(const char* user);


   /**
    * @brief
    * @param group Name of group
    * @param memberCount Number of members of group
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbGetGroup() loads the specified group record into main
    * memory.
    *
    * #### Parameter Values
    * The \p group parameter is passed a character string as the name of group
    * to be retrieved.
    *
    * The \p memberCount parameter receives the number of members of the
    * specified group.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the user manager (UM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 9826 | You are not a user manager. | The logged on user must have user manager privileges in order to delete users. |
    * | 9730 | You are not the creator of the group group. |  |
    * | 9698 | No such group as group. |  |
    * | 10178 | All users are members of the group PUBLIC |  |
    *
    * @see TdbGetMembers()
    */
   int TdbGetGroup(char* group, int* memberCount);

   
   /**
    * @brief Get the group name for a group specified by number
    * @param groupName Group name
    * @param gid Group identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbGetGroupName() returns the group name for a group
    * specified by the \p gid number.
    *
    * #### Parameter Values
    * The \p groupName parameter is passed a string buffer that receives the
    * name of the group to be retrieved. This buffer must be at least 128 bytes
    * in size.
    *
    * The \p gid parameter is passed the group identification number of the
    * group to be retrieved.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    */
   int TdbGetGroupName(char* groupName, int gid);

   
   /**
    * @brief Retrieve the members of a group
    * @param members Names of group member
    * @param length Size of members buffer
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbGetMembers() retrieves all members of the group
    * previously loaded by a call to TdbGetGroup(). Member names are separated
    * by a zero byte.
    *
    * #### Parameter Values
    * The \p members parameter is passed a string buffer that receives the
    * names of the members of the selected group.
    *
    * On input, the \p length is to be assigned the size of the \p members
    * buffer. On output, it receives a count of the number of bytes written
    * to the \p members buffer.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 3 |  | Operation completed successfully, but there is more data to fetch by repeated calls to this function. |
    * | 9826 | You are not a user manager |  |
    *
    * @see TdbGetGroup()
    */
   int TdbGetMembers(char* members, int* length);
   
   
   /**
    * @brief Retrieve the profile record for a TRIP user
    * @param user Name of user
    * @param profile User profile record
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbGetUserProfile() reads the user profile record for the
    * specified user from the CONTROL file. This user profile record contains
    *  two types of information about the user. The first type is informational
    * only (e.g. telephone number and address), and does not affect the
    * behavior of the system. The second type of information (e.g. preferred
    * date format and start module) can be used to modify the behavior of the
    * TRIP system.
    *
    * A user profile record can only be obtained by the user to which it
    * belongs, the owner of the user, and user SYSTEM.
    *
    * #### Parameter Values
    * The \p user parameter is passed a character string as the name of the
    * user whose profile is retrieved.
    *
    * The \p profile is passed a user profile record to receive the profile
    * record for the named user.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the user manager (UM) privilege,
    * be the SYSTEM user, or be the user who owns the object.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 9634 | No such user as username. |  |
    * | 9826 | You are not a user manager |  |
    * | 10690 | You are not the creator of user name. |  |
    *
    * @see TdbPutUserProfile()
    */
   int TdbGetUserProfile(char* user, user_profile_rec* profile);

   
   /**
    * @brief Retrieve the manager rights for a TRIP user
    * @param user Name of user
    * @param UM User manager rights flag
    * @param FM File manager rights flag
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbGetUserRights() returns the user and file manager rights
    * for the TRIP user ID supplied. The user name is passed via the \p user
    * parameter and the user manager and file manager rights are returned in
    * the \p UM and \p FM parameters respectively. Note that a TRIPsystem user
    * group cannot have manager rights, and hence this routine must have a
    * TRIPsystem user ID as the value of the \p user parameter.
    *
    * #### Parameter Values
    * The \p user parameter is passed a character string as the name of the
    * user whose rights are retrieved.
    *
    * The \p UM parameter is passed a byte to receive the user manager rights
    * of the user. The lowest bit set indicates that the user has user manager
    * rights within the TRIP system; if this bit is clear, the user has no
    * user manager rights.
    *
    * The \p FM parameter is passed a byte to receive the file manager rights
    * of the user. The lowest bit set indicates that the user has file manager
    * rights within the TRIP system; if this bit is clear, the user has no
    * file manager rights.
    *
    * #### Privileges Required
    * The current TRIPsystem user must be the SYSTEM user.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 9634 | No such user as username |  |
    * | 9666 | User id id is a group id. |  |
    * | 9826 | You are not a user manager |  |
    *
    * @see TdbPutUserRights()
    */
   int TdbGetUserRights(const char* user, boolean* UM, boolean* FM);

   
   /**
    * @brief Store the profile record for a TRIP user
    * @param profileRecord Profile record for user
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbPutUserProfile() writes into the CONTROL file the user
    * profile record for the specified user. The calling process needs
    * TRIPsystem user manager rights (if it is not his/her own profile). The
    * profile record contains information about the user which is of two types.
    * The first type is informational only, and does not affect the behavior
    * of the system. The second type of information stored in the user profile
    * record can be used to modify the behavior of the TRIP system.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the user manager (UM) privilege
    * to execute this function.
    *
    * Any user is permitted to alter their own profile.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 35043 | User profile modified. | Operation completed successfully. |
    * | 9826 | You are not a user manager |  |
    *
    * @see TdbGetUserProfile()
    */
   int TdbPutUserProfile(user_profile_rec* profileRecord);
   
   
   /**
    * @brief Set the manager rights for a TRIP user
    * @param UM User manager right
    * @param FM File manager right
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbPutUserRights() uses the TRIP user ID previously loaded
    * with a call to the function TdbGetUserRights(), and records the new user
    * and file manager rights for this user in the CONTROL file.
    *
    * File and user manager rights can only be granted and revoked by user
    * SYSTEM. When a user is given manager rights, the ownership of that user
    * is automatically transferred to SYSTEM .
    *
    * #### Parameter Values
    * The \p UM parameter is passed a single byte as a flag to indicate the
    * setting of the user manager rights for the user being checked. The
    * lowest bit set indicates that the user has user manager rights within
    * the TRIP system; if this bit is clear, the user has no user manager
    * rights.
    *
    * The \p FM parameter is passed a single byte as a flag to indicate the
    * setting of the file manager rights for the user being checked. The lowest
    * bit set indicates that the user has file manager rights within the TRIP
    * system; if this bit is clear, the user has no file manager rights.
    *
    * #### Privileges Required
    * The current TRIPsystem user must be the SYSTEM user.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 38627 | User name now has FM manager rights. | Success. File manager rights granted. |
    * | 38659 | User name now has UM manager rights. | Success. User manager rights granted. |
    * | 38691 | User name now has FM and UM manager rights. | Success. User and file manager rights granted. |
    * | 38723 | User name now has no manager rights. | Success. Manager rights revoked |
    * | 9922 | You are not a SYSTEM manager. |  |
    * | 7362 | Manager rights of the user SYSTEM cannot be changed. |  |
    * | 11714 | FM rights cannot be revoked if own databases exists. | File manager rights cannot be revoked for a user who owns a database. |
    * | 11746 | UM rights cannot be revoked if own users/groups exists. | User manager rights cannot be revoked for a user who owns users or groups. |
    *
    * @see TdbGetUserRights()
    */
   int TdbPutUserRights(boolean UM, boolean FM);

   
   /**
    * @brief Retrieve the current O/S user ID
    * @param OSuser Operating system user ID
    * @param length Length of operating system user ID.
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbOsUser() returns the operating system user ID for the
    * current TRIPsystem session.
    *
    * #### Parameter Values
    * The \p OSuser parameter is passed a character string to receive the name
    * of the operating system user ID for the current session.
    *
    * The \p length parameter is passed a pointer to an \c int variable to 
    * receive the length of the returned operating system user ID.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    */
   int TdbOsUser(char* OSuser, int* length);

   
   /**
    * @brief Set a user password
    * @param user Name of user
    * @param password New password
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbSetPassword() allows the TRIP System Manager to set the
    * password of an existing TRIP user.
    *
    * #### Parameter Values
    * The \p user parameter is passed a character string as the name of the
    * user whose password is being set.
    *
    * The \p password parameter is passed a character string as the new
    * password string for the user.
    *
    * #### Privileges Required
    * From version 4 of TRIP, any user who has user manager (UM) rights may
    * use this function. For older versions of TRIP, the current user must be
    * the SYSTEM user.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 32163 | Password is changed. | Operation completed successfully. |
    * | 9922 | You are not a SYSTEM manager. |  |
    * | 6722 | Password must contain alphanumeric characters. |  |
    * | 9634 | No such user as username. |  |
    * | 9762 | You are not the creator of this record. |  |
    */
   int TdbSetPassword(const char* user, const char* password);

   
   /**
    * @brief Switch to TRIP System ID
    * @param password Password string
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbSystemMode() is used to temporarily switch the current
    * session from the logged in TRIP user ID to that of the TRIP system
    * manager. A call to this routine with the correct system managers
    * password will grant the calling process user, file and system manager
    * rights within the current session. A subsequent call to the routine with
    * an empty password will cause the rights of the current session to revert
    * to those of the logged in TRIP user ID.
    *
    * #### Parameter Values
    * The \p password parameter is passed a character string as the password
    * for the TRIP user SYSTEM. If the password has zero length then the TRIP
    * will revert to the original user.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 10242 | Invalid password. | The supplied password did not match that of the user SYSTEM. |
    */
   int TdbSystemMode(const char* password);


   /**
    * @brief Retrieve the current TRIP user ID
    * @param userName Name of current TRIP user
    * @param length Length of user name
    * @return 1 or other odd value on success, even value on error
    * @ingroup grusadmxref
    *
    * #### Description
    * The function TdbTdbsUser() returns the TRIP ID for the current TRIP
    * session, i.e. the logged in TRIP user name which was used in the call to
    * the TdbCheckUser() function.
    *
    *
    * #### Parameter Values
    * The \p userName parameter is passed a character string to receive the
    * name of the current TRIP user.
    *
    * The \p length parameter is passed a pointer to an \c int variable to 
    * receive the count of the number of bytes in the returned user name.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    */
   int TdbTdbsUser(char* userName, int* length);

   
   // ------------------------------------------------------------------------

   /**
    * @defgroup basemgmt Database Management
    * @brief Database administration and information.
    *
    * The TRIPtoolkit database management functions offer functionality for
    * querying status on and defining databases, clusters and thesauri.
    */

   
   /** @name General Database Functions
    *  General database administration and information functions.
    *  @{
    */

   
   /**
    * @brief Retrieve selected information about a database
    * @param base Database name
    * @param mode Mode of operation
    * @param cvalue Information as character string
    * @param ivalue Information as integer
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * Returns various information about a database.
    *
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string as the name of the
    * database to interogate.
    *
    * The \p mode parameter is passed an \c int as a flag to request information
    * about the database named by the \p base parameter. Valid values are:
    *
    * | Symbolic Name | Output Parameter | Purpose |
    * | ------------- | -------------- | --------- |
    * | BASEINFO_XML | ivalue | The ivalue parameter receives the value 1 (one) if the database is XML enabled, and 0 (zero) if it is not. |
    * | BASEINFO_CHARFOLD | ivalue | Character folding class |
    * | BASEINFO_RNAMEFLD | cvalue | Record name field |
    * | BASEINFO_MAXFLD | ivalue | Highest field number used |
    * | BASEINFO_PNAMEFLD | cvalue | Part name field |
    * | BASEINFO_LANGUAGE | ivalue | Database language code. The ivalue parameter receieves a value as defined by the LANGUAGE_* constants (e.g. LANGUAGE_ENGLISH, etc). |
    * | BASEINFO_CHARSET | ivalue | The character set set for the database. The ivalue parameter receieves a value as defined by the CHARSET_* constants (e.g. CHARSET_UTF8, CHARSET_LA1, etc). |
    * | BASEINFO_RNUMFLD | ivalue | The record number field for the database. |
    * | BASEINFO_MAXRNUM | ivalue | The highest record number in the database. |
    * | BASEINFO_OWNER | cvalue | The name of the TRIP user that owns the database |
    * | BASEINFO_GRAPH | ivalue | The ivalue parameter receives the value 1 (one) if the database is a graph database, and 0 (zero) if it is not. |
    * | BASEINFO_CLUSTERBASES | cvalue | If the database is an opened cluster, the cvalue parameter receives a comma-separated list of the clustered databases. |
    * | BASEINFO_DBNUMBER | ivalue | The number of the CONTROL database record that holds the database design |
    * | BASEINFO_DBDESC | cvalue | The description/comment text for the database |
    * | BASEINFO_CONNECTOR | ivalue | The ivalue parameter receives the value 1 (one) if the database is a TRIPcof connector database, and 0 (zero) if it is not. |
    * | BASEINFO_RECORDCOUNT | ivalue | The number of records in the database |
    * | BASEINFO_DESIGN_CREDATE | cvalue | The date at which the database design was created |
    * | BASEINFO_DESIGN_CRETIME | cvalue | The time at which the database design was created |
    * | BASEINFO_DESIGN_MODDATE | cvalue | The date at which the database design was last modified|
    * | BASEINFO_DESIGN_MODTIME | cvalue | The time at which the database design was last modified |
    * | BASEINFO_THESAURUS | ivalue | The ivalue parameter receives the value 1 (one) if the name is that of a thesaurus, or 0 (zero) if it is not. |
    * | BASEINFO_CONTAINER | ivalue | The ivalue parameter receives the value 1 (one) if the name is that of a classification container, or 0 (zero) if it is not. |
    * | BASEINFO_UTC | ivalue | The ivalue parameter receives the value 1 (one) if the database uses UTC timestamps, or 0 (zero) if it uses the older TRIP-specific timestamp format. |
    *
    * The \p cvalue is passed a character string buffer that receives
    * information represented as charcter data.
    *
    * The \p ivalue is passed a pointer to a 32-bit signed integer that
    * receives information represented as an integer value.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM)
    * privilege to execute this function.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 7394 | No record name field defined in base. | The specified database has no record name field. May occur when the mode parameter is set to BASEINFO_RNAMEFLD and the specified database has no part record name field. |
    * | 17986 | Database base has no part name field. | The specified database has no part name field. May occur when mode is set to BASEINFO_PNAMEFLD and the specified database has no part name field. |
    * | 18466 | Unrecognized option: mode | The option value specified in the mode parameter is not valid. |
    *
    * @see TdbBaseStatus()
    */
   int TdbBaseInfo(const char* base, int mode, char* cvalue, int* ivalue);


   /**
    * @brief Retrieve information on a named database cluster
    * @param mode Mode of operation
    * @param cluster Name of database cluster
    * @param outstr Buffer to receive string information 
    * @param outstrsz The allocated size of the outstr buffer
    * @param outint Pointer to 64-bit integer to receive integer information
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * Returns various information about a database cluster.
    *
    * #### Parameter Values
    * The \p cluster parameter is passed a character string as the name of the
    * database cluster to query.
    *
    * The \p mode parameter is passed an \c int as a flag to request information
    * about the database named by the \p base parameter. Valid values are:
    *
    * | Symbolic Name | Output Parameter | Purpose |
    * | ------------- | -------------- | --------- |
    * | CLUSTERINFO_RECORDCOUNT | outint | The outint parameter receives the total number of records in the database cluster. |
    * | CLUSTERINFO_MEMBERCOUNT | outint | The outint parameter receives the total number of databases in the database cluster. |
    * | CLUSTERINFO_BASECOUNT | outint | The outint parameter receives the total number of immediate members of the database cluster. |
    * | CLUSTERINFO_DATABASES | outint,outstr | The outstr parameter receives a comma separated list of all databases in the cluster. Set the outstr parameter to NULL to obtain the required minumum buffer size in the outint parameter. |
    * | CLUSTERINFO_MEMBERS | outint,outstr | The outstr parameter receives a comma separated list of the immedate members of the database cluster. Set the outstr parameter to NULL to obtain the required minumum buffer size in the outint parameter. |
    * | CLUSTERINFO_UPDATED | outstr | The outstr parameter receives the most recent date and time that the databasse in the cluster have been updated. |
    * 
    * #### Since
    * This function was introduced in TRIP version 8.4-0
    * 
    * @see TdbBaseStatus()
    * @see TdbSearchInfo()
    * @see TdbSessionInfo()
    * @see TdbBaseInfo()
    */
   int TdbClusterInfo(int mode, const char* cluster, char* outstr, size_t outstrsz, int64_t* outint);
   
   
   /**
    * @brief Retrieve certain information concerning the state of a given database or permanent cluster
    * @param base Name of a database or a permanent cluster
    * @param uppdateDate Date of last update to the database
    * @param indexDate Date of last index.
    * @param recordCount Number of records in database.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The TdbBaseStatus() returns information describing the current state of
    * a specified TRIPsystem database or permanent cluster. The information 
    * returned is the date of the last update to the database, the date when
    * the database was last indexed, and the total number of records in the
    * database (un-indexed records included).
    *
    * For the update date and the index date, the format the date is returned
    * in depends on the dateform setting established by the calling process.
    * Changes to the system default dateform can be made using dateform setting
    * in the user's profile, or by calling the TdbShellDefDateForm() function.
    *
    * #### Parameter Values
    * The \p base parameter is passed a string containing the name of the
    * database or permanent cluster for which to get information.
    *
    * The \p uppdateDate parameter receives a string containing the date of
    * the last update to the database. If \p base is a cluster, this date is
    * the most recent update date among the databases in the cluster. The 
    * string buffer passed as this parameter has to be large enough to
    * accommodate a date in the current format.
    *
    * The \p indexDate parameter receives a string containing the date of last
    * index. If \p base is a cluster, this parameter will receive an empty
    * string.
    *
    * The \p recordCount parameter receives an integer count of the number of
    * records currently in the database. If \p base is a cluster, this parameter
    * receives the total record count of all databases in the cluster.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 15234 | Missing database name. | The name of the database to retrieve information about must be specified. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 8802 | Only alphanumerics/underscores allowed in database/cluster names. | The specified database name contained invalid characters. |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 1934 | Database name not found. | The specified database did not exist. |
    * | 2530 | No access to database name. | The logged on user did not have access to the specified database. |
    *
    * @see TdbShowControl()
    * @see TdbNextControl()
    * @see TdbClusterInfo()
    * @see TdbShellDefDateForm()
    */
   int TdbBaseStatus(const char* base, char* revisionDate, char* indexDate,
                     int* recordCount);
   

   
   /** @} */     // END OF GENERAL DB FUNCTIONS BLOCK
   
   
   /** @name Access Management Functions
    *  Functions for access management to databases, clusters and thesauri.
    *  @{
    */

   /**
    * @brief Check access rights to a database
    * @param base Name of database
    * @param accessMode Access mode to be checked.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The TdbCheckBase() will report whether the calling process has the
    * specified degree of access to the database in question. This can be
    * useful if, for instance, the calling process needs to ensure that only
    * the owner of a database can perform a certain action.
    *
    * Note that in order to be able to change a user's access to a database,
    * using either of the functions TdbGetDbAccess() or TdbPutDbAccess(), the
    * calling process must first register itself as the file manager by
    * calling this function with the mode set to \p CHECK_OWNERSHIP .
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string as the name of a
    * database to be checked for the user's current level of access.
    *
    * The \p accessMode parameter is passed a ( long ) int value for the access
    * mode to be checked. Valid values are:
    *
    * | Symbolic Value | Constant Value | Purpose |
    * | -------------- | -------------- | ------- |
    * | CHECK_READ | 1 | The calling process requires read access to the database |
    * | CHECK_UPDATE | 2 | The calling process requires update access to the database |
    * | CHECK_OWNERSHIP | 3 | The calling process requires file manager (FM) access to the database |
    * | CHECK_DELETE | 4 | The calling process requires access to delete records, i.e. that it has write access to ALL fields in the database. |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 15234 | Missing database name. | The name of the database to retrieve information about must be specified. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 8802 | Only alphanumerics/underscores allowed in database/cluster names. | The specified database name contained invalid characters. |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 610 | Database name not found. | The specified database did not exist. |
    * | 2530 | No access to database name. | The logged on user did not have access to the specified database. |
    * | 9538 | No write access to database name. | The logged on user did not have write access to the specified database. |
    * | 10594 | No delete rights to database name. | The logged on user did not have delete access to the specified database. |
    * | 16770 | You are not the owner of this DB cluster.. |  |
    * | 2498 | You have no FM access to the database. |  |
    *
    * @see TdbCheckDbCluster()
    * @see TdbGetDbAccess()
    * @see TdbPutDbAccess()
    * @see TdbSetBase()
    */
   int TdbCheckBase(const char* base, int accessMode);

   
   /**
    * @brief Check access rights to a database cluster
    * @param clusterName Name of database cluster
    * @param accessMode Access mode
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The TdbCheckDbCluster() is used to ensure a level of access to a specific
    * database cluster. This can be used, for instance, to guarantee that only
    * the owner of a cluster can perform certain operations.
    *
    * #### Parameter Values
    * The \p clusterName parameter is passed a character string as the name of
    * the (predefined) database cluster to be examined.
    *
    * The \p accessMode parameter is passed an \c int that represents the
    * access mode to be checked. Valid values are:
    *
    * | Symbolic Constant | Value of Constant | Purpose |
    * | ----------------- | ----------------- | ------- |
    * | CHECK_READ | 1 | The calling process requires read access to the cluster |
    * | CHECK_UPDATE | 2 | The calling process requires updating of the cluster |
    * | CHECK_OWNERSHIP | 3 | The calling process requires file manager (FM access to the cluster |
    * | CHECK_DELETE | 4 | The calling process requires delete access to records, i.e. has write access to ALL fields in the cluster |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 15234 | Missing database name. | The name of the database to retrieve information about must be specified. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 8802 | Only alphanumerics/underscores allowed in database/cluster names. | The specified database name contained invalid characters. |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 1934 | Database name not found. | The specified database did not exist. |
    * | 2530 | No access to database name. | The logged on user did not have access to the specified database. |
    * | 9538 | No write access to database name. | The logged on user did not have write access to the specified database. |
    * | 10594 | No delete rights to database name. | The logged on user did not have delete access to the specified database. |
    * | 16770 | You are not the owner of this DB cluster.. |  |
    * | 2498 | You have no FM access to the database. |  |
    *
    * @see TdbCheckBase()
    */
   int TdbCheckDbCluster(const char* clusterName, int accessMode);

   
   /**
    * @brief Check access rights to a thesaurus
    * @param thesaurus Name of thesaurus
    * @param accessMode Access mode to be checked
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbCheckThes() is used to verify that the calling process
    * can gain a specified level of access to the thesaurus named in the
    * \p thesaurus parameter. This can be useful if, for example, you wish to
    * guarantee that only the owner of the thesaurus can perform certain
    * functions.
    *
    * #### Parameter Values
    * The \p thesaurus parameter is passed a character string as the name of
    * the thesaurus.
    *
    * The \p accessMode parameter is passed an \c int as the access mode to be
    * checked. Valid values are:
    *
    * | Symbolic Constant | Constant Value | Purpose |
    * | ----------------- | -------------- | ------- |
    * | CHECK_READ | 1 | The calling process requires read access to the thesaurus |
    * | CHECK_UPDATE | 2 | The calling process requires update access to the thesaurus |
    * | CHECK_OWNERSHIP | 3 | The calling process requires owner access to the thesaurus |
    * | CHECK_DELETE | 4? | The calling process requires delete access, i.e. has write access to ALL fields in the thesaurus |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 15234 | Missing database name. | The name of the database to retrieve information about must be specified. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 8802 | Only alphanumerics/underscores allowed in database/cluster names. | The specified database name contained invalid characters. |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 1934 | Database name not found. | The specified database did not exist. |
    * | 2530 | No access to database name. | The logged on user did not have access to the specified database. |
    * | 9538 | No write access to database name. | The logged on user did not have write access to the specified database. |
    * | 10594 | No delete rights to database name. | The logged on user did not have delete access to the specified database. |
    * | 16770 | You are not the owner of this DB cluster.. |  |
    * | 2498 | You have no FM access to the database. |  |
    *
    * @see TdbCheckBase()
    * @see TdbCheckDbCluster()
    */
   int TdbCheckThes(const char* thesaurus, int accessMode);
    
    
    /**
     * @brief Load the base access record for a TRIP user
     * @param user Name of user
     * @param baseAccess Base access record
     * @return 1 or other odd value on success, even value on error
     * @ingroup basemgmt
     *
     * #### Description
     * The function TdbGetDbAccess() reads from the CONTROL file the base
     * access record for the specified user and the database, thesaurus or
     * cluster previously loaded by a call to one of the TdbCheckBase(),
     * TdbCheckThes() or TdbCheckDbCluster() functions. The \p base access
     * parameter to those routines must have specified the \c CHECK_OWNERSHIP
     * option.
     *
     * Each byte in the base access record that is used to specify the
     * total/selective field access is a bit mask where bit 0 indicates read
     * access and bit 1 write access.
     *
     * #### Parameter Values
     * The \p user parameter is passed a character string as the name of the
     * user whose base access record is to be retrieved.
     *
     * The \p baseAccess parameter is passed a base access record structure to
     * receive the base access record for the user.
     *
     * #### Privileges Required
     * The current TRIPsystem user must possess the file manager (FM) privilege
     * to execute this function.
     *
     * #### Return Codes
     * | Code | Description | Explanation |
     * | ---- | ----------- | ----------- |
     * | 1 | Operation completed successfully | The function completed its operations without error. |
     * | 2498 | You have no FM access to the database. |  |
     * | 9794 | You are not a file manager. |  |
     * | 9634 | No such user as name. |  |
     * | 10562 | You are the owner of this database, and have all rights. |  |
     *
     * @see TdbCheckBase()
     * @see TdbCheckThes()
     * @see TdbCheckDbCluster()
     */
   int TdbGetDbAccess(char* user, base_access_rec* baseAccess);

   
   /**
    * @brief Store the base access record for a TRIP user
    * @param baseAccess Base access record
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbPutDbAccess() writes to the CONTROL file the base access
    * record previously loaded by a call to the TdbGetDbAccess() function.
    * Before the access record can be obtained, however, the function
    * TdbCheckBase() have been called, specifying access mode CHECK_OWNERSHIP
    * option.
    *
    * Each byte in the base access record used to specify the total/selective
    * field access is a bit mask where bit 0 indicates read access and bit 1
    * write access.
    *
    * #### Parameter Values
    * The \p baseAccess parameter is passed a base access record to describe
    * the access rights to the database for the user.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 33219 | User username has access to database dbname. | Operation completed successfully - specified user was granted access to the database. |
    * | 33283 | Group groupname has access to database dbname. | Operation completed successfully - specified group was granted access to the database. |
    * | 33283 | Group groupname has lost access to database dbname. | Operation completed successfully - access to the database was revoked from the specified group. |
    * | 33347 | Group groupname has lost access to database dbname. | Operation completed successfully - access to the database was revoked from the specified group. |
    * | 38755 | User username has no access to database dbname. | Operation completed successfully. Access to the database was revoked from a user who did not have access to it. |
    * | 38787 | Group groupname has no access to database dbname. | Operation completed successfully. Access to the database was revoked from a group that did not have access to it. |
    * | 9794 | You are not a file manager. |  |
    *
    * @see TdbGetDbAccess()
    */
   int TdbPutDbAccess(base_access_rec* baseAccess);

   
   
   /** @} */     // END OF DB ACCESS FUNCTIONS BLOCK
   
   
   /** @name Database Design Functions
    *  Functions for management of database designs.
    *  @{
    */

   /**
    * @brief Check for the existence of a field.
    * @param field_name Name of field to check
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * Checks for the existence of a field in the currently loaded database
    * design, so a call to this function must be preceded by a call to
    * TdbGetBaseDesign().
    *
    * @see TdbGetBaseDesign()
    */
   int TdbCheckFieldName(const char* field_name);

   
   /**
    * @brief Create a copy of a database definition
    * @param from_base Name of (existing) database
    * @param to_base Name of (new) database
    * @param design Database specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbCopyBaseDesign() initiates the copying of one database
    * design to another of a different name. Upon a successful return the
    * \p design parameter will contain a handle to a new database design
    * record containing all the general properties of the new database as
    * copied from the old.
    *
    * In order to commit the new database to CONTROL the process must
    * subsequently call the TdbPutBaseDesign() function.
    *
    * Note that the database file name properties (for the BAF, BIF, VIF and
    * LOG files) are not copied by this operation.
    *
    * #### Parameter Values
    * The \p from_base parameter is passed a character string as the name of the
    * existing database whose design is to be copied.
    *
    * The \p to_base parameter is passed a character string as the name of
    * the new database.
    *
    * The \p design parameter is passed a pointer to a TdbHandle variable that
    * will receive the database design record for the new database. Values will
    * be copied form the equivalent record for the old database. The handle must
    * be explicitly released using TdbCloseBaseDesign() when the application
    * no longer needs it.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 34019 | Database design copied. | Operation completed successfully. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 7586 | Already existing database design: name. | Target database must not exist. |
    *
    * #### Since
    * This function was introduced in TRIP version 8.0-0.
    *
    * @see TdbGetBaseDesign()
    * @see TdbPutBaseDesign()
    * @see TdbCloseBaseDesign()
    */
   int TdbCopyBaseDesign(const char* from_base, const char* to_base,
                         TdbHandle* design);
   
   
   /**
    * @brief Delete a previously loaded database design
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbDeleteBaseDef() deletes the database design
    * previously loaded by a call to TdbCheckBase() or TdbGetBaseDesign().
    * Any entry forms or output formats defined for the database are also
    * deleted.
    *
    * This deletion affects not only the database itself, but also any users
    * or groups who had access to that database have their access removed.
    * Considerable care should be exercised in calling this function as other
    * users may have the database open when this function is called. Those
    * other users may experience unexpected behavior from any TRIP application
    * including TRIPclassic and TRIPmanager.
    *
    * The database must be empty at the time of deletion.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 32579 | Database design for name deleted. | Operation completed successfully. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 2498 | You have no FM access to the database. | The logged on user need to have file manager rights in order to use this function. |
    * | 7938 | The database name is not empty. | There must not be any records in a database that is to be deleted. |
    *
    * @see TdbCheckBase()
    * @see TdbGetBaseDesign()
    */
   int TdbDeleteBaseDef();
   

   /**
    * @brief Store a database definition
    * @param handle Database design handle
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbPutBaseDesign() writes the database definition assocated
    * with the provided \p handle into the CONTROL file. A call to this function
    * must have been preceded by a call to either of the
    * TdbGetBaseDesign() or TdbCopyBaseDesign() functions.
    *
    * #### Parameter Values
    * The \p handle parameter is passed handle to the database obtained via
    * either of the TdbGetBaseDesign() or TdbCopyBaseDesign() functions.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 32515 | Database design for name altered. | Database design successfully altered. |
    * | 32547 | Database design for name created. | Database design successfully created. |
    * | 33411 | Database design for name altered (check logical names). | Database design successfully altered, but there may be something wrong with the BAF/BIF/VIF file names. |
    * | 33443 | Database design for name created (check logical names). | Database design successfully created, but there may be something wrong with the BAF/BIF/VIF file names. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 11842 | Unbalanced parenthesis in database description. |  |
    * | 14786 | id is not a valid character folding class. |  |
    *
    * @see TdbPutBaseDesign()
    * @see TdbCopyBaseDesign()
    * @see TdbGetBaseDesign()
    */
   int TdbPutBaseDesign(TdbHandle handle);

   /**
    * @brief Load a text separator specification
    * @param separatorSpec Separator specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbGetTextSeparators() loads the separator specification
    * record of a database into the given record area. It can only be used
    * after calling the TdbGetBaseDesign() function.
    *
    * #### Parameter Values
    * The \p separatorSpec parameter is passed a separator specification
    * record to receive the separator specification record for the database.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * |  |  |  |
    *
    * @see TdbGetBaseDesign()
    * @see TdbPutBaseDesign()
    
    */
   int TdbGetTextSeparators(text_sepa_rec* separatorSpec);
   

   /**
    * @brief Store a text separator specification
    * @param separatorSpec Separator specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbPutTextSeparators() stores the separator specification
    * record into the internal database definition buffer that has been loaded
    * using the TdbGetBaseDesign() function. A subsequent call to the function
    * TdbPutBaseDesign() commits the database definition to the CONTROL file.
    *
    * #### Parameter Values
    * The \p separatorSpec parameter is passed a separator specification record.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 16642 | {cls} is not a valid separator class. |  |
    * | 16674 | {chr} is not a valid class cls character. |  |
    *
    * @see TdbGetTextSeparators()
    * @see TdbGetBaseDesign()
    * @see TdbPutBaseDesign()
    */
   int TdbPutTextSeparators(text_sepa_rec* separatorSpec);

   
   /**
    * @brief Load a database design
    * @param handle Pointer to a TdbHandle variable that receives a handle
    *        to the database design.
    * @param base Name of database
    * @param accessMode Access rights required to the database design
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function loads the definition of the specified TRIP database and
    * returns a \p handle to it. Use the TdbGetBaseProperty() function to
    * read the properties of the database design.
    *
    * The \p accessMode parameter specifies how the calling process intends to
    * use the database design. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | MODE_READ | bit 0 | Require read access to the database design |
    * | MODE_WRITE | bit 1 | Require modify access to the database design |
    * | MODE_DELETE | bit 2 | Require delete access to the database design |
    * | MODE_COPY | bit 3 and bit 1 | Require copy access to the database design |
    *
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the (file manager) FM privilege
    * to execute this function unless the \p accessMode is set to MODE_READ.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 611 | Database {name} not found. | The specified database did not exist. This is expected when creating a new database. |
    * | 15234 | Missing database name. |  |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 3042 | {Name} is not a database. |  |
    * | 2530 | No access to database {name}. | The logged on user did not have access to the specified database. |
    * | 9538 | No write access to database {name}. | The logged on user did not have write access to the specified database. |
    * | 10594 | No delete rights to database {name}. | The logged on user did not have delete access to the specified database. |
    * | 2498 | You have no FM access to the database. |  |
    *
    * @see TdbPutBaseDesign()
    * @see TdbGetBaseProperty()
    * @see TdbCloseBaseDesign()
    */
   int TdbGetBaseDesign(TdbHandle* handle, const char* base, int accessMode);
   

   /**
    * @brief Close a database design handle
    * @param handle Pointer to a handle to database obtained via the
    *        TdbGetBaseDesign() function. The handle value will be set to
    *        NULL upon successful completion of this function.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * A database handle must be closed using this function when the application
    * not longer needs it. Failure to do so will result in memory leaks.
    *
    * @see TdbGetBaseDesign()
    */
   int TdbCloseBaseDesign(TdbHandle* handle);
   

   /**
    * @brief Retreieve database design properties
    * @param handle Handle to a database design obtained from TdbGetBaseDesign()
    * @param property_id The ID of the property to retrieve
    * @param numeric_value Receives property values returned as integers.
    * @param string_value_buffer Receives property values returned as strings.
    * @param string_buffer_size the allocated size of \p string_value_buffer.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function retrieves the value of the database design property
    * identified by the \p property_id parameter. Valid property IDs are:
    *
    * | Property ID | Returned As | Description |
    * | ----------- | ----------- | ------------|
	* | BASE_PROPERTY_NAME | String | The name of the database |
	* | BASE_PROPERTY_COMMENT | String | The description of the database |
	* | BASE_PROPERTY_LOCATION | String | The storage loction of the database. This is the configuration symbol (as entered into TDBS_CONF) or the directory path,  depending on which was entered into the database design. This property is only valid if all database files share the same storage location.  |
	* | BASE_PROPERTY_BAF_FILE | String | The BAF file path. Is either a fully qualified path or the filename prefixed by a configuration symbol indicating a storage location (e.g. MYLOCATION:MYBASE.BAF) |
	* | BASE_PROPERTY_BIF_FILE | String | The BIF file path. Is either a fully qualified path or the filename prefixed by a configuration symbol indicating a storage location (e.g. MYLOCATION:MYBASE.BIF) |
	* | BASE_PROPERTY_VIF_FILE | String | The VIF file path. Is either a fully qualified path or the filename prefixed by a configuration symbol indicating a storage location (e.g. MYLOCATION:MYBASE.VIF) |
	* | BASE_PROPERTY_LOG_FILE | String | The LOG file path. Is either a fully qualified path or the filename prefixed by a configuration symbol indicating a storage location (e.g. MYLOCATION:MYBASE.LOG) |
	* | BASE_PROPERTY_WORD_CHARS | String | A list of special searchable characters for the database |
	* | BASE_PROPERTY_BAFFIT_ASE1 | String | The name of an ASE to be called during TFORM load before a record is committed to the BAF. |
	* | BASE_PROPERTY_BAFFIT_ASE2 | String | The name of an ASE to be called during TFORM load after a record is committed to the BAF. |
	* | BASE_PROPERTY_CONTAINER | String | The name of the classification container used by the database. |
	* | BASE_PROPERTY_IS_XML | Integer | A True/False flag determining whether the database is XML enabled. |
	* | BASE_PROPERTY_IS_GRAPH | Integer | A True/False flag determining whether the database is graph enabled. |
   * | BASE_PROPERTY_IS_CONNECTOR | Integer | A True/False flag determining whether the database is a connector database. |
	* | BASE_PROPERTY_CHI_WORDS | Integer | Type type of Chinese word segmentation to use with this database. 0 = none, 1 = max, 2 = all, 3 = word. |
	* | BASE_PROPERTY_USE_AUTO_REORG | Integer | A True/False flag determining whether the database is allowed to perform automatic index reorganisation during the indexing process. |
	* | BASE_PROPERTY_USE_AUDIT | Integer | A True / False flag determining whether the database is to create an audit trail for each user of the database. Note that this functionality has been superceded by the DEBIT functions as documented in the TRIP Manager Guide. |
	* | BASE_PROPERTY_USE_LOG_DELETE | Integer | A True / False flag determining whether the database logfile will record the entire content of deleted records. |
	* | BASE_PROPERTY_PID_FIELD | Integer | The name of the record part name field. |
	* | BASE_PROPERTY_CHARSET | String | The name of the character set that the database uses. New databases should use UTF. For a complete list, consult the TRIP Manager Guide. |
	* | BASE_PROPERTY_LANGUAGE | String | The three-letter code for the natural language primarily used by the text and phrase values in the database. |
	* | BASE_PROPERTY_APPL_ID | String | An application specific id string, not used by TRIP itself. |
	* | BASE_PROPERTY_OUTPUT_FORMAT | String | The name of the default output format |
	* | BASE_PROPERTY_ENTRY_FORM | String | The name of the default TRIPclassic entry form |
	* | BASE_PROPERTY_RID_FIELD | String | The name of the record name field |
	* | BASE_PROPERTY_COUNTER_FIELD | String | The name of the record number field |
	* | BASE_PROPERTY_CHAR_FOLDING | String | The character folding class for the database. This is a three-letter code for a natural language supported by TRIP. |
   *
   * #### Since
   * This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseDesign()
	*/
   int TdbGetBaseProperty(TdbHandle handle, int property_id, int* numeric_value,
                          char* string_value_buffer, int string_buffer_size);
   
   /**
    * @brief Assign a database design property.
	* @param handle Handle to a database design obtained from TdbGetBaseDesign()
	* @param property_id The ID of the property to assign
	* @param numeric_value Numeric property value
	* @param string_value String property value
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
	*
	* #### Description
	* This function retrieves the value of the database design property
	* identified by the \p property_id parameter. Valid property IDs are:
	*
	* | Property ID | Value Type | Description |
	* | ----------- | ----------- | ------------|
	* | BASE_PROPERTY_NAME | String | The name of the database |
	* | BASE_PROPERTY_COMMENT | String | The description of the database |
	* | BASE_PROPERTY_LOCATION | String | The storage loction of the database. This is the configuration symbol (as entered into TDBS_CONF) or the directory path,  depending on which was entered into the database design. This property is only valid if all database files share the same storage location.  |
	* | BASE_PROPERTY_BAF_FILE | String | The BAF file path. Is either a fully qualified path or the filename prefixed by a configuration symbol indicating a storage location (e.g. MYLOCATION:MYBASE.BAF) |
	* | BASE_PROPERTY_BIF_FILE | String | The BIF file path. Is either a fully qualified path or the filename prefixed by a configuration symbol indicating a storage location (e.g. MYLOCATION:MYBASE.BIF) |
	* | BASE_PROPERTY_VIF_FILE | String | The VIF file path. Is either a fully qualified path or the filename prefixed by a configuration symbol indicating a storage location (e.g. MYLOCATION:MYBASE.VIF) |
	* | BASE_PROPERTY_LOG_FILE | String | The LOG file path. Is either a fully qualified path or the filename prefixed by a configuration symbol indicating a storage location (e.g. MYLOCATION:MYBASE.LOG) |
	* | BASE_PROPERTY_WORD_CHARS | String | A list of special searchable characters for the database |
	* | BASE_PROPERTY_BAFFIT_ASE1 | String | The name of an ASE to be called during TFORM load before a record is committed to the BAF. |
	* | BASE_PROPERTY_BAFFIT_ASE2 | String | The name of an ASE to be called during TFORM load after a record is committed to the BAF. |
	* | BASE_PROPERTY_CONTAINER | String | The name of the classification container used by the database. |
	* | BASE_PROPERTY_IS_XML | Integer | A True/False flag determining whether the database is XML enabled. |
	* | BASE_PROPERTY_IS_GRAPH | Integer | A True/False flag determining whether the database is graph enabled. |
   * | BASE_PROPERTY_IS_CONNECTOR | Integer | A True/False flag determining whether the database is a connector database. |
   * | BASE_PROPERTY_CHI_WORDS | Integer | Type type of Chinese word segmentation to use with this database. 0 = none, 1 = max, 2 = all, 3 = word. |
   * | BASE_PROPERTY_USE_AUTO_REORG | Integer | A True/False flag determining whether the database is allowed to perform automatic index reorganisation during the indexing process. |
	* | BASE_PROPERTY_USE_AUDIT | Integer | A True / False flag determining whether the database is to create an audit trail for each user of the database. Note that this functionality has been superceded by the DEBIT functions as documented in the TRIP Manager Guide. |
	* | BASE_PROPERTY_USE_LOG_DELETE | Integer | A True / False flag determining whether the database logfile will record the entire content of deleted records. |
	* | BASE_PROPERTY_PID_FIELD | Integer | The name of the record part name field. |
	* | BASE_PROPERTY_CHARSET | String | The name of the character set that the database uses. New databases should use UTF. For a complete list, consult the TRIP Manager Guide. |
	* | BASE_PROPERTY_LANGUAGE | String | The three-letter code for the natural language primarily used by the text and phrase values in the database. |
	* | BASE_PROPERTY_APPL_ID | String | An application specific id string, not used by TRIP itself. |
	* | BASE_PROPERTY_OUTPUT_FORMAT | String | The name of the default output format |
	* | BASE_PROPERTY_ENTRY_FORM | String | The name of the default TRIPclassic entry form |
	* | BASE_PROPERTY_RID_FIELD | String | The name of the record name field | 
	* | BASE_PROPERTY_COUNTER_FIELD | String | The name of the record number field |
	* | BASE_PROPERTY_CHAR_FOLDING | String | The character folding class for the database. This is a three-letter code for a natural language supported by TRIP. |
	*
	* The field language (BASE_PROPERTY_LANGUAGE) and character folding 
	* (BASE_PROPERTY_CHAR_FOLDING) properties have the following valid
	* values:
	*
	* | Language Code | Language Name | Language Effects | Character Folding Effects |
	* | ------------- | ------------- | ---------------- | ------------------------- |
	* | ENG | English | English stemmer may be used | Diacritic characters such as &Auml; will be indexed and searched as their non-diacritic form. |
	* | FIN | Finnish | Finnish stemmer may be used | Diacritic characters other than &auml; and &ouml; will be indexed and searched as their non-diacritic form. |
	* | GER | German | German stemmer may be used | Diacritic characters other than &uuml;, &ouml;, &auml; and &szlig; will be indexed and searched as their non-diacritic form. |
	* | NOR | Norwegian | Norwegian stemmer may be used | Diacritic characters other than &AElig;, &uuml; and &oslash; will be indexed and searched as their non-diacritic form. |
	* | SWE | Swedish | Swedish stemmer may be used | Diacritic characters other than &aring;, &auml; and &ouml; will be indexed and searched as their non-diacritic form. |
	*
	* The character set (BASE_PROPERTY_CHARSET) property has the
	* following valid values.
	*
	* | Character Set | Description |
	* | ------------- | ----------- |
	* | UTF | Unicode UTF-8 |
	* | LA1 | Windows-1252 (a variant of ISO-8859-1) |
	* | LA2 | ISO-8859-2 |
	* | LA3 | ISO-8859-3 |
	* | GBK | Chinese national standard charatcter set
	*
	* The changes made to the database design using this function do not take
	* effect immediately. Call the TdbPutBaseDesign() function to commit the
	* design changes.
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseDesign()
	* @see TdbPutBaseDesign()
	*/
   int TdbPutBaseProperty(TdbHandle handle,int property_id, int numeric_value,
                          const char* string_value);
   
   
   
   /** @} */     // END OF DB DESIGN FUNCTIONS BLOCK
   
   /** @name Field Design Functions
    *  Functions for management of database fields.
    *  @{
    */
   
   
   /**
    * @brief Load a field specification from a database design
    * @param fieldHandle Pointer to a handle receieving the field design
    * @param baseHandle Handle to a database design
    * @param fieldName Name of field to retrieve or NULL to retrieve by ordinal
    * @param ordinal Ordinal number of field to retrieve or 0 (zero) to retrieve by name
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    *
    * This function loads the specification for a database field and returns
    * a handle to it in the \p fieldHandle parameter. The field can either be
    * loaded via its name or by its ordinal number. The ordinal number for a
    * field is not the same as the field number, and always goes from 1 to the
    * total number of fields.
    *
    * A call to the function TdbGetBaseDesign() has to have preceded a call to
    * this function.
    *
    * #### Parameter Values
    * The \p fieldHandle parameter is passed a pointer to a TdbHandle variable
    * that will receive a handle to the field on output. The handle must be
    * released by calling TdbCloseBaseField() when the application no longer
    * needs it.
    *
    * The \p baseHandle parameter is a handle to a database design opened
    * via the TdbGetBaseDesign() function.
    *
    * The \p fieldName parameter names the field to load, or NULL if the
    * field is to be loaded via its ordinal number.
    *
    * The \p ordinal parameter specifies the ordinal number of the field to
    * load, or 0 (zero) if the field is to be loaded via its name. The ordinal
    * number is always in the range 1 to the number of fields in the database,
    * and is not related to field number.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function if the database design handle has been opened
    * with any other access mode than MODE_READ.
    *
    * #### Since
    * This function was introduced in TRIP version 8.0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Could not get field specification. |
    *
    * @see TdbGetBaseDesign()
    * @see TdbGetBaseFieldByNumber()
    */
   int TdbGetBaseField(TdbHandle* fieldHandle, TdbHandle  baseHandle,
                       const char* fieldName, int ordinal);



   /**
    * @brief Load a field specification from a database design
    * @param fieldHandle Pointer to a handle receieving the field design
    * @param baseHandle Handle to a database design
    * @param fieldNumber The number of the field to retrieve
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    *
    * This function loads the specification for a database field and returns
    * a handle to it in the \p fieldHandle parameter. The field is loaded by
    * its assigned number.
    *
    * A call to the function TdbGetBaseDesign() has to have preceded a call to
    * this function.
    *
    * #### Parameter Values
    * The \p fieldHandle parameter is passed a pointer to a TdbHandle variable
    * that will receive a handle to the field on output. The handle must be
    * released by calling TdbCloseBaseField() when the application no longer
    * needs it.
    *
    * The \p baseHandle parameter is a handle to a database design opened
    * via the TdbGetBaseDesign() function.
    *
    * The \p fieldNumber parameter specifies the number of the field to load.
    * Note that this is not an ordinal number and that fields in a database 
    * may have gaps in their numbering.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function if the database design handle has been opened
    * with any other access mode than MODE_READ.
    *
    * #### Since
    * This function was introduced in TRIP version 8.0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Could not get field specification. |
    *
    * @see TdbGetBaseDesign()
    * @see TdbGetBaseField()
    */
   int TdbGetBaseFieldByNumber(TdbHandle* fieldHandle, TdbHandle  baseHandle,
      int fieldNumber);

   /**
    * @brief Delete a field from a database design
    * @param fieldHandle [IN/OUT] Handle to a field in a database design
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbDeleteFieldSpec() deletes a given field from a database
    * design in memory, which must have previously been loaded using
    * TdbGetBaseDesign(), and the field design must have been loaded by
    * calling the TdbGetBaseField() function. Upon a subsequent call to
    * TdbPutBaseDesign(), the field will be deleted from the data dictionary.
    * This routine can only be used successfully when the database is empty.
    *
    * #### Parameter Values
    * The \p fieldHandle parameter is passed a pointer to a \c TdbHandle 
    * variable that refers to the field design to delete. The handle variable
    * that the parameter points to will be set to NULL upon successful
    * completion of this function.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Since
    * This function was introduced in TRIP version 8.0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 7906 | Field name is missing. |  |
    * | 14978 | Thesaurus fields may not be deleted. | Predefined fields in a thesaurus may not be deleted. |
    * | 18146 | XML fields may not be deleted. | Predefined fields in an XML database may not be deleted. |
    * | 1698 | Non-existing field name: name |  |
    * | 7810 | Deletion of fields is legal only when the database is empty. |  |
    *
    * @see TdbGetBaseField()
    * @see TdbGetBaseDesign()
    * @see TdbPutBaseDesign()
    */
   int TdbDeleteBaseField(TdbHandle* fieldHandle);

   
   /**
    * @brief Store a field specification record
    * @param handle Field specification handle
    * @ingroup basemgmt
    * @return 1 or other odd value on success, even value on error
    *
    * #### Description
    * The function TdbPutBaseField() stores the field specification record
    * referred to by the provided \p handle into the internal database
    * definition buffer for later writing to the CONTROL file by a call to the
    * TdbPutBaseDesign() function. This routine can only be called after a
    * database design has been loaded by the TdbGetBaseDesign() and
    * TdbGettBaseField() function have been called.
    *
    * #### Parameter Values
    * The \p handle parameter is passed a handle to a field specification
    * record with updated details of a field in the relevant database.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Since
    * This function was introduced in TRIP version 8.0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 2146 | Non-existing field: field. |  |
    * | 6786 | Already existing field name. |  |
    * | 6850 | Invalid field type. |  |
    * | 7106 | Field name is already the unique record name field. | There can be only one record name field in a database. |
    * | 7266 | Record name field must be of type {PHR}. |  |
    * | 7426 | field is already copyright protected by field. |  |
    * | 7458 | Field name must be: letter followed by alphanums or underscores. |  |
    * | 7618 | Changing of field type is legal only when the database is empty. |  |
    * | 7650 | Removal of record id is legal only when the database is empty. |  |
    * | 7714 | Pattern restriction is only valid for field type {PHR}. |  |
    * | 7746 | File reference restriction is only valid for field type {PHR}. |  |
    * | 8226 | Field name name coincides with the database name. | A field can not have the same name as the database. |
    * | 8258 | Record id field must not be specified for non-empty databases. |  |
    * | 8482 | Field field is already the record number field. |  |
    * | 8514 | Record number field must be of type {INT}. |  |
    * | 8546 | Removal of rec no field is legal only when the database is empty. |  |
    * | 8578 | Record number field must not be specified for non-empty databases. |  |
    * | 9058 | Invalid interval. |  |
    * | 11810 | Unbalanced parenthesis in field comment. |  |
    * | 12674 | Changing part field status is allowed only for empty databases. |  |
    * | 14946 | The fieldtype of a thesaurus field may not be changed. |  |
    * | 15426 | Changing from/to one paragraph is only allowed for empty databases. |  |
    * | 15650 | Only the CTX field in a thesaurus can be the record name field. |  |
    * | 15810 | Field field is already the unique part name field. |  |
    * | 15842 | Part name field must be of type {PHR}. |  |
    * | 15874 | Removal of part name field is legal only when the database is empty. |  |
    * | 15906 | Part name field must not be specified for non-empty databases. |  |
    * | 15938 | Part name field must be specified as part field. |  |
    * | 17154 | A part record field cannot be made the record name field. |  |
    * | 17186 | A part record field cannot be made the record number field. |  |
    * | 17378 | Word indexing is only valid for {PHR} fields. |  |
    * | 18018 | The fieldtype of an XML field may not be changed. | The predefined fields of an XML database may not be altered. |
    *
    * @see TdbCloseBaseField()
    * @see TdbGetBaseField()
    * @see TdbPutFieldProperty()
    */
   int TdbPutBaseField(TdbHandle handle);
   
   
   /**
    * @brief Closes a field specification handle
    * @param handle Pointer to field handle returned from TdbGetBaseField()
    * @ingroup basemgmt
    * @return 1 or other odd value on success, even value on error
    *
    * #### Description
    * The function TdbCloseBaseField() must be called to close a handle to
    * a field specification when the application not longer needs it.
    *
    * #### Parameter Values
    * The \p handle parameter is passed a pointer to the field specification
    * handle to close. The handle variable that the parameter points to will
    * be set to NULL upon successful completion of this function.
    *
    * #### Since
    * This function was introduced in TRIP version 8.0.
    *
    * @see TdbGetBaseField()
    * @see TdbGetFieldProperty()
    * @see TdbPutBaseField()
    * @see TdbPutFieldProperty()
    */
   int TdbCloseBaseField(TdbHandle* handle);
   
   /**
    * @brief Retrieve a field design property.
    * @param handle Handle to a field obtained using TdbGetBaseField()
    * @param property_id ID of property to retrieve
    * @param numeric_value Receives integer property values
    * @param string_value_buffer Receives string property values
    * @param string_buffer_size Allocated size of \p string_value_buffer
    * @ingroup basemgmt
    * @return 1 or other odd value on success, even value on error
    *
    * #### Description
    * Retrieves a property of a field design. The field design is identified
    * by the \p handle parameter and the type of property to retreieve is
    * identified by the \p property_id parameter.
    *
    * Valid values for \p property_id are:
    *
    * | Property ID | Type | Description |
    * | ----------- | ---- | ----------- |
    * | FIELD_PROPERTY_NAME | String | Field name |
    * | FIELD_PROPERTY_NUMBER | Integer | Field number |
    * | FIELD_PROPERTY_TYPE_NAME | String | Field type name |
    * | FIELD_PROPERTY_TYPE_NUMBER | Integer | Field type number |
    * | FIELD_PROPERTY_TYPE | String + Integer | Field type and number |
    * | FIELD_PROPERTY_COPYRIGHT_FIELD | String | Copyright field name |
    * | FIELD_PROPERTY_INDEX_MODE | Integer | Field indexing mode |
    * | FIELD_PROPERTY_IS_ORIG | Integer | Indicates if the TEXT or PHRASE field stores values in their exact, original form, or if processing to identify sentences and paragraphs is done. |
    * | FIELD_PROPERTY_IS_RID_FIELD | Integer | Indicates if the PHRASE field is a record name field |
    * | FIELD_PROPERTY_HAS_PATTERN | Integer | Indicates if the field has a value restriction pattern |
    * | FIELD_PROPERTY_HAS_FILE_REF | Integer | Indicates if the field's values are restricted by a field in another database |
    * | FIELD_PROPERTY_IS_PID_FIELD | Integer | Indicates if the field is a part name field |
    * | FIELD_PROPERTY_HAS_DEFAULT | Integer | Indicates if the field has a default value |
    * | FIELD_PROPERTY_IS_UNIQUE | Integer | Indicates if the field value must be unique |
    * | FIELD_PROPERTY_HAS_CONCORDANCE | Integer | Indicates if the field is used with the built-in text classifiction |
    * | FIELD_PROPERTY_IS_COUNTER_FIELD | Integer | Indicates if the field is a record number field |
    * | FIELD_PROPERTY_IS_PART_FIELD | Integer | Indicates if the field is a part record field |
    * | FIELD_PROPERTY_COST | Integer | Access cost of the field |
    * | FIELD_PROPERTY_MIN_ITEMS | Integer | Minimum number of subfields |
    * | FIELD_PROPERTY_MAX_ITEMS | Integer | Maximum number of subfields, or zero if there is no upper limit. |
    * | FIELD_PROPERTY_COMMENT | String | Description of the fielf |
    * | FIELD_PROPERTY_RESTRICTIONS | String | Field value restriction definition |
    * | FIELD_PROPERTY_LOAD_ASE | String | Name of ASE routine to call during TFORM load |
    * | FIELD_PROPERTY_INDEX_ASE | String | Name of ASE routine to call during indexing |
    *
    * The field type number property (FIELD_PROPERTY_TYPE_NUMBER) can return
    * the following values:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | FIELD_TEXT | 1 | TExt type field |
    * | FIELD_PHRASE | 3 | PHRase type field |
    * | FIELD_INTEGER | 9 | INteger type field |
    * | FIELD_NUMBER | 10 | NUMber type field |
    * | FIELD_DATE | 11 | DAte type field |
    * | FIELD_TIME | 12 | TIme type field |
    * | FIELD_STRING | 14 | STring type field |
    *
    * The field index mode property (FIELD_PROPERTY_INDEX_MODE) is a bit mask
    * that can have a combination of the following values:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | FIELD_INDEX_NONE | 0 | Not indexed |
    * | FIELD_INDEX_NORMAL | 1 | Indexed normally |
    * | FIELD_INDEX_WORD | 5 | Word-indexed |
    * | FIELD_INDEX_SEPARATE | 17 | Separately indexed |
    * | FIELD_INDEX_CLASS | 33 | Field is used in classification |
    *
    * #### Since
    * This function was introduced in TRIP version 8.0.
    *
    * @see TdbGetBaseField()
    */
   int TdbGetFieldProperty(TdbHandle handle, int property_id,
                           int* numeric_value, char* string_value_buffer,
                           int string_buffer_size);
   
   /**
	* @brief Assign a field design property.
	* @param field_design_handle Handle to a field obtained using TdbGetBaseField()
	* @param property_id ID of property to assign
	* @param numeric_value Integer property value
	* @param string_value String property value
	* @ingroup basemgmt
	* @return 1 or other odd value on success, even value on error
	*
	* #### Description
	* Assigns a property to a field design. The field design is identified
	* by the \p handle parameter and the type of property to retreieve is
	* identified by the \p property_id parameter.
	*
	* Valid values for \p property_id are:
	*
	* | Property ID | Type | Description |
	* | ----------- | ---- | ----------- |
	* | FIELD_PROPERTY_NAME | String | Field name |
	* | FIELD_PROPERTY_NUMBER | Integer | Field number |
	* | FIELD_PROPERTY_TYPE_NAME | String | Field type name |
	* | FIELD_PROPERTY_TYPE_NUMBER | Integer | Field type number |
	* | FIELD_PROPERTY_TYPE | String + Integer | Field type and number |
	* | FIELD_PROPERTY_COPYRIGHT_FIELD | String | Copyright field name |
	* | FIELD_PROPERTY_INDEX_MODE | Integer | Field indexing mode |
	* | FIELD_PROPERTY_IS_ORIG | Integer | Indicates if the TEXT or PHRASE field stores values in their exact, original form, or if processing to identify sentences and paragraphs is done. |
	* | FIELD_PROPERTY_IS_RID_FIELD | Integer | Indicates if the PHRASE field is a record name field |
	* | FIELD_PROPERTY_HAS_PATTERN | Integer | Indicates if the field has a value restriction pattern |
	* | FIELD_PROPERTY_HAS_FILE_REF | Integer | Indicates if the field's values are restricted by a field in another database |
	* | FIELD_PROPERTY_IS_PID_FIELD | Integer | Indicates if the field is a part name field |
	* | FIELD_PROPERTY_HAS_DEFAULT | Integer | Indicates if the field has a default value |
	* | FIELD_PROPERTY_IS_UNIQUE | Integer | Indicates if the field value must be unique |
	* | FIELD_PROPERTY_HAS_CONCORDANCE | Integer | Indicates if the field is used with the built-in text classifiction |
	* | FIELD_PROPERTY_IS_COUNTER_FIELD | Integer | Indicates if the field is a record number field |
	* | FIELD_PROPERTY_IS_PART_FIELD | Integer | Indicates if the field is a part record field |
	* | FIELD_PROPERTY_COST | Integer | Access cost of the field |
	* | FIELD_PROPERTY_MIN_ITEMS | Integer | Minimum number of subfields |
	* | FIELD_PROPERTY_MAX_ITEMS | Integer | Maximum number of subfields, or zero if there is no upper limit. |
	* | FIELD_PROPERTY_COMMENT | String | Description of the fielf |
	* | FIELD_PROPERTY_RESTRICTIONS | String | Field value restriction definition |
	* | FIELD_PROPERTY_LOAD_ASE | String | Name of ASE routine to call during TFORM load |
	* | FIELD_PROPERTY_INDEX_ASE | String | Name of ASE routine to call during indexing |
	*
	* The field type number property (FIELD_PROPERTY_TYPE_NUMBER) can return
	* the following values:
	*
	* | Symbolic Name | Constant Value | Purpose |
	* | ------------- | -------------- | ------- |
	* | FIELD_TEXT | 1 | TExt type field |
	* | FIELD_PHRASE | 3 | PHRase type field |
	* | FIELD_INTEGER | 9 | INteger type field |
	* | FIELD_NUMBER | 10 | NUMber type field |
	* | FIELD_DATE | 11 | DAte type field |
	* | FIELD_TIME | 12 | TIme type field |
	* | FIELD_STRING | 14 | STring type field |
	*
	* The field index mode property (FIELD_PROPERTY_INDEX_MODE) is a bit mask
	* that can have a combination of the following values:
	*
	* | Symbolic Name | Constant Value | Purpose |
	* | ------------- | -------------- | ------- |
	* | FIELD_INDEX_NONE | 0 | Not indexed |
	* | FIELD_INDEX_NORMAL | 1 | Indexed normally |
	* | FIELD_INDEX_WORD | 5 | Word-indexed |
	* | FIELD_INDEX_SEPARATE | 17 | Separately indexed |
	* | FIELD_INDEX_CLASS | 33 | Field is used in classification |
	*
	* Note that some of the properties (such as field type) cannot be
	* altered if the database has contents.
	*
	* The changes made to the field design using this function do not take
	* effect immediately. Call TdbPutBaseField() store the design changes
	* to the in-memory field design object, and TdbPutBaseDesign() to commit 
	* the database design.
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseField()
	* @see TdbPutBaseField()
	* @see TdbPutBaseDesign()
	*/
   int
   TdbPutFieldProperty(TdbHandle field_design_handle,int property_id,
                       int numeric_value, const char* string_value);

   

   /** @} */     // END OF DB FIELD FUNCTIONS BLOCK
   
   /** @name Field Group Design Functions
    *  Functions for management of database field groups.
    *  @{
    */
   
   /**
    * @brief Load a field group definition from a database design 
	* @param fgrp_design_handle [OUT] Receives a handle to the field group
	* @param base_design_handle [IN] Handle to a database design
	* @param name [IN] Name of field group to retrieve (or NULL to retrieve by ordinal)
	* @param ordinal [IN] Ordinal number of field group to retrieve (or zero to retrieve by name)
    * @ingroup basemgmt
	*
	* #### Description
	* Loads a field group definition from a database design and makes it
	* accessible via a handle (the \p fgrp_design_handle parameter). If 
	* the named field group does not exist and the database design is opened
	* in write mode, a new field group will be initialized. 
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbPutBaseFieldGroup()
	* @see TdbGetFieldGroupProperty()
	* @see TdbPutFieldGroupProperty()
	* @see TdbFieldGroupAddField()
	*/
   int
   TdbGetBaseFieldGroup(TdbHandle* fgrp_design_handle,
                        TdbHandle base_design_handle,
                        const char* name, int ordinal);
   
   /**
	* @brief Store a field group definition
	* @param handle Handle to a field group obtained from TdbGetBaseFieldGroup()
	* @return 1 or other odd value on success, even value on error
	* @ingroup obsoleted
	*
	* #### Description
	* The function TdbPutBaseFieldGroup() stores the field group definition in
	* the internal database definition buffer for later writing to the CONTROL
	* file by a call to the TdbPutBaseDesign() function. This routine can only
	* be called after a database definition has been loaded by the 
	* TdbGetBaseDesign() function, and the field group intiialized or fetched 
	* by the TdbGetBaseFieldGroup() function.
	*
	* To commit the field group definition to CONTROL, along with other
	* modifications made to the database design, call TdbPutBaseDesign().
	*
	* #### Privileges Required
	* The current TRIPsystem user must possess the file manager (FM) privilege
	* to execute this function.    
	*
	* @see TdbGetBaseFieldGroup()
	* @see TdbGetBaseDesign()
	* @see TdbPutBaseDesign()
	*/
   int TdbPutBaseFieldGroup(TdbHandle handle);
   
   
   /**
    * @brief Close a database field group
	* @param handle [IN/OUT] Pointer to field group handle returned from TdbGetBaseFieldGroup()
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
	*
	* #### Description
	* The function TdbCloseBaseFieldGroup() must be called to close 
	* a handle to a field specification when the application not longer
	* needs it.
	*
	* #### Parameter Values
	* The \p handle parameter is passed a pointer to the field group
	* handle to close. The handle variable that the parameter points to will
	* be set to NULL upon successful completion of this function.
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseFieldGroup()
	*/
   int TdbCloseBaseFieldGroup(TdbHandle* handle);
   
   /**
	* @brief Delete a database field group
	* @param handle [IN/OUT] Pointer to field group handle for the field group to delete
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
	*
	* #### Description
	* The function TdbDeleteBaseFieldGroup() removes a field group from
	* its database design. The field group must have been loaded by a
	* call to the TdbGetBaseFieldGroup() function and the database design
	* must have been opened in write mode.
	*
	* To commit the removal of the field group to CONTROL, along with other
	* modifications made to the database design, call TdbPutBaseDesign().
	*
	* #### Parameter Values
	* The \p handle parameter is passed a pointer to the field group
	* handle to delete. The handle variable that the parameter points 
	* to will be set to NULL upon successful completion of this function.
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseFieldGroup()
	* @see TdbPutBaseDesign()
	*/
   int TdbDeleteBaseFieldGroup(TdbHandle* handle);
   
   /**
    * @brief Retrieve field group properties
	* @param handle Field group handle
	* @param property_id  ID of the field group property to retrieve
	* @param numeric_value Receives property values returned as integers.
	* @param string_value_buffer Receives property values returned as strings.
	* @param string_buffer_size the allocated size of \p string_value_buffer.
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
	*
	* #### Description
	* This function retrieves the value of the field group design property
	* identified by the \p property_id parameter. Valid property IDs are:
	*
	* | Property ID | Returned As | Description |
	* | ----------- | ----------- | ------------|
	* | FGROUP_PROPERTY_NAME | String | The name of the field group |
	* | FGROUP_PROPERTY_NUMBER | Integer | Numeric ID of the field group |
	* | FGROUP_PROPERTY_TYPE | Integer | The type of the field group (FGROUP_TYPE_*) |
	* | FGROUP_PROPERTY_TYPENAME | String | The type of the field group as a string |
	* | FGROUP_PROPERTY_COMMENT | String | Description of the field group |
	* | FGROUP_PROPERTY_SIZE | Integer | Number of fields in the field group |
	* | FGROUP_PROPERTY_FIELDS | String or Integer | If \p numeric_value is provided and \p string_value_buffer is NULL, \p numeric_value will receive the size of the string buffer that is required. If \p string_value_buffer is not NULL, it will receive a comma separated list of the names of the fields in the group. |
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseFieldGroup()
	*/
   int
   TdbGetFieldGroupProperty(TdbHandle handle, int property_id,
                            int* numeric_value,
                            char* string_value_buffer, int string_buffer_size);
   
   /**
	* @brief Assign field group properties
	* @param handle Field group handle
	* @param property_id  ID of the field group property to assign
	* @param numeric_value Numeric property value
	* @param string_value String property value
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
	*
	* #### Description
	* This function assigns the value of the field group design property
	* identified by the \p property_id parameter. Valid property IDs are:
	*
	* | Property ID | Returned As | Description |
	* | ----------- | ----------- | ------------|
	* | FGROUP_PROPERTY_NAME | String | The name of the field group |
	* | FGROUP_PROPERTY_TYPE | Integer | The type of the field group (FGROUP_TYPE_*) |
	* | FGROUP_PROPERTY_TYPENAME | String | The type of the field group as a string |
	* | FGROUP_PROPERTY_COMMENT | String | Description of the field group |
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseFieldGroup()
	*/
   int
   TdbPutFieldGroupProperty(TdbHandle handle, int property_id,
                            int numeric_value, const char* string_value);
   
   /**
    * @brief Add a field to a field group
	* @param handle Field group handle
	* @param field_name Name of field to add
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
	*
	* #### Description
	* Add a field to a field group from the database that the field group
	* is associated with. The database design must have been opened in 
	* write mode. 
	*
	* A field group with part record fields can only contain part record
	* fields. You cannot mix head and part record fields in the same group.
	*
	* To save the changes to the field group, call TdbPutBaseFieldGroup() when
	* all fields have been added. To commit the database design with the new
	* or updated field group to CONTROL, call TdbPutBaseDesign() afterwards.
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseFieldGroup()
	* @see TdbPutBaseFieldGroup()
	* @see TdbGetBaseDesign()
	* @see TdbPutBaseDesign()
	*/
   int
   TdbFieldGroupAddField(TdbHandle handle, const char* field_name);
   
   /**
    * @brief Remove a field from a field group
	* @param handle Field group handle
	* @param field_name Name of field to remove
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
	*
	* #### Description
	* Remove a field from a field group. The database design must have 
	* been opened in write mode.
	*
	* To save the changes to the field group, call TdbPutBaseFieldGroup(). 
	* To commit the database design with the updated field group to CONTROL,
	* call TdbPutBaseDesign() afterwards.
	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseFieldGroup()
	* @see TdbPutBaseFieldGroup()
	* @see TdbGetBaseDesign()
	* @see TdbPutBaseDesign()
	*/   
   int TdbFieldGroupDelField(TdbHandle handle, const char* field_name);
   
   /**
    * @brief Retrieve a field from a field group 
	* @param field_group_handle [IN] Handle to a field group
	* @param field_handle [OUT] Receives a handle to the requested field design
	* @param field_name [IN] Name of field to retrieve, or NULL to retrieve by ordinal
	* @param ordinal [IN] Ordinal number of field to retrieve, or 0 (zero) to retrieve by name
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
	*
	* #### Description
	* Retrieve the design object for a field in a field group. The database
	* design must have been opened in at least read mode. The \p ordinal
	* parameter, if assigned, has a value unique to the field group and goes
	* from 1 to the total number of fields in the group.
	* 
	* The properties of the returned field can be inspected and modified using
	* the TdbGetFieldProperty() and TdbPutFieldProperty() methods. To modify the
	* field design, the database design must have been opened in write mode.
 	*
	* #### Since
	* This function was introduced in TRIP version 8.0.
	*
	* @see TdbGetBaseFieldGroup()
	* @see TdbGetBaseDesign()
	*/
   int
   TdbFieldGroupGetField(TdbHandle field_group_handle,
                         TdbHandle* field_handle,
                         const char* field_name,
                         int ordinal);
   
   /** @} */     // END OF DB FIELD GROUP FUNCTIONS BLOCK


   /**
    * @name Refererential Integrity and Defaults
    * @brief Functions for management of database referential integrity
    *        constraints and default values.
    *  @{
    */

   /**
    * @brief Retrieve the integrity rules for the current database.
    * @param rules Receives a pointer to an integrity rules structure
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    *
    * The database design must have been opened in read or write mode
    * using either TdbGetBaseDesign() or TdbGetBaseDef().
    *
    * Rules will either define an outgoing or incoming foreign key
    * specification. The function allocates memory as required, and
    * returns a pointer to that memory in the \p rules argument.
    *
    * This memory cannot be released by the calling application as it
    * is kernel memory, and must instead be released by calling
    * TdbFreeIntegrityRules().
    *
    * @see TdbGetBaseDesign()
    * @see TdbFreeIntegrityRules()
   */
   int TdbGetIntegrityRules(base_integrity_spec** rules);
   
   /**
    * @brief Release an integrity rules structure.
    * @param rules Pointer to the rules instance to release
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    *
    * This function must be called to release the memory and resources
    * associated with the data returned by TdbGetIntegrityRules().
    *
    * @see TdbGetIntegrityRules()
    */
   int TdbFreeIntegrityRules(base_integrity_spec* rules);
   
   /**
    * @brief Delete all integrity rules for the current database.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    *
    * Each rule is followed to its associated other database and
    * corresponding rule in that database is also deleted.
    *
    * The database design must have been opened in write mode using either
    * TdbGetBaseDesign() or TdbGetBaseDef().
    *
    * @see TdbGetIntegrityRules()
    */
   int TdbDeleteIntegrityRules();
   
   /**
    * @brief Validates that the defined foreign key, when set, will succeed.
    * @param keyfield Foreign key field in current database
    * @param parentdb Name of associated database
    * @param parentfld Field in associated database
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    */
   int TdbCheckForeignKey(Char* keyfield, Char* parentdb, Char* parentfld);
   
   /**
    * @brief Establishes a new foreign key relationship.
	* @param keyfield Phrase field to use as foreign key in current (dependent) database
	* @param parentdb Name of database the foreign key will refer to
	* @param parentfld Phrase field in \p parentdb that the foreign key will refer to
	* @param upd Integrity constraint for update operations in the parent database
	* @param del Integrity constraint for delete operations in the parent database
	* @return 1 or other odd value on success, even value on error
	* @ingroup basemgmt
    *
	* #### Description
    * Establishes a new foreign key relationship from a field in the child
    * database to a field in a parent database (i.e. this function must always
    * be called from the database holding the foreign key, not the database
    * holding the master data). This API validates the foreign key definition,
    * and then records an incoming link in the current database's CONTROL
    * record, and an outgoing link in the parent database's CONTROL record.
	*
	* The database design must have been opened in write mode using either
	* TdbGetBaseDesign() or TdbGetBaseDef().
	*
	* The integrity constraint values that may be used for the \p upd and
	* \p del parameters are:
	* 
	* | Constraint| Description |
	* | ----------| ----------- |
	* | INTEGRITY_RESTRICT | Block the operation if foreign key constrains would be violated |
	* | INTEGRITY_CASCADE | Cascade the update or delete operation to the dependent database |
	* | INTEGRITY_NOACTION | Allow the operation, even if foreign key constrains would be violated |
	* | INTEGRITY_SETNULL | Remove the value from the record(s) in the dependent database |
	* | INTEGRITY_SETDEFAULT | Assign the value from the record(s) in the dependent database to their default. Requires that the foreign key field has been declared with a default value. |
	*
	* @see TdbGetBaseDesign()
    */
   int TdbSetForeignKey(Char* keyfield, Char* parentdb, Char* parentfld, int upd, int del);
   
   /**
    * @brief Remove a foreign key from the current database.
    * @param keyfield Name of a foreign key field in the current database
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This will also follow the reference and remove it from the the associated
    * database design.
    *
    * The database design must have been opened in write mode using either
    * TdbGetBaseDesign() or TdbGetBaseDef().
    *
    * @see TdbGetBaseDesign()
    */
   int TdbDeleteForeignKey(Char* keyfield);
   
   /**
    * @brief Retrieve the default value for the specified field.
    * @param fieldName Name of field for which to retrieve the default value
    * @param fieldValue Buffer to receive the default value
    * @param fieldLength Allocated size of the fieldValue buffer
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    *
    * The database design must have been opened in read or write mode
    * using either TdbGetBaseDesign() or TdbGetBaseDef().
    *
    * If no default value is defined, the function returns 0, otherwise
    * it copies the default value to the provider \p fieldValue buffer.
    * If the buffer pointed to by \p fieldValue is of insufficient size
    * (as defined by \p fieldLength on input), the function will return
    * an error and \p fieldLength will contain the number of bytes required
    * in the buffer.
    *
    * @see TdbPutDefaultValue
    * @see TdbDeleteDefaultValue
    */
   int TdbGetDefaultValue(Char* fieldName, Char* fieldValue,
                          int* fieldLength);
   
   /**
    * @brief Set the default value for a field.
    * @param fieldName Name of field to assign a default value to
    * @param fieldValue Default value to assign
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    *
    * The database design must have been opened in write mode
    * using either TdbGetBaseDesign() or TdbGetBaseDef().
    *
    * If the value is invalid for the field defined, the function returns
    * an appropriate error code. Equally, if the field is not of a type
    * that can accept a default, the function returns an appropriate error
    * code.
    * 
    * @see TdbGetDefaultValue
    * @see TdbDeleteDefaultValue
    */
   int TdbPutDefaultValue(Char* fieldName, Char* fieldValue);
   
   /**
    * @brief Remove any existing default value for a field.
    * @param fieldName Name of field to remove the default value from
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    *
    * The database design must have been opened in write mode
    * using either TdbGetBaseDesign() or TdbGetBaseDef().
    *
    * @see TdbGetDefaultValue
    * @see TdbPutDefaultValue
    */
   int TdbDeleteDefaultValue(Char* fieldName);
   
   
   /** @} */     // END OF REFERERENTIAL INTEGRITY FUNCTIONS BLOCK

   
   /** @name Thesaurus Design Functions
    *  Functions for management of thesaurus designs.
    *  @{
    */

   /**
    * @brief Create a copy of a thesaurus definition
    * @param from_thes Name of source thesaurus
    * @param to_thes Name of new thesaurus
    * @param design Thesaurus specification record for new thesaurus.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbCopyThesDef() starts the copying of a thesaurus to
    * different one of a new name. The calling process must subsequently call
    * TdbPutThesDef() to commit the new thesaurus to the CONTROL file.
    *
    * #### Parameter Values
    * The \p old_thes parameter is passed a character string as the name of the
    * thesaurus design to be copied.
    *
    * The \p new_thes parameter is passed a character string as the name of the
    * thesaurus to be created.
    *
    * The \p design parameter is passed a pointer to a TdbHandle variable that
    * will receive the thesaurus specification for the newly created thesaurus.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * @see TdbCopyThesDesign()
    * @see TdbPutThesDef()
    */
   int TdbCopyThesDesign(const char* from_thes, const char* to_thes,
                         TdbHandle* design);

   
   /**
    * @brief Delete a thesaurus definition
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbDeleteThesDef() deletes the thesaurus definition
    * previously loaded via a call to the TdbCheckThes() function.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the (file manager) FM privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 34403 | Thesaurus design for name deleted. | Operation completed successfully. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 12450 | Predefined system thesaurus THES cannot be deleted. |  |
    * | 8386 | You have no FM access to the thesaurus name. | The current user need to be file manager in order to access this function. |
    * | 8322 | The thesaurus name is not empty. | Only empty thesauri may be deleted. |
    *
    * @see TdbGetThesDef()
    * @see TdbPutThesDef()
    */
   int TdbDeleteThesDef(void);

  /**
   * @brief Load a thesaurus design
   * @param handle Pointer to a TdbHandle variable that receives a handle
   *        to the thesaurus design.
   * @param name Name of thesaurus
   * @param accessMode Access rights required to the thesaurus design
   * @return 1 or other odd value on success, even value on error
   * @ingroup basemgmt
   *
   * #### Description
   * This function loads the definition of the specified TRIP thesaurus and
   * returns a \p handle to it. Use the TdbGetBaseProperty() function to
   * read the properties of the thesaurus design.
   *
   * The \p accessMode parameter specifies how the calling process intends to
   * use the database design. Valid values are:
   *
   * | Symbolic Name | Constant Value | Purpose |
   * | ------------- | -------------- | ------- |
   * | MODE_READ | bit 0 | Require read access to the thesaurus design |
   * | MODE_WRITE | bit 1 | Require modify access to the thesaurus design |
   * | MODE_DELETE | bit 2 | Require delete access to the thesaurus design |
   * | MODE_COPY | bit 3 and bit 1 | Require copy access to the thesaurus design |
   *
   *
   * #### Privileges Required
   * The current TRIPsystem user must possess the (file manager) FM privilege
   * to execute this function unless the \p accessMode is set to MODE_READ.
   *
   * #### Return Codes
   * | Code | Description | Explanation |
   * | ---- | ----------- | ----------- |
   * | 1 | Operation completed successfully | The function completed its operations without error. |
   * | 611 | Database {name} not found. | The specified database did not exist. |
   * | 15234 | Missing database name. |  |
   * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
   * | 3042 | {Name} is not a database. |  |
   * | 2530 | No access to database {name}. | The logged on user did not have access to the specified database. |
   * | 9538 | No write access to database {name}. | The logged on user did not have write access to the specified database. |
   * | 10594 | No delete rights to database {name}. | The logged on user did not have delete access to the specified database. |
   * | 2498 | You have no FM access to the database. |  |
   *
   * @see TdbPutThesDesign()
   * @see TdbGetBaseProperty()
   * @see TdbCloseBaseDesign()
   */
   int TdbGetThesDesign(TdbHandle* handle, const char* name, int accessMode);
   
   /**
    * @brief Store a thesaurus definition
    * @param handle Handle to the thesaurus design to store
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function the thesaurus definition assocated with the provided
    * \p handle into the CONTROL file. A call to this function must have
    * been preceded by a call to either of the TdbGetThesDesign() or
    * TdbCopyThesDesign() functions.
    *
    * #### Parameter Values
    * The \p handle parameter is passed handle to the database obtained via
    * either of the TdbGetThesDesign() or TdbCopyThesDesign() functions.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 34275 | Thesaurus design for {name} altered (Check logical names). | Operation completed successfully. |
    * | 34307 | Thesaurus design for {name} created (Check logical names). | Operation completed successfully. |
    * | 34339 | Thesaurus design for {name} altered. | Operation completed successfully. |
    * | 34371 | Thesaurus design for {name} created. | Operation completed successfully. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 11842 | Unbalanced parenthesis in database description. |  |
    * | 14786 | {id} is not a valid character folding class. |  |
    *
    * @see TdbGetThesDesign()
    */
   int TdbPutThesDesign(TdbHandle handle);
   
   /** @} */     // END OF THES DESIGN FUNCTIONS BLOCK

    
   /** @name Cluster Management Functions
    *  Functions for management of database clusters.
    *  @{
    */

   /**
    * @brief Delete a database cluster
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbDeleteDbCluster() deletes a database cluster from the
    * data dictionary. Any user access to that cluster is also removed. The
    * cluster to be deleted must have been previously loaded by a call to the
    * TdbGetClusterDesign() function.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 36163 | Database cluster name deleted. | Operation completed successfully. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 16770 | You are not the owner of this DB cluster. |  |
    *
    * @see TdbGetClusterDesign()
    * @see TdbPutClusterDesign()
    */
   int TdbDeleteDbCluster(void);
   

   /**
    * @brief Load a database cluster specification
    * @param cluster_handle Receives a handle to the cluster specification
    * @param cluster_name Name of cluster to retrieve
    * @param access_mode How to open the cluster specification
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbGetClusterDesign() loads and makes current the
    * specification of a stored database cluster.
    *
    * #### Parameter Values
    * The \p cluster_handle parameter is passed a TdbHandle variable that will
    * receieve a handle to the cluster specification on output. The handle must
    * be closed by calling TdbCloseClusterDesign() when the application is done
    * with it.
    *
    * The \p cluster_name parameter is passed a character string as the name
    * of the cluster whose specification record is to be retrieved.
    *
    * The \p access_mode parameter is passed an \c int as the access mode for
    * the cluster specification. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | MODE_READ | bit 0 | Require read access to the cluster |
    * | MODE_WRITE | bit 1 | Require modify access to the cluster |
    * | MODE_DELETE | bit 2 | Require delete access to the cluster |
    * | MODE_COPY | bit 3 and bit 1 | Require copy access to the cluster |
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function with an access mode other than \c MODE_READ.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 8802 | Only alphanumerics/underscores allowed in database/cluster names. | The specified cluster name contained invalid characters. |
    * | 17442 | name is not a database cluster. |  |
    * | 16770 | You are not the owner of this DB cluster. |  |
    *
    * @see TdbPutClusterDesign()
    * @see TdbCloseClusterDesign()
    */
   int TdbGetClusterDesign(TdbHandle* cluster_handle,
                           const char* cluster_name,
                           int access_mode);
   
   
   /**
    * @brief Store a database cluster specification
    * @param handle Handle to cluster obtained via TdbGetClusterDesign()
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * The function TdbPutClusterDesign() will store the specification of a
    * database cluster defined in the cluster specification design that the
    * provided \p handle refers to.
    *
    * The cluster specification record must previously have been obtained by a
    * call to the TdbGetClusterDesign() function.
    *
    * #### Parameter Values
    * The \p handle parameter is passed a handle to a cluster specification,
    * with changes to the named predefined cluster of databases.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 39235 | Database cluster design for {name} altered. | Operation completed successfully. |
    * | 39267 | Database cluster design for {name} created. | Operation completed successfully. |
    * | 9794 | Database name {name} is the same as the DB cluster name. |  |
    *
    * @see TdbGetClusterDesign()
    * @see TdbClusterGetMember()
    * @see TdbClusterAddMember()
    * @see TdbClusterDelMember()
    * @see TdbClusterDelMembers()
    * @see TdbClusterEnumMembers()
    * @see TdbGetClusterProperty()
    * @see TdbPutClusterProperty()
    */
   int TdbPutClusterDesign(TdbHandle handle);
   
   
   /**
    * @brief Close a cluster design handle
    * @param handle Pointer to a handle to database cluster obtained via the
    *        TdbGetClusterDesign() function. The handle value will be set to
    *        NULL upon successful completion of this function.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * A cluster handle must be closed using this function when the application
    * not longer needs it. Failure to do so will result in memory leaks.
    *
    * @see TdbGetClusterDesign()
    */
   int TdbCloseClusterDesign(TdbHandle* handle);

   
   /**
    * @brief Copy a database cluster
    * @param sourceHandle Handle to cluster to copy
    * @param newName Name of new cluster to create
    * @param targetHandle Receives a handle to the cluster copy
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function copies a database cluster. The new cluster must be
    * committed by passing the \p targetHandle to the TdbPutClusterDesign()
    * function.
    *
    * #### Parameter Values
    * The \p sourceHandle parameter is passed a handle to a cluster design
    * to be copied. This handle is obtained using TdbGetClusterDesign().
    *
    * The \p newName parameter identifies the name of the new cluster to
    * create.
    *
    * The \p targetHandle parameter is a pointer to a TdbHandle variable
    * that recieves a handle to the copy the the source cluster. The new
    * cluster must be committedto the CONTROL database by calling the
    * TdbPutClusterDesign() function with this handle as argument.
    *
    * @see TdbGetClusterDesign()
    * @see TdbPutClusterDesign()
    * @see TdbCloseClusterDesign()
    */
   int TdbCopyClusterDesign(TdbHandle sourceHandle, const char* newName,
                        TdbHandle* targetHandle);
   
   /**
    * @brief Add a member to a cluster
    * @param handle Handle to a database cluster design obtained using
    *        the TdbGetClusterDesign() function.
    * @param name Name of database or cluster to add as member.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function adds a member to a database cluster. The named member
    * be a database or another cluster. The change must be committed by
    * calling TdbPutClusterDesign().
    *
    * There is no set upper limit for how large a cluster can be, but in
    * order for clusters to remain possible to adninistrate with TRIPclassic,
    * clusters should not have more than 30 members.
    *
    * @see TdbGetClusterDesign()
    * @see TdbPutClusterDesign()
    * @see TdbClusterDelMember()
    * @see TdbClusterDelMember()
    */
   int TdbClusterAddMember(TdbHandle handle, const char* name);
   
   
   /**
    * @brief Delete a member from a cluster
    * @param handle Handle to a database cluster design obtained using
    *        the TdbGetClusterDesign() function.
    * @param position 1-based position of member to delete or 0 (zero) to
    *        delete a member by name.
    * @param name Name of member to delete, or NULL to delete a member by
    *        its position.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function deletes a specified member from a cluster. The member is
    * specified by either its \p name or its 1-based \p position in the cluster
    * design. The change must be committed by calling TdbPutClusterDesign().
    *
    * @see TdbGetClusterDesign()
    * @see TdbPutClusterDesign()
    * @see TdbClusterDelMember()
    * @see TdbClusterGetMember()
    * @see TdbClusterEnumMembers()
    */
   int TdbClusterDelMember(TdbHandle handle, int position, const char* name);

   
   /**
    * @brief Delete all members from a cluster
    * @param handle Handle to a database cluster design obtained using
    *        the TdbGetClusterDesign() function.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function deletes all members from a cluster, e.g. in preparation
    * for supplying a new list of cluster members. So instead of adding and
    * deleting individual members, one can first use this function to clear
    * the cluster design, then call TdbClusterAddMember() repeatedly to add
    * the new list of members. The change must be committed by calling
    * TdbPutClusterDesign().
    *
    * @see TdbGetClusterDesign()
    * @see TdbPutClusterDesign()
    * @see TdbClusterAddMember()
    */
   int TdbClusterDelMembers(TdbHandle handle);

   /**
    * @brief Iterator function to retrieve the names of all cluster members.
    * @param handle Handle to a database cluster design obtained using
    *        the TdbGetClusterDesign() function.
    * @param position [IN/OUT] Initialize to 1 to start retrieving. The value
    *        will be updated by the function as member names are returned.
    * @param name Character string buffer to receive a cluster member name
    * @param namesize The allocated size of the \p name buffer
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function is used to retrieve the names of the members of a cluster,
    * as in the following example:
    * 
    * \code{.cpp}
    char name[128];
    int position;
    TdbHandle handle;
    
    if (IsSuccess(TdbGetClusterDesign(&handle,"MYCLUSTER",MODE_READ))
    {
      position = 1;
      while (IsSuccess(TdbClusterEnumMembers(handle,&position,name,128)))
      {
         printf("%s\n",name);
      }
      TdbCloseClusterDesign(&handle);
    }
    * \endcode
    *
    * @see TdbGetClusterDesign()
    * @see TdbCloseClusterDesign()
    */
   int TdbClusterEnumMembers(TdbHandle handle, int* position,
                             char* name, int namesize);

   
   /**
    * @brief Retrieive the name of a cluster member.
    * @param handle Handle to a database cluster design obtained using
    *        the TdbGetClusterDesign() function.
    * @param position The 1-based position of the cluster member to retrieve.
    * @param name Character string buffer to receive a cluster member name
    * @param namesize The allocated size of the \p name buffer
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function is used to retrieve the name of the cluster member
    * identified by the \p position parameter. The upper range limit of
    * the \p position is the number of members in the cluster, and can be
    * obtained using the TdbGetClusterProperty() function with the
    * CLUSTER_PROPERTY_MEMBERCOUNT property ID.
    *
    * @see TdbGetClusterDesign()
    * @see TdbGetClusterProperty()
    */
   int TdbClusterGetMember(TdbHandle handle, int position,
                           char* name, int namesize);
   
   /**
    * @brief Retrieve a property of a database cluster
    * @param handle Handle to a database cluster design obtained using
    *        the TdbGetClusterDesign() function.
    * @param property_id ID of property to retrieve.
    * @param numeric_value Pointer to an integer that receives property values
    *        returned as integers.
    * @param string_value_buffer A character string buffer that receives
    *        property values returned as character string data.
    * @param string_buffer_size The allocated size of the \p string_value_buffer
    *        parameter.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function retrieves the value of the cluster design property
    * identified by the \p property_id parameter. Valid property IDs are:
    *
    * | Property ID | Returned As | Description |
    * | ----------- | ----------- | ------------|
    * | CLUSTER_PROPERTY_NAME | String | The name of the cluster |
    * | CLUSTER_PROPERTY_COMMENT | String | The description of the cluster |
    * | CLUSTER_PROPERTY_MEMBERCOUNT | Integer | The number of members in the cluster |
    *
    * @see TdbGetClusterDesign()
    */
   int TdbGetClusterProperty(TdbHandle handle, int property_id,
                             int* numeric_value, char* string_value_buffer,
                             int string_buffer_size);
   
   
   /**
    * @brief Assign a property of a database cluster
    * @param handle Handle to a database cluster design obtained using
    *        the TdbGetClusterDesign() function.
    * @param property_id ID of property to retrieve.
    * @param numeric_value Assign the value of an integer based property.
    * @param string_value Assign the value of a character string based
    *        property.
    * @return 1 or other odd value on success, even value on error
    * @ingroup basemgmt
    *
    * #### Description
    * This function is used to assign the value of the cluster design property
    * identified by the \p property_id parameter. Valid property IDs are:
    *
    * | Property ID | Value Type | Description |
    * | ----------- | ---------- | ------------|
    * | CLUSTER_PROPERTY_COMMENT | String | The description of the cluster |
    *
    * To commit the changes, call the TdbPutClusterDesign() function.
    *
    * @see TdbGetClusterDesign()
    * @see TdbPutClusterDesign()
    */
   int TdbPutClusterProperty(TdbHandle handle, int property_id,
                             int numeric_value, const char* string_value);
   
   /** @} */     // END OF CLUSTER FUNCTIONS BLOCK

   
   
   
   // ------------------------------------------------------------------------
   
   /**
    * @defgroup reports Report Management
    * @brief Management of reports / output formats
    *
    * The TRIP report management functions offer functionality for administration
    * (creation, alteration and deletetion) of reports (a.k.a. output formats).
    */

   /**
    * @brief Check the syntax of an output format
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbCheckOutputFormat() checks the syntax of an output format
    * previously loaded by a call to either the TdbGetOutputFormat() or
    * TdbCopyOutputFormat() functions.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 3394 | Format syntax error at or near line lineno. |  |
    * | 5218 | No initial left delimiter in format spec. | Premature end of format specification. |
    *
    * @see TdbCopyOutputFormat()
    * @see TdbGetOutputFormat()
    */
   int TdbCheckOutputFormat(void);
   
   
   /**
    * @brief Retrieve the name of the current defined default entry or output form
    * @param base Name of database
    * @param form Name of form
    * @param mode Mode or type of form.
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The TdbCurrentForm() returns the name of the current defined default
    * entry or output form, as defined by the calling process with the CCL
    * commands DEfine Format= {x}  and DEfine EFOrm= {x}  .
    *
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string as the name of the
    * database for which form information is required.
    *
    * The \p form parameter is passed a character string to receive a string
    * as the name of the form
    *
    * The \p mode parameter is passed an \c int that defines the type of the
    * current form to retrieve. Valid values are:
    *
    * | Symbolic Constant | Constant Value | Explanation |
    * | ----------------- | -------------- | ----------- |
    * | CURRENT_OUTPUT | 1 | Retrieve the current default output form |
    * | CURRENT_ENTRY | 2 | Retrieve the current default entry form |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 610 | Database {1} not found. |  |
    * | 1730 | Default data entry form for {1} not specified. |  |
    * | 2530 | No access to database {1}. |  |
    * | 3042 | {1} is not a database. |  |
    * | 15234 | Missing database name. |  |
    * | 21666 | No default output format defined for {1}. |  |
    */
   int TdbCurrentForm(char* base, char* form, int mode);

   
   /**
    * @brief Create a copy of an output format
    * @param old_base Name of source database
    * @param old_form Name of source output format
    * @param new_base Name of target database
    * @param new_form Name of new output format
    * @param error Error position
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbCopyOutputFormat() starts the copying of an output format
    * from one database to another or to a form with a different name within
    * the same database. The calling process must subsequently call the
    * functions TdbCheckOutputFormat() and TdbPutOutputFormat() to store the
    * new format in the CONTROL file.
    *
    * #### Parameter Values
    * The \p old_base parameter is passed a character string as the name of
    * the source database from which the existing form will be copied.
    *
    * The \p old_form parameter is passed a haracter string as the name of
    * the source output format to be copied.
    *
    * The \p new_base parameter is passed a character string as the name of
    * the target database design to which the new form will be copied.
    *
    * The \p new_form parameter is passed a character string as the name of
    * the new output format to be created by the copy operation.
    *
    * The \p error parameter is passed a pointer to an \c int variable that 
    * will receive the positional number of the argument that caused the failure.
    * Valid values are:
    *
    * | Error Value | Explanation |
    * | ----------- | ----------- |
    * | 0 | No errors detected |
    * | 1 | The source database does not exist |
    * | 2 | The source output format does not exist. Alternatively the user calling this function is not owner of the format. |
    * | 3 | The target database does not exist. Alternatively, the user calling this function has no write access to the database |
    * | 4 | The new output format cannot be created. The TRIPkernel detected a parsing error. |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 34051 | Output format copied. | The function completed its operations without error. |
    * | 7970 | Output format {database.format} not found. | The specified output format did not exist in the source database. |
    * | 8066 | Already existing output format. |  |
    * | 11554 | Database name is missing. |  |
    * | 15010 | 1 is a database cluster. | The database name passed to the function refers to a database cluster and not a single database. |
    *
    * @see TdbPutOutputFormat()
    */
   int TdbCopyOutputFormat(char* old_base, char* old_form, char* new_base,
                           char* new_form, int* error);
   

   /**
    * @brief Delete an output format
    * @param base Database name
    * @param format Format name
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbDeleteOutputFormat() deletes an output format. A format
    * can only be deleted by the owner of the format or the owner of the
    * database to which the format belongs.
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string as the database name.
    *
    * The \p format parameter is passed a character string as the name of the
    * output format to be deleted.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 7522 | Database {2} is OS write protected. |  |
    * | 11170 | Cannot find COMMAND file {1}{2}{3} |  |
    * | 9090 | Database {2} is locked by another user, try again later. |  |
    * | 11554 | Database name is missing. |  |
    * | 11586 | Output format name is missing. |  |
    * | 14722 | You are not the creator of {1}{3}{2} |  |
    * | 15010 | {1} is a database cluster. | The database name passed to the function refers to a database cluster and not a single database. |
    * | 15618 | Only alphanumerics/underscores allowed in names. |  |
    * | 17474 | A TRIP runtime version cannot create databases. |  |
    *
    * @see TdbGetOutputFormat()
    */
   int TdbDeleteOutputFormat(char* base, char* format);

   
   /**
    * @brief Load the description of a CONTROL record
    * @param comment String buffer that receives the description of a CONTROL record.
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbGetComment() loads the description of a CONTROL record
    * into the buffer provided. The call has to be preceded by a call to one of
    *  the TdbCopyOutputFormat(), TdbCopyProcedure(), TdbGetOutputFormat() or
    * TdbGetProcedure() functions.
    *
    * #### Parameter Values
    * The \p comment parameter is passed a character string as buffer for
    * storing the description of a CONTROL record.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * @see TdbCopyOutputFormat()
    * @see TdbCopyProcedure()
    * @see TdbGetOutputFormat()
    * @see TdbGetProcedure()
    */
   int TdbGetComment(char* comment);

   
   /**
    * @brief Load a format line
    * @param line Format line
    * @param lineLength Length of format line
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbGetFormLine() loads a single line from a format into the
    * buffer provided. The format must have been previously loaded into the
    * internal form buffer by using the TdbGetOutputFormat() function. Repeated
    * calls to TdbGetFormLine() can be made until a status code of zero is
    * returned this allows processing of the complete form.
    *
    * #### Parameter Values
    * The \p line parameter is passed a character string to receive a line
    * from the selected output format specification.
    *
    * The \p lineLength parameter is passed a pointer to an \c int variable to 
    * receive the length of the line to be returned from the output format 
    * specification.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 | No more lines. |  |
    *
    * @see TdbGetOutputFormat()
    */
   int TdbGetFormLine(char* line, int* lineLength);
   
   
   /**
    * @brief Load an output format
    * @param base Database name
    * @param format Format name
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbGetOutputFormat() loads an output format for the
    * specified database into the internal format buffer for further
    * processing. The output format can only be loaded by the owner of the
    * format or the owner of the database to which the format belongs.
    *
    * This function is intended primarily for use within the TRIPclassic environment.
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string as the name of the
    * database for which an output format is to be retrieved.
    *
    * The \p format parameter is passed a character string as the name of the
    * format to be retrieved.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 7970 | Output format database.format not found. | The specified output format did not exist in the source database. |
    * | 11554 | Database name is missing. |  |
    * | 11586 | Output format name is missing. |  |
    * | 14722 | You are not the creator of {1}{3}{2} |  |
    * | 15618 | Only alphanumerics/underscores allowed in names. |  |
    */
   int TdbGetOutputFormat(char* base, char* format);

   
   /**
    * @brief Store the description of a CONTROL record
    * @param comment Description of CONTROL record
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbPutComment() stores a description of the CONTROL record
    * previously loaded into the internal buffer. The call has to be preceded
    * by a call to one of the TdbCopyOutputFormat(), TdbCopyProcedure(),
    * TdbGetOutputFormat() or TdbGetProcedure() functions.
    *
    * #### Parameter Values
    * The \p comment parameter is passed a character string as the description
    * of a CONTROL record.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbCopyOutputFormat()
    * @see TdbCopyProcedure()
    * @see TdbGetOutputFormat()
    * @see TdbGetProcedure()
    */
   int TdbPutComment(char* comment);

   
   /**
    * @brief Load a format line into an internal format buffer
    * @param line Line to load into format
    * @param lineLength Length of line
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbPutFormLine() loads a single line into the internal
    * output format buffer, which must have been previously loaded with an
    *  output format by a call to either of the TdbGetOutputFormat() or
    * TdbCopyOutputFormat() functions.
    *
    * The first call to TdbPutFormLine() will empty the format buffer and make
    * the supplied line the first line in the format, while subsequent calls
    * append lines to the format. The proper way to modify an existing output
    * format is therefore to first read all lines by repeatedly calling
    * TdbGetFormLine() until it returns an error code, then put each line back
    * (whether it was modified or not) by calling the TdbPutFormLine()
    * function. It is for the application program to provide intermediate
    * storage for the lines. Calls to TdbGetFormLine() and TdbPutFormLine()
    * should not be intermingled.
    *
    * The functions TdbCheckOutputFormat() and then TdbPutOutputFormat() must
    * be called if the format is to be stored.
    *
    * #### Parameter Values
    * The \p line parameter is passed a character string as the line to be
    * loaded into the format.
    *
    * The \p line length parameter is passed an \c int as the length of the
    * format line held in the \p line parameter.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbGetOutputFormat()
    */
   int TdbPutFormLine(char* line, int lineLength);

   
   /**
    * @brief Store an output format
    * @return 1 or other odd value on success, even value on error
    * @ingroup reports
    *
    * #### Description
    * The function TdbPutOutputFormat() writes the output format from an
    * internal format buffer into the CONTROL file. The internal buffer must
    * previously have been loaded by a call to either the TdbGetOutputFormat()
    * or TdbCopyOutputFormat() . Furthermore, before the format is committed
    * to CONTROL a syntax check must be performed by a call of the
    * TdbCheckOutputFormat() .
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 32643 | Format {1} altered for database {2}. | The function completed its operations without error. |
    * | 32675 | Format {1} created for database {2}. | The function completed its operations without error. |
    * | 11970 | Syntax check of {2}{3}{1} has not been performed |  |
    * | 14722 | You are not the creator of {1}{3}{2} |  |
    * | 17474 | A TRIP runtime version cannot create databases. |  |
    *
    * @see TdbCopyOutputFormat()
    * @see TdbGetOutputFormat()
    */
   int TdbPutOutputFormat(void);


   // ------------------------------------------------------------------------

   /**
    * @defgroup datamanipulation Data Manipulation
    * @brief Data manipulation functions
    *
    * The data manipulation functions cover all functionality for
    * manipulation of database contents.
    */
   

   /**
    * @brief Append a record part to the end of the part list for the currently loaded record
    * @param cursor The handle to a cursor object.
    * @param partName The name of the record part to be appended, or an empty string if the database has no part name field.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbAppendRecordPart() function adds a new part record to the record
    * identified through the \p cursor parameter. The new part is placed at
    * the end of the list of existing part records.
    *
    * The record being updated must have been loaded into memory using one of
    * the functions TdbGetRecordInBase() or TdbGetRecordInSearch(), which will
    * establish a record control.
    *
    * The cursor must have been attached to the record control pointer by
    * using TdbSetCursor() function prior to calling this function.
    *
    * The added record part is only committed to the database after a
    * successful call to the TdbPutRecord() function has been made using the
    * same record control that was associated with the \p cursor parameter.
    * Otherwise the part record is ignored.
    *
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a cursor reference. This must have
    * been created by a call to the TdbSetCursor() function. This parameter
    * is read-only.
    *
    * The \p partName parameter is passed s string of characters as the name
    * of the part record name for the newly created part record. If the database
    * has no part name field, the \p partName parameter should set to an empty
    * string.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 16034 | Already existing record part name: {1} |  |
    * | 17986 | Database {1} has no part name field. |  |
    * | 21698 | Argument {1} - value out of range. |  |
    *
    * @see TdbGetRecordInBase()
    * @see TdbGetRecordInSearch()
    * @see TdbSetRecordPart()
    * @see TdbSetCursor()
    * @see TdbPutRecord()
    */
   int TdbAppendRecordPart(TdbHandle cursor, const char* partName);
   

   /**
    * @brief Check the domain of a cursor object
    * @param cursor Cursor object identifier
    * @param recordControl Pointer handle that receives the record control
    * @param field Number of field addressed by cursor object
    * @param paragraph Receives the paragraph number
    * @param item Receives the item (sentence or subfield) number
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbCheckCursor() function decomposes a cursor object and returns
    * each component in the appropriate argument. In order these are the
    * record control object associated with the cursor, then the field,
    * paragraph and item set by a call to the TdbSetCursor() function.
    *
    * Note that the domain of the cursor object may reflect implicit updates
    * from use of the TdbGetLine() or TdbGetNextItem() functions.
    *
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a cursor object identifier, which was
    * created by a call to the TdbSetCursor() function.
    *
    * The \p recordControl parameter is passed a record control identifier.
    * The record control identifier returned is the one to which the cursor
    * was attached in a call to the TdbSetCursor() function.
    *
    * The \p field parameter is passed a pointer to an \c int variable to 
    * receive the number of the field that the cursor object is currently addressing.
    *
    * The \p paragraph paragraph is passed a pointer to an \c int variable to receive 
    * the number of the paragraph that the cursor object is currently addressing. 
    * Note that this parameter is only meaningful if the field in the cursor domain
    * is of a TExt type field.
    *
    * The \p item parameter is passed a pointer to an \c int variable to receieve the 
    * number of the item currently addressed by the cursor object. If the field in the
    * cursor domain is of field type TExt then the item corresponds to the
    * sentence number within the paragraph. If, however, the field in the
    * cursor domain is of any other field type then the item corresponds to
    * the subfield within the field.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 14658 | Undefined cursor. |  |
    *
    * @see TdbGetLine()
    * @see TdbSetCursor()
    */
   int TdbCheckCursor(TdbHandle cursor, TdbHandle* recordControl,
                      int* field, int* paragraph, int* item);
   

   /**
    * @brief Validate data for a particular field prior to attempting a commit operation
    * @param recordControl Record control object identifier.
    * @param field Field number
    * @param item Subfield number
    * @param data Data string to be validated
    * @param length Number of data bytes
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbCheckData() function validates supplied data against the database
    * design to ensure that the data can be committed at a later time.
    *
    * The checks this function makes are that the maximum number of subfields
    * is not exceeded, the minimum number of subfields is satisfied and that
    * the data value(s) are within any range limits or pattern/list
    * restrictions specified in the database design.
    *
    * Note that this validation is performed only for fields of PHRase, NUMber,
    * INteger, DAte or TIme type. No validation is performed for fields of
    * type \p TExt or \p STring.
    *
    * Any changes to a database design must be made with extreme care because
    * existing records could fail subsequent calls to this function if the
    * constraints are different.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a record control handle.
    * The record control object must have been previously attached to a 
    * database design by calling the TdbSetBase() function.
    *
    * The \p field parameter is passed an \c int value that is the number of
    * the field for which the given data is to be validated.
    *
    * The \p item parameter is passed an \c int as the number of item within
    * the field for which given data is to be validated. This ensures that
    * TRIPkernel can verify the maximum allowed items in a field has not been
    * exceeded.
    *
    * The \p data parameter is passed a character string as the data to be
    * validated.
    *
    * The \p length parameter is passed a pointer to an \c int variable to
    * receieve the number of bytes in the \p data parameter that are to be 
    * validated.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 2178 | Invalid NUM. |  |
    * | 2210 | Invalid DAT. |  |
    * | 21698 | Argument {1} - value out of range. |  |
    *
    * @see TdbCheckDate()
    * @see TdbValidValues()
    */
   int TdbCheckData(TdbHandle recordControl, int field, int item, char *data,
                    int length);

   
   /**
    * @brief Check for existence of indexed data in a PHRase field
    * @param recordControl Record control identifier
    * @param field Field number
    * @param data Data string
    * @param length Number of data bytes
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbCheckExistence() performs a search against the database
    * to which the record control object is attached, in the PHRase field
    * specified. If an exact match is found, i.e. the supplied data string
    * corresponds to a full subfield in the given field, the routine returns a
    * success code.
    *
    * The record control must have been attached to a database design using
    * the function TdbSetBase() prior to attempting this call.
    *
    * Note that in order to successfully locate the data in the field, the
    * database index must be up to date.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a record control handle.
    * This record control must have been associated with a database
    * design by calling the function TdbSetBase() prior to calling
    * TdbCheckExistence().
    *
    * The \p field parameter is passed an \c int as the field number of a
    * PHRase field where the data is located. Note that only PHRase fields are
    * permitted!
    *
    * The \p data parameter is passed a character string as the data whose
    * existence in the database is to be checked.
    *
    * The \p length parameter is passed an \c int as the number of bytes in
    * the \p data parameter to be checked.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 2114 | The DB-indices are being updated. Please, try later. |  |
    * | 2594 | Specified PHR does not exist in the database. |  |
    * | 8290 | OS open file limit of {1} reached. |  |
    * | 17218 | The {1} file of {2} is no {1} file. |  |
    *
    * @see TdbCheckData()
    * @see TdbSetBase()
    * @see TdbValidValues()
    */
   int TdbCheckExistence(TdbHandle recordControl, int field, char *data,
                         int length);
   

   /**
    * @brief Check a field for existence in a database design, and read its characteristics if present
    * @param recordControl Record control identifier
    * @param name Field name
    * @param number Field number
    * @param type Field type
    * @param flags Integer (bit vector)
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * Return type and index mode for a field, and return its name if indicated
    * by a positive field number argument, or its number if indicated by name
    * with zero as the number argument.
    *
    * The function TdbCheckField() will perform two functions based on the
    * arguments to the routine call.
    *
    * If the \p number parameter is set to zero, the field name is searched
    * for in the database design. If, however, the \p number parameter is
    * non-zero and the \p name parameter is blank, the field number is
    * searched for in the database design.
    *
    * If a field is found, by either method, then the type of the field and
    * its associated flags are returned in the \p type and \p flags parameters
    * respectively.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a record control handle.
    * This object must have been attached to a database design in a 
    * call to the function TdbSetBase() prior to calling TdbCheckField().
    *
    * The \p name parameter is passed a character string as the field name to
    * be checked. If The check is performed by number, this buffer must be 
    * at least 65 bytes long to allow for the maximum field name length
    * (64 bytes) plus the terminating null character.
    *
    * The \p number is passed a pointer to an \c int variable that must
    * be set to zero if the field characteristics are retrieved by \p name, 
    * or a field number if the field characteristics are to be retrived by 
    * number.
    *
    * The \p type parameter is passed a pointer to an \c int variable to receive 
    * the type of field found. Valid values are:
    *
    * | Symbolic Constant | Constant Value | Purpose |
    * | ----------------- | -------------- | ------- |
    * | FIELD_TEXT | 1 | Named field is of type TExt |
    * | FIELD_PHRASE | 3 | Named field is of type PHRase |
    * | FIELD_INTEGER | 9 | Named field is of type INteger |
    * | FIELD_NUMBER | 10 | Named field is of type NUMber |
    * | FIELD_DATE | 11 | Named field is of type DAte |
    * | FIELD_TIME | 12 | Named field is of type TIme |
    * | FIELD_STRING | 14 | Named field is of type STring |
    *
    * The \p flags parameter is passed a pointer to an \c int variable to receieve
    * a bit vector representing a number of flags concerning the processing options 
    * of the field. In the list of valid values, below, the name of server-side C macros 
    * are given that can be used by server-side TRIPtoolkit applications to query
    * the flags. Valid values are:
    *
    * | Bit number | C macro | Purpose |
    * | ---------- | ------- | ------- |
    * | 0 | IsIndexed(x) | If bit is set then the field is indexed |
    * | 1 | IsWriteable(x) | If set the calling process have write access to the field |
    * | 2 | IsPartField(x) | If set the field is a part field |
    * | 3 |  | Undefined. The setting of this bit cannot be guaranteed to be maintained between different version of TRIPsystem |
    * | 4 | IsMandatory(x) | If set the field is a mandatory field |
    * | 5 | IsLayoutRetained(x) | If set the field has the layout retained attribute enabled |
    * | 6 | IsReadable(x) | If set the calling process has read access to the field |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 2146 | Non-existing field: {1} |  |
    * | 14434 | Field no {1} has previously been deleted. |  |
    * | 14466 | Highest field number is {1}. |  |
    *
    * @see TdbGetFieldSpec()
    * @see TdbSetBase()
    */
   int TdbCheckField(TdbHandle recordControl, char* name, int* number,
                     int* type, int* flags);


   /**
    * @brief Retrieve the number or name of a record part
    * @param recordControl Record control identifier
    * @param partID Number of part to check, receives number when check is done by name
    * @param partName Name of part to check, receives name when check is done by number
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbCheckPartName() checks the currently loaded record for
    * the existence of a part record, identified by number or name.
    *
    * If the \p partID parameter is zero when calling the function the record
    * is searched for a part with the supplied part name and, if found, the
    * number of the part is returned in the \p partID parameter. If the
    * \p partID parameter is non-zero the name of the specified part is
    * returned in the \p partName parameter.
    *
    * The return code indicates failure in some cases, and so should be
    * checked against the special constants that are defined for this routine.
    *
    * The record for which the record parts are checked must have been
    * previously loaded into the record control area by a call to the
    * TdbGetRecordInBase() , TdbGetRecordInSearch() or
    * TdbGetRecordInSearchRis() functions.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a record control object. 
    * This object must have been attached to a database design by a call
    * to the TdbSetBase() function and a record loaded into this record
    * control before calling TdbCheckPartName() .
    *
    * The \p partID parameter is passed pointer to an integer that holds the
    * number of the part to check. This number can be zero if the check is 
    * to be made using the \p partName value.
    * Note that this number is not guaranteed to remain the same between
    * invocations of this routine as the list of record parts is always
    * renumbered because of the deletion or insertion of record parts. Programs
    * should not, therefore, rely on this number for their correct behavior.
    *
    * The \p partName parameter is passed a character string as the name of the
    * record part for which to get the part number. This name is taken to be the 
    * content of the record part name field specified in the database general 
    properties. The part name can be up to \c 255 characters long.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 16002 | Non-existing record part: {1} |  |
    * | 16034 | Already existing record part name: {1} |  |
    * | 17986 | Database {1} has no part name field. |  |
    * | 21698 | Argument {1} - value out of range. |  |
    *
    * @see TdbAppendRecordPart()
    * @see TdbDeleteRecordPart()
    * @see TdbInsertRecordPart()
    * @see TdbSetRecordPart()
    * @see TdbUndeleteRecordPart()
    */
   int TdbCheckPartName(TdbHandle recordControl, int* partID, char* partName);


   /**
    * @brief Retrieve the record number of a record indicated by name
    * @param recordControl Record control identifier
    * @param recordID Record identifier
    * @param name Name of the record
    * @return See the *Return Codes* table below.
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbCheckRecordName() searches the record name field of a
    * database for the specified value. The return code from this function is
    * always a failure code, thus there are special constants to check whether
    *  it has failed because the record exists or because it does not exist. In
    * the case where it does exist, the record ID will reflect the unique
    * number assigned to the record with that name.
    *
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a record control handle.
    * This object must have been associated with a database design by
    * calling the function TdbSetBase() before a call to
    * TdbCheckRecordName().
    *
    * The \p recordID parameter is passed a pointer to an \c int variable to receive 
    * the number of the record in the database.
    *
    * The \p name parameter is passed a character string as the name of the
    * record. This name is taken to be the content of the record name field
    * specified in the associated database's general properties. The record
    * name can be up to \c 255 characters long.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 130 | Record control structure not created. |  |
    * | 2882 | Non-existing record: {1} |  |
    * | 7074 | Already existing record name: {1} |  |
    * | 7394 | No record name field defined in {1}. |  |
    *
    * @see TdbGetRecordInBase()
    * @see TdbSetBase()
    */
   int TdbCheckRecordName(TdbHandle recordControl, int* recordID, char* name);


   /**
    * @brief Check a character string for conflict with a reserved word
    * @param symClass Symbol class
    * @param symbol Character string
    * @param length Length of symbol
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbCheckSymbol() directs a check against the TRIP data
    * dictionary for conflict between a supplied term and a reserved word,
    * such as the name of an open database or the name of a field in an open
    * database.
    *
    * If the symbol is found, the routine returns the symbol class and the
    * length in the \p symClass and \p length respectively. Also the content
    * of the \p symbol parameter are normalized to upper case and any
    * abbreviated symbol expanded.
    *
    * #### Parameter Values
    * The \p symClass parameter is passed a pointer to an \c int variable that
    * represents the symbol class. This variable should be specified on input 
    * as one of the following choices. If you specify a wildcard class, such as 
    * CHECK_ALL, on return from the routine the class argument will contain the 
    * exact class of the symbol if it wasfound. Valid values are:
    *
    * | Symbolic Constant | Constant Value | Symbol |
    * | ----------------- | -------------- | ------ |
    * | CHECK_BASENAME | 2 | Database name |
    * | CHECK_VIEWNAME | 3 | View name |
    * | CHECK_TEXT | 4 | TExt field |
    * | CHECK_PHRASE | 5 | PHRase field |
    * | CHECK_NUMBER | 6 | NUMber field |
    * | CHECK_DATE | 7 | DAte field |
    * | CHECK_TIME | 8 | TIme field |
    * | CHECK_ALL | 9 | All classes |
    * | CHECK_FIELDS | 10 | All field and view type classes |
    * | CHECK_MAPS | 14 | Indirect or mapped fields |
    *
    * The \p symbol parameter is passed a character string as the string to be
    * checked. On return from TdbCheckSymbol() the \p symbol parameter contains
    * the matched symbol, normalized and extended to its full length if an
    * abbreviated symbol was input.
    *
    * The \p length parameter is passed a pointer to an \c int variable that on
    * input must be set to the allocated size of the \p symbol buffer, allowing
    * for terminating null character. On output, the \p length parameter will
    * contain the number of bytes in the matched symbol name returned via the
    * \p symbol parameter.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Symbol not found |  |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbShellSymbol()
    */
   int TdbCheckSymbol(int* symClass, char* symbol, int* length);



   /**
    * @brief Create a record control object
    * @param recordControl Record control identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbCreateRecordControl() creates a record control object to
    * give access to a record within a database. In order to be used for such
    * a purpose, the calling process must follow one of two paths.
    *
    * If the database to which the object is to be attached is known, the
    * calling process should call the function TdbSetBase() to load the
    * database definition and to define the access level. Later records can be
    * read from the database using the TdbGetRecordInBase() function.
    *
    * If, instead, the database to which the object is to be attached can
    * change based on the result of a search, the calling process should call
    * TdbGetRecordInSearch() or TdbGetRecordInSearchRis() to retrieve the
    * record, and then if necessary call TdbSetBase() to elevate the access
    * level from read only.
    *
    * The only form of searching supported via the record control object is
    * record name lookup, using the TdbCheckRecordName().
    *
    * As the record control object can be large, it should be deleted when not
    * further required. For this, use the TdbDeleteRecordControl(). Because of
    * the size of this internal data structure it is unwise to call
    * TdbCreateRecordControl() within a loop. Use TdbEraseRecordControl() instead of
    * TdbDeleteRecordControl() if you wish to reuse an existing record control
    * object.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a pointer to a \c TdbHandle variable 
    * that will receive a handle to the created record control object.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbCheckRecordName()
    * @see TdbDeleteRecordControl()
    * @see TdbEraseRecordControl()
    * @see TdbGetRecordInBase()
    * @see TdbGetRecordInSearch()
    * @see TdbGetRecordInSearchRis()
    * @see TdbPutRecord()
    * @see TdbSetBase()
    * @see TdbSetCursor()
    * @see TdbSetRecordPart()
    */
   int TdbCreateRecordControl(TdbHandle* recordControl);


   /**
    * @brief Create a time stamp
    * @param tstamp Time stamp
    * @param type Time stamp style
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbCreateTimeStamp() creates a unsigned 32-bit integer representation that
    * can be used as a format independent comparison of times and dates.
    * The selection between a date and time combination or a simple time value
    * is decided by the value passed via the \p type parameter.
    *
    * #### Parameter Values
    * The \p tstamp parameter is passed a pointer to an unsigned 32-bit
    * integer variable that will receive the created time stamp.
    *
    * The \p type parameter is passed an \c int value defining the form of 
    * the tme stamp to return. Valid values are:
    *
    * | Symbolic Constant | Constant Value | Time Stamp Content |
    * | ----------------- | -------------- | ------------------ |
    * | DATE_AND_TIME | 0 | The time stamp is to include both date and time values. |
    * | DATE_ONLY | 1 | The time stamp in to include only a date value. |
    *
    * #### Return Codes
    * This function always succeeds
    */
   int TdbCreateTimeStamp(unsigned int* tstamp, int type);

   
   /**
    * @brief Retrieve information about the current context
    * @param base Name of database
    * @param recordControl Record control identifier
    * @param record Sequence number of record in database
    * @param part Part number
    * @param field Field number
    * @param paragraph Paragraph number
    * @param item Item number
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbCurrentItem() returns information concerning the calling
    * process' current context, such as the record control object currently in
    * use, the database currently open, the record currently loaded, etc. The
    * function can only be used in ASE routines. Note that the field, paragraph
    * and item information is not always valid, at which times the arguments
    * will be set zero upon return.
    *
    * Care must be taken in treatment of the record control object as this
    * object belongs to and is created by TRIP kernel itself and attempts to
    * delete it will cause significant problems for the execution of further
    * actions.
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string to receive the name
    * of the database.The buffer must be at least 65 bytes in length so that 
    * it can accommodate a database/cluster name of maximum size.
    *
    * The \p length parameter is passed a pointer to an \c int variable to 
    * receive the length of the string that was returned via the \p base parameter.
    *
    * The \p recordControl parameter is a pointer to a TdbHandle variable that
    * receives the handle to the record currently in use by the system. This
    * record control will have been created earlier by a call to the
    * TdbCreateRecordControl() function.
    *
    * The \p record parameter is a pointer to an \c int variable to receive the 
    * unique sequence number of the record within the database and which the 
    * record control is referring.
    *
    * The \p part parameter is passed a pointer to an \c int variable to receive the 
    * part number, if any, that is currently referenced by the record control identifier.
    *
    * The \p field parameter is passed a pointer to an \c int variable to receive the
    * field number, if any, that is currently referenced by the record control identifier.
    *
    * The \p paragraph is passed a pointer to an \c int variable to receive the paragraph 
    * number that is currently referenced by the record control identifier. This
    * parameter, although always provided with an call argument, is only
    * meaningful if the field currently in use is of type TExt.
    *
    * The \p item parameter is passed a pointer to an \c int variable to receive the item
    * number. If the field of the current item is a TExt field the item corresponds to
    * the sentence within the paragraph. If the field is of any type other than
    * TExt the item corresponds to the subfield within the field.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 3170 | Unexpected error. |  |
    */
   int TdbCurrentItem(char* base, TdbHandle* recordControl, int* record,
                      int* part, int* field, int* paragraph, int* item);

   
   /**
    * @brief Delete a cursor object
    * @param cursor Cursor identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbDeleteCursor() releases memory associated with the
    * specified cursor object. This action has no effect on the data in the
    * domain referenced by the cursor. Cursors are created automatically by
    * TdbSetCursor() and TdbSetRecordPart().
    * 
    * \note When processing large amounts of data, it is strongly recommended
    * to reuse existing cursor objects rather than deleteing and recreating
    * them. Reusing cursors will have a positive performance impact.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a pointer to a \c TdbHandle variable
    * that on input holds the cursor to delete. On successful output, the handle 
    * variable will be set to NULL.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Invalid cursor |  |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbSetCursor()
    * @see TdbSetRecordPart()
    */
   int TdbDeleteCursor(TdbHandle* cursor);
   
   
   /**
    * @brief Delete a data item in a cursor domain
    * @param cursor Record cursor
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbDeleteItem() deletes the specific item referenced by the
    * domain of the cursor, as previously established by calling the
    * TdbSetCursor() function.
    *
    * The domain may also cover a whole field or paragraph, in which case all
    * of the domain is deleted.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a pointer to the handle containing
    * the cursor to delete.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    *
    * @see TdbGetItem()
    * @see TdbInsertItem()
    * @see TdbReplaceItem()
    * @see TdbSetCursor()
    */
   int TdbDeleteItem(TdbHandle cursor);


   /**
    * @brief Delete a record control
    * @param recordControl Record control identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbDeleteRecordControl() deletes temporary work space that
    * was previously created via a call to TdbCreateRecordControl(). The memory
    * used by the work space is returned to the system. This routine also
    * deletes all of the cursors associated with the record control.
    *
    * The data structure behind a record control is large and will take 
    * considerable effort to recreate; as far as possible these structures
    * should be created and reused rather than being repeatedly created and
    * deleted. Use TdbEraseRecordControl() for this purpose.
    *
    * \note Care should be taken not to delete record control structures created as
    * internal work spaces by the TRIP kernel, such as the record control used
    * by the TdbCurrentItem() function.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a pointer to a \c TdbHandle variable
    * that on input holds the handle of the record control to delete. On successful
    * output, the parameter is set to NULL.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * @see TdbCreateRecordControl()
    * @see TdbEraseRecordControl()
    */
   int TdbDeleteRecordControl(TdbHandle* recordControl);

   
   /**
    * @brief Delete a record from a database
    * @param recordControl Record control
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbDeleteRecordInBase() deletes the record from the BAF
    * file. The record to be deleted must first be loaded into the specified
    * record control work area via a call to one of the functions
    * TdbGetRecordInBase() , TdbGetRecordInSearch() or
    * TdbGetRecordInSearchRis(). Each of these routines establishes a link
    * between a record control and a particular database.
    *
    * Use of this routine requires that the TRIPsystem user either be the owner
    * of the database or has unrestricted write access to that database. The
    * access mode for the database is set using the TdbSetBase() function.
    *
    * #### Parameter Values
    * The \p recordControl is passed a \c TdbHandle as a record control that is
    * associated with the record to delete.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 18338 | No database connection. |  |
    * | 14754 | Database {2} is open for read only. |  |
    * | 14818 | No write access to this record. |  |
    * | 33507 | Record {1} deleted. | The function completed its operations without error. |
    *
    * @see TdbSetBase()
    * @see TdbGetRecordInBase()
    * @see TdbGetRecordInSearchRis()
    */
   int TdbDeleteRecordInBase(TdbHandle recordControl);

   
   /**
    * @brief Delete a record part
    * @param cursor Cursor identifier
    * @param mode Deletion mode
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbDeleteRecordPart() is used to delete the record part to
    * which a cursor refers. If the mode is set to one, the record part itself
    * is not deleted but its name removed from the listing in the record head.
    * It is then possible to later restore it by a call to the
    * TdbUndeleteRecordPart() function within the same record.
    *
    * The cursor must have been attached to the record control area by using
    * TdbSetRecordPart() prior to attempting to call this function.
    *
    * You must call TdbPutRecord() to commit the changes.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a handle representing the cursor that
    * was set earlier by a call to the TdbSetCursor() function.
    *
    * The \p mode parameter is passes an \c int as the mode of deletion.
    * Valid values are:
    *
    * | Constant Value | Purpose |
    * | -------------- | ------- |
    * | 0 | Delete the part without the possibility of restoration. |
    * | 1 | Delete the part name from the record's part name listing. The part itself is not deleted until the record is stored by a call to PUT RECORD. |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    *
    * @see TdbSetRecordPart()
    * @see TdbUndeleteRecordPart()
    * @see TdbSetCursor()
    */
   int TdbDeleteRecordPart(TdbHandle cursor, int mode);

   
   /**
    * @brief Reinitialize a record control
    * @param recordControl Record control
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbEraseRecordControl() initializes an existing record
    * control work space, previously created with a call to the
    * TdbCreateRecordControl() function. The existing state of the record
    * control is cleared, resetting the control structure to indicate an empty
    * record. Any database associated with the record control structure is
    * unaffected by this operation.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a handle to a record control.
    * This record control will have been created earlier with a call to the
    * TdbCreateRecordControl() or some similar function.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbCreateRecordControl()
    * @see TdbDeleteRecordControl()
    */
   int TdbEraseRecordControl(TdbHandle recordControl);

   
   /**
    * @brief Export a STring field to a file
    * @param cursor Cursor identifier
    * @param filename Filename
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbExportBlob() writes the content of a STRING field to an
    * external file.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TdbHandle as the cursor that
    * identifies the field whose content is to be exported. This cursor must
    * have been defined earlier with a call to the TdbSetCursor() function.
    *
    * The \p filename parameter is passed a character string as the name of
    * the server-side file where information is to be written.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 10626 | Error opening output file {1}{2}{3} |  |
    * | 14658 | Undefined cursor. |  |
    * | 21634 | {1} is not a blob field. |  |
    *
    * @see TdbSetCursor()
    * @see TdbImportBlob()
    * @see TdbGetBlobBlock()
    */
   int TdbExportBlob(TdbHandle cursor, char* filename);


   /**
    * @brief Retrieve the content of a STRING field.
    * @param cursor Cursor pointing to a STRING field.
    * @param blockNr Block number to retrieve. Increase by 1 for each time around until the entire blob has been transferred.
    * @param blockSize The size of the block to retrieve.
    * @param block Output parameter receiving the requested blob block.
    * @param blockLen Number of bytes returned in the block output parameter.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbGetBlobBlock() function is used to retrieve the contents of a
    * STRING field.
    *
    * #### Parameter Values
    * The cursor in the \p cursor parameter must be set to refer to a STRING
    * field. No other field types are valid with this function. This is a
    * read-only value.
    *
    * The \p blockNr parameter must be set to 1 for the first block, then
    * incremented by one for each time this function is called until the entire
    * BLOB has been retrieved. This is a read-only value.
    *
    * The \p blockSize parameter is to be set to the size of the block to
    * retrieve. Note that this value should not be larger than 32768,
    * especially in the client-side version of this function. It is also
    * vitally important (for both versions) that this figure remains the same
    * for all calls to this function until the entire STRING value has been
    * retrieved. This is a read-only value.
    *
    * The \p block output parameter recieves a byte sequence with the requested
    * STRING value block.
    *
    * The \p blockLen output parameter recieves the byte count of the returned
    * \p block data.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Requested block does not exist. | This error occurs when the blockNr parameter is set too high. |
    * | 1 | Returned block is the last one. |  |
    * | 3 | Block returned ok - more blocks are available. | When this return code is recieved, the application is expected to call this function again with the blockNr parameter incremented by one. |
    * | 14658 | Invalid cursor. |  |
    * | 21634 | field is not a BLOB field. |  |
    *
    * @see TdbGetBlobSize()
    * @see TdbSetCursor()
    */
   int TdbGetBlobBlock(TdbHandle cursor, int blockNr, int blockSize,
                       char* block, int* blockLen);

   
   /**
    * @brief Returns the size of the content of a STRING field.
    * @param cursor Cursor pointing to the STRING field to return size for.
    * @param blobSize Size (byte count) of the field contents.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbGetBlobSize() function is used to retrieve the size of STRING
    * fields.
    *
    *
    * #### Parameter Values
    * The cursor in the \p cursor parameter must be set to refer to a STRING
    * field. No other field types are valid with this function. This is a
    * read-only value.
    *
    * The \p blobSize output parameter receives the size of the STRING field
    * contents.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 14658 | Invalid cursor. |  |
    * | 21634 | field is not a BLOB field. |  |
    *
    * @see TdbGetBlobBlock()
    * @see TdbSetCursor()
    */
   int TdbGetBlobSize(TdbHandle cursor, int* blobSize);


   /**
    * @brief Return Hit Part details
    * @param recordControl Record control
    * @param seqNumber 1-based sequence number of hit part
    * @param partNumber Part number
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetHitPart() retrieves a part record identified by its
    * ordinal number (using the \p seqNumber parameter) within the part hit vector
    * for a record in a search set. The actual part record number assoicated
    * with the input \p seqNumber is returned in the \p partNumber output 
    * parameter. The \p partNumber value can then be used to specify the 
    * part record by calling the TdbSetRecordPart() function.
    * 
    * For example, if a record has 24 part records and a search matches parts
    * 8 and 13, passing \p seqNumber 1 yields \p partNumber 8, and \p seqNumber 2
    * yields \p partNumber 13.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a handle to the record control
    * associated with the record to be retrieved.
    *
    * The \p seqNumber parameter is passed an \c int as the sequence
    * number of a hit part in the record. 
    *
    * The \p partNumber parameter is passed a pointer to a variable of type \c int
    * to receive the number of the part record.
    *
    * #### Since
    * This function was introduced in TRIP version 3.3-1.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | There are no more hit parts |  |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    *
    * @see TdbGetRecordInSearchRis()
    * @see TdbSetRecordPart()
    */
   int TdbGetHitPart(TdbHandle recordControl, int seqNumber, int* partNumber);
   

   /**
    * @brief Load a data item into a buffer
    * @param cursor Cursor identifier
    * @param data_item Data item string
    * @param item_length Length of data item
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetItem() copies the data in the sentence or subfield
    * specified by a cursor into the buffer provided, and must be preceded by
    * a call to the TdbSetCursor() function.
    *
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TdbHandle as a cursor identifier
    * created by a call to the TdbSetCursor() function.
    *
    * The \p data_item parameter is passed a character string to receive the
    * returned data item.
    *
    * The \p item_length parameter is passed a pointer to a 32-bit signed
    * integer to receive the count of data characters transferred to the data
    * item. The returned length excludes the terminating null-byte.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Operation failed | The sentence/subfield to which the cursor refers is empty |
    * | 1 | Operation completed successfully | The whole sentence/subfield was transferred |
    * | 3 | Operation completed successfully | Part of the sentence/subfield was transferred, but there is more to get |
    * | 12610 | The cursor does not specify a field number |  |
    *
    * @see TdbSetCursor()
    */
   int TdbGetItem(TdbHandle cursor, char* data_item, int* item_length);

   
   /**
    * @brief Load a text line into a buffer
    * @param cursor Record cursor
    * @param line Returned text line
    * @param line_length Length of string transferred
    * @param maximum_length Maximum number of bytes to be transferred
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetLine() loads a line from the field specified by the
    * cursor into the text buffer provided. Repeated calls to this function
    * loads the data in the field, one line at a time, into the buffer, until
    * the call fails with a return status not equal to one.
    *
    * Lines are divided by word boundaries, with the partitioning governed by
    * the \p maximum_length parameter.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TdbHandle as the record cursor
    * identifier, which was created in an earlier call to the TdbSetCursor()
    *  function.
    *
    * The \p line parameter is passed a character string to receive the
    * returned text line.
    *
    * The \p line_length parameter is passed a pointer to an \c int variable to 
    * receive the length of the string transferred into the text line buffer. For 
    * portable applications, the length does not include the terminating null-byte.
    *
    * The \p maximum_length parameter is passed an \c int as the maximum
    * number of bytes (null-byte excluded) to be transferred into the text
    * line buffer.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Operation failed | The sentence/subfield to which the cursor refers is empty |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 14658 | Undefined cursor. |  |
    * | 12610 | The cursor does not specify a field number |  |
    *
    * @see TdbSetCursor()
    */
   int TdbGetLine(TdbHandle cursor, char* line, int* line_length,
                  int maximum_length);

   
   /**
    * @brief Load the design of a link database
    * @param base Name of link database
    * @param recordNameField Name of record name field
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetLinkBase() loads the design of a link database if it
    * is available to the user (read access is enough). It is used within
    * TRIPclassic during the creation of entry form links to list the fields of
    * the link database.
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string as the name of the
    * link database.
    *
    * The \p recordNameField parameter is passed a character string as the name
    * of the record name field.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 610 | Database 1 not found. |  |
    * | 2530 | No access to database 1. |  |
    * | 3042 | 1 is not a database. |  |
    * | 8802 | Only alphanumerics/underscores allowed in database/cluster names. |  |
    * | 9890 | Link database must have a record name field. |  |
    * | 15234 | Missing database name. |  |
    */
   int TdbGetLinkBase(char* base, char* recordNameField);

   
   /**
    * @brief Load a record from a link database during data entry
    * @param recordControl Record control identifier
    * @param record_name Name of record
    * @param length Length of record name
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetLinkRecord() loads a record from a link database
    * during data entry. The record is specified by record name. The function
    * TdbSetLinkBase() must be called prior to this.
    *
    * This function is intended primarily for use within the TRIPclassic
    * environment.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a handle to a record control.
    *
    * The \p record_name parameter is passed a character string as the name of
    * the record to be retrieved.
    *
    * The \p length parameter is passed an \c int as the length of the record
    * name.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 1314 | Database {1} is OS read protected. |  |
    * | 8290 | OS open file limit of 1 reached. |  |
    * | 11426 | No such string in linked database 1. |  |
    * | 17218 | The {1} file of {2} is no {1} file. |  |
    *
    * @see TdbSetLinkBase()
    */
   int TdbGetLinkRecord(TdbHandle recordControl, char* record_name,
                        int length);


   /**
    * @brief Retrieve the next hit word from a record
    * @param recordControl Record control identifier
    * @param part Number of part containing next hit word
    * @param field Number of field containing next hit word.
    * @param paragraph Number of paragraph containing next hit word
    * @param item Number of subfield or sentence containing next hit word.
    * @param word_position Start position of next hit word within item
    * @param length Length of word.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetNextHitWord() returns the position of the next hit
    * word within the record. A call to either the TdbGetRecordInSearch() or
    * TdbGetRecordInSearchRis() functions must have been made to load the
    * record control area.
    *
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a \c TdbHandle as the record
    * control identifier.
    *
    * The \p part parameter is passed a pointer to an \c int variable to receive 
    * the number of the part that contains the next hit word. If the database design 
    * does not specify part records then the value returned is zero.
    *
    * The \p field parameter is passed a pointer to an \c int variable that will 
    * receive the number of the field containing the next hit word.
    *
    * The \p paragraph parameter is passed a pointer to an \c int variable that 
    * will receive the number of the paragraph containing the next hit word.
    *
    * The \p item parameter is passed a pointer to an \c int variable to receive 
    * the number of the item that contains the next hit word. If the field type is 
    * TExt then the \p item number is the number of the sentence in which the hit 
    * word is located. Otherwise, the \p item number is the number of the subfield
    * in which the hit word is located.
    *
    * The \p word_position parameter is passed a pointer to an \c int variable to 
    * receive the start position of the next hit word within the item. The position 
    * is the number of characters from the start of the item and is counted from zero.
    *
    * The \p length parameter is passed a pointer to an \c int variable to receive 
    * the length of the word.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Operation failed | There are no more hit words |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbGetRecordInSearch()
    * @see TdbGetRecordInSearchRis()
    */
   int TdbGetNextHitWord(TdbHandle recordControl, int* part, int* field,
                         int* paragraph, int* item, int* word_position,
                         int* length);
   
   
   /**
    * @brief Load the next data item into a buffer
    * @param cursor Cursor identifier
    * @param data_item Returned data item
    * @param length Length of string transferred
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetNextItem() copies the next item specified by the
    * cursor (set with a call to TdbSetCursor() ) into the \p data_item
    * parameter. Repeated calls to TdbGetNextItem() will transfer all of the
    * data from the field in the \p data_item parameter and adjusting the
    * \p length parameter as necessary. Calls can continue until there is a
    * return code of one, which indicates that all the data in the domain of
    * the cursor has been returned.
    *
    * If the cursor domain specifies a single subfield or sentence
    * TdbGetNextItem() will behave as though the domain started at the
    * subfield or sentence indicated, and continued to the end of the field.
    * That is by specifying a cursor domain of a single sentence or subfield,
    * the domain is implicitly set to cover all of the data within the field
    * that follows the sentence or subfield indicated in the cursor setting.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TdbHandle as the cursor identifier
    * set in a prior call to the TdbSetCursor() function.
    *
    * The \p data_item parameter is passed an application-allocated memory buffer
    * large enough to contain the largest item value to retrieve. 
    *
    * The \p length parameter is passed a signed 32-bit integer to receive the
    * length of the string transferred to the data item buffer. The returned
    * length excludes the terminating null-byte.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Operation failed | There is no more data in the field |
    * | 1 | Operation completed successfully | The final piece of data in the current sentence/subfield has been copied |
    * | 3 | Operation completed successfully | A piece of data has been copied, but there is more to get in the current sentence/subfield |
    *
    * @see TdbSetCursor()
    */
   int TdbGetNextItem(TdbHandle cursor, char* data_item, int* length);


   /**
    * @brief Get a record from a search set or a database
    * @param recordControl Record control to receive the loaded record
    * @param mode  Retrieval mode; one of the RECORD_FROM_* constant values
    * @param search Number of the search if mode indicates search set retrieval
    * @param ordinal Ordinal number of record in search result or the record number in the database
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetRecord() loads a record from a search set or directly
    * from a database into a work area for further processing. This work area must 
    * have been created by a call to the TdbCreateRecordControl() function. The \p ordinal
    * parameter must always be specified.
    *
    * @note This function replaces the functions TdbGetRecordInSearch(),
    * TdbGetRecordInSearchRis(), TdbGetRecordInBase() and TdbGetSortedRecord(), 
    * providing all their functionality in single interface, plus allows for
    * extremely large search sets with RIS numbers over 2bn.
    *
    * Retriving additional information on a record, such as record ID or the
    * database it belongs to, requires use of the TdbRecordInfo() function.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a \c TdbHandle as the TRIP record
    * control.
    *
    * The \p mode parameter indicates from which source to retrieve the record
    * and must be set to one of the following values:
    *
    * | Symbol | Description |
    * | ------ | ----------- |
    * | RECORD_FROM_BASE | Record is to be retrieved from the a database, like with TdbGetRecordInBase(). The \p ordinal parameter specifies a RID. The \p search parameter is ignored. |
    * | RECORD_FROM_SEARCH | Record is to be retrieved from a search set. The \p ordinal parameter specifies a RIS. |
    * | RECORD_FROM_SORTED_SEARCH | Record is to be retrieved from a sorted search set. The \p ordinal parameter specifies a RIS. |
    *
    * If \p mode set to \c RECORD_FROM_SEARCH or \c RECORD_FROM_SORTED_SEARCH,
    * the \p search parameter is passed an integer as the number of the search
    * set from which the record is to be loaded. If \p mode is set to
    * \c RECORD_FROM_BASE, this parameter is ignored.
    *
    * The \p ordinal parameter is passed an integer as the ordinal number of
    * the record in the search result if \p mode set to \c RECORD_FROM_SEARCH
    * or \c RECORD_FROM_SORTED_SEARCH. If \p mode is set to \c RECORD_FROM_BASE
    * this parameter specifies a record ID.
    *
    * #### Return Codes
    * The return code from the TdbGetRecord() function is a bit mask
    * that has the following structure. The return value is a combination of
    * one or more of these bits.
    *
    * | Bit # | Value | Explanation |
    * | ----- | ----- | ----------- |
    * | 0 | 1 | The operation completed successfully. |
    * | 1 | 2 | The first hit record in an open database has been retrieved. This is only useful when searching in clusters. |
    * | 2 | 4 | The record has been deleted from the database. |
    * | 3 | 8 | The record is locked by another process. |
    * | 4 | 16 | At end - no more records. |
    *
    * If an error occurs (i.e. the function returns an even value equal to zero
    * or greater than 32) the associated TRIP error code can be retrieved by
    * calling TdbRetCode().
    *
    * @see TdbCreateRecordControl()
    * @see TdbGetSortedRecord()
    */
   int TdbGetRecord(TdbHandle recordControl, int mode, int search,
                    int64_t ordinal);

   
   /**
    * @brief Gets the content of a buffer.
    * @param cursor Cursor identifier
    * @param buffer_address Address of buffer
    * @param buffer_length Size of the buffer in number of bytes
    * @param part_number Part record number
    * @param filled_length Number of bytes written to buffer
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetRecordBuffer() fills the provided buffer with
    * TFORM-formatted data from the record specified by the \p cursor. The
    * \p cursor, which is set by calling TdbSetCursor(), may specify the
    * entire record or a single field.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TdbHandle as the identifier of a
    * cursor previously created in a call to the TdbSetCursor() function.
    * 
    * The \p buffer_address parameter is passed the address of the memory
    * buffer that will receive the TFORM data.
    *
    * The \p buffer_length parameter is passed the size (number of bytes) of
    * the buffer.
    *
    * The \p filled_length parameter is passed a pointer to an \c int variable
    * to receive the number of bytes written to the buffer indicated by the
    * \p buffer_address parameter.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Operation failed | There is no data to fetch |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 3 | Operation completed successfully | A piece of data has been copied, but there is more to get in the current sentence/subfield |
    * |  |  |  |
    *
    * @see TdbSetCursor()
    * @see TdbPutRecordBuffer()
    */
   int TdbGetRecordBuffer(TdbHandle cursor, char* buffer_address,
                          int buffer_length, int part_number,
                          int* filled_length);


   /**
    * @brief Load a record from a database
    * @param recordControl Record control
    * @param RID Record number in database
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetRecordInBase() loads a single record from a database
    * into a work area for further processing. The work area (created by a
    * previous call to the function TdbCreateRecordControl() ) must have
    * already been labelled with the name of the database via a call to the
    * TdbSetBase() function.
    *
    * The particular record to be loaded into the work area is specified by
    * its RID, i.e. its internal record number provided and maintained
    * internally by the TRIPsystem.
    *
    * A RID value of zero will cause the record that follows the current one
    * to be loaded. On the first call, this will be the first record in the
    * database. Repeated calls to TdbGetRecordInBase() with the \p RID
    * parameter set to zero will consequently process all existing records
    * in the database until the end is reached and the function returns 16.
    * Similarly, giving the \p RID parameter a value of -1 will result in the
    * previous record being loaded.
    *
    * @note The records will be retrieved in the order that they were loaded into the
    * database. However, if records have been deleted from the database then
    * the RID numbers will not be consecutive.
    *
    * #### Parameter Values
    * The \p recordControl is passed a \c TdbHandle as the record control
    * associated with the record to be retrieved from the database.
    *
    * The \p RID parameter is passed an \c int as the number of the record
    * in a TRIP database.
    *
    * #### Return Codes
    * The return code from the TdbGetRecordInBase() function is a bit mask that
    * has the following structure. The return value is a combination of one or
    * more of these bits.
    *
    * | Bit # | Value | Explanation |
    * | ----- | ----- | ----------- |
    * | 0 | 1 | The operation completed successfully. |
    * | 1 | 2 | The first hit record in an open database has been retrieved. This is only useful when searching in clusters and does not apply to this function. |
    * | 2 | 4 | The record has been deleted from the database. |
    * | 3 | 8 | The record is locked by another process. |
    * | 4 | 16 | At end - no more records. |
    *
    * If an error occurs (i.e. the function returns an even value equal to zero
    * or greater than 32) the associated TRIP error code can be retrieved by
    * calling TdbRetCode().
    *
    * @see TdbCreateRecordControl()
    * @see TdbGetRecordInSearch()
    * @see TdbGetRecordInSearchRis()
    * @see TdbSetBase()
    */
   TDB_DEPRECATED("Deprecated by TdbGetRecord")
   int TdbGetRecordInBase(TdbHandle recordControl, int RID);


   /**
    * @brief Load a record from a search result
    * @param recordControl Record control identifier
    * @param search Search number
    * @param RID Record number in TRIP database
    * @param base Database name.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetRecordInSearch() loads a record from a search set
    * into a work area, created by a call to TdbCreateRecordControl(), for
    * further processing. On the first call the \p RID parameter should be set
    * to zero; on return from the routine the \p RID parameter indicates the
    * actual RID of the retrieved record. Subsequent calls to
    * TdbGetRecordInSearch() will cause all of the records within the search
    * set to be passed, one at a time, into the record control work area.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a \c TdbHandle as the record control
    * identifier associated with the record.
    *
    * The \p search parameter is passed a pointer to an \c int variable that
    * rerepresents the number of a search set. If this variable is set to zero
    * on input, the latest search set will be used and its actual number will 
    * be returned in this parameter on output. A non-zero value on input for
    * the variable that this parameter points to indicates a specific search
    * set from which to obtain a record.
    *
    * The \p RID parameter is passed an \c int as the number of the TRIP record 
    * to retrieve.
    *
    * The \p base parameter is passed a character string of at least 65 bytes 
    * in length to receive the name of the database that the retrieved records
    * are taken from. For a search of a single database this name will remain 
    * the same however for searches of multi-file views or predefined database
    * clusters this name will change to reflect the database that contains the 
    * record.
    *
    * #### Return Codes
    * The return code from the TdbGetRecordInSearch() function is a bit mask
    * that has the following structure. The return value is a combination of
    * one or more of these bits.
    *
    * | Bit # | Value | Explanation |
    * | ----- | ----- | ----------- |
    * | 0 | 1 | The operation completed successfully. |
    * | 1 | 2 | The first hit record in an open database has been retrieved. This is only useful when searching in clusters. |
    * | 2 | 4 | The record has been deleted from the database. |
    * | 3 | 8 | The record is locked by another process. |
    * | 4 | 16 | At end - no more records. |
    *
    * If an error occurs (i.e. the function returns an even value equal to zero
    * or greater than 32) the associated TRIP error code can be retrieved by
    * calling TdbRetCode().
    *
    * @see TdbCreateRecordControl()
    * @see TdbGetRecord()
    */
   TDB_DEPRECATED("Deprecated by TdbGetRecord")
   int TdbGetRecordInSearch(TdbHandle recordControl, int* search, int* RID,
                            char* base);

   
   /**
    * @brief Load a record from a search result, specified by RIS
    * @param recordControl TRIP record control.
    * @param search Search number
    * @param RIS Ordinal number of record in search result
    * @param RID TRIP record identifier
    * @param base Database name
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetRecordInSearchRis() loads a record from a search set
    * into a work area for further processing. This work area will have been
    * created by a call to the TdbCreateRecordControl() function. The \p RIS
    * parameter must always be specified.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a \c TdbHandle as the TRIP record
    * control.
    *
    * The \p search parameter is passed a pointer to an \c int variable that represents 
    * the number of the search set from which the record is to be loaded. If this variable 
    * is set to zero on input, the latest search set will be used and its number returned
    * in this parameter on output. If this variable is set to a non-zero value on input,
    * it indicates the a specific search set from which to obtain a record.
    *
    * The \p RIS parameter is passed an integer as the 1-based ordinal number of the
    * record in the search set.
    *
    * The \p RID parameter is passed a pointer to an \c int variable that will receieve
    * the TRIP record identifier number of as the retrieved record. This variable must
    * be set to zero before calling the routine.
    *
    * The \p base parameter is passed a character string to receive the name
    * of the database of the retrieved record. If the search was performed 
    * across a multi-file view or a predefined cluster then successive calls
    * to this function will update this value with the name of the database
    * the record has been read from. The memory area that the \p base
    * parameter refers to must be at least 65 bytes inlength.
    *
    * #### Return Codes
    * The return code from the TdbGetRecordInSearchRis() function is a bit mask
    * that has the following structure. The return value is a combination of
    * one or more of these bits.
    *
    * | Bit # | Value | Explanation |
    * | ----- | ----- | ----------- |
    * | 0 | 1 | The operation completed successfully. |
    * | 1 | 2 | The first hit record in an open database has been retrieved. This is only useful when searching in clusters. |
    * | 2 | 4 | The record has been deleted from the database. |
    * | 3 | 8 | The record is locked by another process. |
    * | 4 | 16 | At end - no more records. |
    *
    * If an error occurs (i.e. the function returns an even value equal to zero
    * or greater than 32) the associated TRIP error code can be retrieved by
    * calling TdbRetCode().
    *
    * @see TdbCreateRecordControl()
    * @see TdbGetSortedRecord()
    * @see TdbGetRecord
    */
   TDB_DEPRECATED("Deprecated by TdbGetRecord")
   int TdbGetRecordInSearchRis(TdbHandle recordControl, int* search, int RIS,
                               int* RID, char* base);
   
   
   /**
    * @brief Load a record from a thesaurus
    * @param recordControl Record control
    * @param termNumber Ordinal term number in display list
    * @param level Hierarchical level of record
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetRecordInThes() loads a specified record from the
    * current thesaurus display into a work area, created by a call to
    * TdbCreateRecordControl(). Each numbered term in the display list
    * corresponds to the controlled term ( CTX ) of a record in the thesaurus.
    * The hierarchical level of the term within the display list is returned
    * on completion of the loading operation.
    *
    * The records should be processed in order, starting with the record whose
    * controlled term is term number one in the list and then successively
    * incrementing the term number until all records have been retrieved and
    * the routine returns an error code. The term itself can be obtained by
    * calling TdbGetDisplayTerm() or by reading the CTX field of the thesaurus
    * record.
    *
    * A call to this routine must be preceded by a call to the function
    * TdbExecuteCcl() (for TRIPsystem versions 3.4 and later) or to the
    * functions TdbShellToDao() and TdbDaoOrder() (for TRIPsystem versions
    * earlier than 3.4) to execute a CCL order generating a display list of
    * the thesaurus.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a \c TdbHandle as the record control.
    *
    * The \p termNumber parameter is passed an \c int as the ordinal number
    * of the term in the (current) display list. The record loaded will be
    * that whose controlled term ( CTX ) is the term specified by the term
    * number.
    *
    * The \p level parameter is passed a pointer to an \c int variable to receive 
    * the hierarchical level of the record. Zero indicates that the record's controlled
    * term is a main term in the current list.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 1314 | Database {1} is OS read protected. |  |
    * | 12514 | CT={2}{3} in thesaurus record {1} causes a loop. |  |
    *
    * @see TdbCreateRecordControl()
    * @see TdbGetDisplayTerm()
    */
   int TdbGetRecordInThes(TdbHandle recordControl, int termNumber, int* level);

   
   /**
    * @brief Loads a record from a sorted search result.
    * @param search_no Search number
    * @param rec_ctl TRIP record control
    * @param ris Ordinal number of record in the sorted search result
    * @param rid TRIP record identifier
    * @param base Database name
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetSortedRecord() loads a record from a search set
    * sorted with the TdbSortRecords() function into a work area for further
    * processing. This work area must have been created by a call to the
    * TdbCreateRecordControl() function. The \p ris parameter must always be
    * specified.
    *
    * #### Parameter Values
    * The \p rec_ctl parameter is passed a handle to a TRIP record control.
    *
    * The \p search_no parameter is passed an \c int as the number of the
    * search from which the record is to be loaded. A search number of zero
    * indicates the current search, in which case the argument will be updated
    * to receive the number of the search on return from the routine.
    *
    * The \p ris parameter is passed an integer as the ordinal number of the
    * record in the search result.
    *
    * The \p rid parameter is passed a pointer to an \c int variable to receive 
    * the TRIP record identifier. The argument must be set to zero before calling 
    * the routine.
    *
    * The \p base parameter is passed a character string as the database name.
    * If the search was performed across a multi-file view or a predefined
    * cluster then successive calls to this function will update this argument
    * with the name of the database the record has been read from.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * The return code from the TdbGetSortedRecord() function is a bit mask that
    * has the following structure. The return value is a combination of one or
    * more of these bits.
    *
    * | Bit # | Value | Explanation |
    * | ----- | ----- | ----------- |
    * | 0 | 1 | The operation completed successfully. |
    * | 1 | 2 | The first hit record in an open database has been retrieved. This is only useful when searching in clusters. |
    * | 2 | 4 | The record has been deleted from the database. |
    * | 3 | 8 | The record is locked by another process. |
    * | 4 | 16 | At end - no more records. |
    * If an error occurs (i.e. the function returns an even value equal to
    * zero or greater than 32) the associated TRIP error code can be retrieved
    * by calling TdbRetCode().
    *
    * @see TdbCreateRecordControl()
    * @see TdbSortRecords()
    */
   TDB_DEPRECATED("Deprecated by TdbGetRecord")
   int TdbGetSortedRecord(int search_no, TdbHandle rec_ctl, int ris,
                          int* rid, char* base);
   

   /**
    * @brief Retrieve the TRIP time stamp of a record
    * @param recordControl Record control identifier
    * @param timestamp Time stamp value
    * @param type Time stamp format
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbGetTimeStamp() returns the time stamp of a record.
    * The record must have been previously loaded into a record control area
    * via a call to either TdbGetRecord(), TdbGetRecordInBase(), 
    * TdbGetRecordInSearch() or TdbGetRecordInSearchRis() .
    *
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a \c TdbHandle as the record control
    * identifying the record whose time stamp is to be returned.
    *
    * The \p timestamp parameter is passed a pointer to an unsigned 32-bit integer to
    * receive the record's time stamp value.
    *
    * The \p type parameter is passed an \c int as the formatting type of the time
    * stamp to be returned. The acceptable values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | DATE_AND_TIME | 0 | Return both Date and Time components of the time stamp |
    * | DATE_ONLY | 1 | Return only the Date component of the time stamp |
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbGetRecordInBase()
    * @see TdbGetRecordInSearch()
    * @see TdbGetRecordInSearchRis()
    */
   int TdbGetTimeStamp(TdbHandle recordControl, unsigned int* timestamp,
                       int type);
   
   
   /**
    * @brief Retrieve information about part records in a search
    * @param recordControl TRIP record control identifier
    * @param parthits Count of hits within parts
    * @param partcount Count of parts
    * @param partmax maximum part number
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbHitParts() returns the following information about part
    * records of a record retrieved from a search set using TdbGetRecord() or similar
    * function.
    *
    * #### Parameter Values
    * The \p parthits is passed a pointer to an \c int variable to receive the 
    * number of hit part records retrieved by the search.
    *
    * The \p partcount parameter is passed a pointer to an \c int variable to 
    * receive the count of the part records.
    *
    * The \p partmax is passed a pointer to an \c int variable to receive the 
    * highest part record number retrieved.
    *
    * #### Since
    * This function was introduced in TRIP version 3.3-1.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    *
    * @see TdbGetRecord()
    */
   int TdbHitParts(TdbHandle recordControl, int* parthits, int* partcount,
                   int* partmax);
   
   
   /**
    * @brief Import a binary large object from a file
    * @param cursor Cursor identifier
    * @param filename Filename to import
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbImportBlob() loads an object from the server-side file
    * specified via the \p filename parameter into the field referenced by the
    * \p cursor parameter.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a handle to a cursor, which has
    * previously been set (using the TdbSetCursor() function) to the
    * field to receive the data. Typically the field that \p cursor identifies
    * will be a STRING type field.
    *
    * The \p filename parameter is passed a character string as the name of
    * the server-side file from which data is to be imported. Syntax rules for
    * specifying the name are dependent upon the operating system running the
    * TRIP process. The rules may therefore be different is the call to this
    * function is made by a client running on a platform that adopts a
    * different format for filenames.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 834 | File {1}{2}{3} not found. |  |
    * | 21634 | {1} is not a blob field. |  |
    *
    * @see TdbExportBlob()
    * @see TdbPutBlobBlock()
    */
   int TdbImportBlob(TdbHandle cursor, const char* filename);

 
   /**
    * @brief Load a data item from a buffer into a record control area
    * @param cursor Cursor identifier
    * @param data_item Data item string to copied to record
    * @param item_length Number of bytes to transfer
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbInsertItem() inserts a value (a sentence, a phrase, a
    * number, a date or a time) into the record indicated by the cursor. The
    * cursor must have been previously positioned by a call to the
    * TdbSetCursor() function. The value is inserted before the sentence or
    * subfield indicated by the cursor, unless the cursor domain covers a
    * field or a paragraph. In this case, the value is appended to the end of
    * the domain.
    *
    * The cursor should be repositioned by a call to the function
    * TdbSetCursor() before attempting to call any function which references
    * the cursor domain, as the cursor domain becomes undefined after calling
    * TdbInsertItem().
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TdbHandle as the cursor identifier,
    * which has been set by a an earlier call to the TdbSetCursor() function.
    *
    * The \p data_item parameter is passed a character string as the value
    * copied to the record control area.
    *
    * The \p item_length parameter is passed an \c int as the number of bytes
    * in the \p data_item buffer to transferred.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 2178 | Invalid {NUM}. |  |
    * | 2242 | Invalid {TIM}. |  |
    * | 12482 | Maximum number of subfields for {1} is {3}. |  |
    * | 16706 | Invalid {INT}. |  |
    * | 18178 | Use TdbCheckPartName() for checking/storing part names. |  |
    *
    * @see TdbSetCursor()
    */
   int TdbInsertItem(TdbHandle cursor, char* data_item, int item_length);


   
   /**
    * @brief Insert a record part
    * @param cursor Cursor identifier
    * @param partname Name of record part to insert
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbInsertRecordPart() is used to insert a record part
    * before the part to which the cursor is currently referring. The cursor
    * must have been attached to the record control area by using the function
    * TdbSetRecordPart() prior to attempting to call this function.
    *
    * The function can only be used to update databases that include a part
    * record name field.
    *
    * The record must be saved with a call to the TdbPutRecord() function.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TdbHandle as the cursor identifier.
    *
    * The \p partname parameter is passed a character string as the name of the
    * record part to insert.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 16034 | Already existing record part name: {1} |  |
    * | 17986 | Database {1} has no part name field. |  |
    * | 21698 | Argument {1} - value out of range. |  |
    *
    * @see TdbSetRecordPart()
    * @see TdbPutRecord()
    */
   int TdbInsertRecordPart(TdbHandle cursor, const char* partname);


   /**
    * @brief Retrieve the number of items in a field
    * @param cursor Cursor identifier
    * @param count Count of items in domain
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbItemCount() returns information about the domain of the
    * cursor given in the \p cursor parameter. Depending on the domain covered
    * by the \p cursor parameter this function returns one of the following:
    *
    *   + The number of subfields for all non-TEXT fields
    *   + The number of sentences within a paragraph
    *   + The number of sentences within a field
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a handle to a cursor.
    *
    * The \p count parameter is passed an integer to receive the count of the
    * number of items in the domain of the cursor.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 8290 | OS open file limit of {1} reached. |  |
    * | 17218 | The {1} file of {2} is no {1} file. |  |
    *
    * @see TdbSetCursor()
    */
   int TdbItemCount(TdbHandle cursor, int* count);

   
   /**
    * @brief Lock the record in the record control area
    * @param recordControl Record control identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbLockRecord() locks the record that is currently in the
    * record control area, thereby granting the calling process exclusive write
    * access to it, until the record is released with a call to one of the
    * TdbReleaseRecord(), TdbPutRecord(), TdbDeleteRecordControl(), or
    * TdbEraseRecordControl() functions. Other users keep read access to the
    * record. 
    * 
    * A record is automatically locked by a call to the TdbGetRecord(), 
    * TdbGetRecordInBase(), TdbGetRecordInSearch(), or
    * TdbGetRecordInSearchRis() functions, if the preceding call to
    * TdbSetBase() specified write access.
    *
    * #### Parameter Values
    * The \p recordControl is passed a \c TdbHandle as the record control
    * identifier associated with the record to be locked into memory.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 16418 | TRIPdaemon failed to respond (1), notify your System Mgr. |  |
    * | 16450 | TRIPdaemon failure (1), notify your System Mgr. |  |
    * | 16482 | Your process is unknown to the TRIP daemon, notify your System Mgr. |  |
    * | 16898 | Concurrent user quota (1) exceeded. |  |
    * | 17570 | Invalid TRIPdaemon process name. |  |
    *
    * @see TdbDeleteRecordControl()
    * @see TdbEraseRecordControl()
    * @see TdbGetRecordInBase()
    * @see TdbGetRecordInSearch()
    * @see TdbGetRecordInSearchRis()
    * @see TdbGetRecord()
    * @see TdbPutRecord()
    * @see TdbReleaseRecord()
    * @see TdbSetBase()
    */
   int TdbLockRecord(TdbHandle recordControl);

   
   /**
    * @brief Load a CONTROL file record
    * @param recordControl Record control identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbNextControl() loads a CONTROL file record into a work
    * area specified by the \p recordControl parameter. This work area is
    * maintained by TRIPkernel and must not be deleted or used for any other
    * purpose. The record loaded must have been previously specified by a
    * call to the TdbShowControl() function.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a pointer to TdbHandle variable
    * that will receive a handle to a record control associated with the
    * CONTROL database.
    *
    * #### Privileges Required
    * The current TRIPsystem user needs no special privileges to execute this
    * function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Function failed | There are no more control items |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    *
    * @see TdbShowControl()
    */
   int TdbNextControl(TdbHandle* recordControl);

   
   /**
    * @brief Write data to a STRING field.
    * @param cursor Cursor pointing to a STRING field.
    * @param block Buffer with data to write.
    * @param blockLen Number of bytes in the block to write.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbPutBlobBlock() function is used to store the contents of a STRING
    * field. A block can be max 32KB, so for larger values, repated calls to
    * this routine is required.
    *
    * #### Parameter Values
    * The cursor in the \p cursor parameter must be set to refer to a STRING
    * field. No other field types are valid with this function. This is a
    * read-only value.
    *
    * The \p block is to be assigned a byte sequence to write to the STRING
    * field.
    *
    * The \p blockLen parameter is to be set to the number of bytes in the
    * block to write.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Requested block does not exist. | This error occurs when the blockNr parameter is set too high. |
    * | 1 | Returned block is the last one. |  |
    * | 3 | Block returned ok - more blocks are available. | When this return code is recieved, the application is expected to call this function again with the blockNr parameter incremented by one. |
    * | 14658 | Invalid cursor. |  |
    * | 21634 | field is not a BLOB field. |  |
    *
    * @see TdbGetBlobSize()
    * @see TdbGetBlobBlock()
    * @see TdbSetCursor()
    */
   int TdbPutBlobBlock(TdbHandle cursor, int blockLen, char* block);

   
   /**
    * @brief Load a text line from a buffer into a record control area.
    * @param cursor Cursor identifier
    * @param line Text line to write
    * @param line_length Number of bytes to transfer
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbPutLine() fills the field indicated by the cursor with
    * the contents of the buffer provided as input. The first call to
    * TdbPutLine() after setting the \p cursor parameter causes the existing
    * contents of the field to be replaced. Subsequent calls append the
    * contents of the buffer to the end of the field specified in the domain
    * of the \p cursor parameter. The record can be saved with a call to the
    * TdbPutRecord() function.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TdbHandle as the cursor identifier
    * for the record; this will have been set with a prior call to the
    * TdbSetCursor() function.
    *
    * The \p line parameter is passed a character string as the line of text
    * to copy to the record. Strictly this function copies the text line to
    * the record control associated with the \p cursor argument.
    *
    * The \p line_length parameter is passed an \c int as the count of the
    * number of bytes in the buffer ( \p line ) to transfer.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 610 | Database {1} not found. |  |
    * | 962 | File name error, check logical names for {1}. |  |
    * | 1314 | Database 1 is OS read protected. |  |
    * | 2178 | Invalid {NUM}. |  |
    * | 2210 | Invalid {DAT}. |  |
    * | 2242 | Invalid {TIM}. |  |
    * | 2530 | No access to database {1}. |  |
    * | 3106 | Old version of BIF/VIF files. |  |
    * | 7010 | Database LOG file cannot be opened. |  |
    * | 9538 | No write access to the database {2}. |  |
    * | 11362 | Max 250 databases can be opened during a TRIP session. |  |
    * | 13762 | No write access to field: {1} |  |
    * | 15010 | {1} is a database cluster. | The database name passed to the function refers to a database cluster and not a single database. |
    * | 16706 | Invalid {INT}. |  |
    * | 17314 | No access to DB cluster {1}. |  |
    * | 18178 | Use TdbCheckPartName() for checking/storing part names. |  |
    * | 19490 | Ill-formed or invalid integer. |  |
    * | 19522 | Ill-formed or invalid number. |  |
    * | 19554 | Ill-formed or invalid date. |  |
    * | 19586 | Ill-formed or invalid time. |  |
    * | 19618 | Invalid phrase. |  |
    * | 19650 | The phrase does not match the specified pattern. |  |
    * | 19746 | Subfield contains more than 255 characters. |  |
    * | 19778 | Subfield contains just blank characters. |  |
    *
    * @see TdbPutRecord()
    * @see TdbSetCursor()
    */
   int TdbPutLine(TdbHandle cursor, char* line, int line_length);


   /**
    * @brief Returns the number of paragraphs within a field.
    * @param cursor Cursor identifier
    * @param count Paragraph count
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * Returns the number of paragraphs within a field specified by the
    * cursor position.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a a cursor handle.
    *
    * The \p count parameter is passed a pointer to an integer to receive the
    * count of the number of paragraphs in the domain of the \p cursor
    * system identifier.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 1314 | Database {1} is OS read protected. |  |
    *
    * @see TdbSetCursor()
    */
   int TdbParaCount(TdbHandle cursor, int* count);
   

   /**
    * @brief Returns the number of part records within a record.
    * @param cursor Cursor identifier
    * @param count Paragraph count
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbPartCount() returns the number of part records within a
    * record identified by the \p cursor parameter. The record must have been
    * loaded by a call to one of the TdbGetRecord(), TdbGetRecordInBase(),
    * TdbGetRecordInSearch() or TdbGetRecordInSearchRis() functions.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a \c TbdHandle as a cursor identifier.
    *
    * The \p count parameter is passed a pointer to an integer that receives
    * the count of the number of paragraphs in the domain of the \p cursor
    * system identifier.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 14658 | Undefined cursor. |  |
    *
    * @see TdbGetRecord()
    * @see TdbGetRecordInBase()
    * @see TdbGetRecordInSearch()
    * @see TdbGetRecordInSearchRis()
    * @see TdbSetCursor()
    */
   int TdbPartCount(TdbHandle cursor, int* count);


   /**
    * @brief Write a record into a database
    * @param recordControl Record control identifier
    * @param recordId Record number
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbPutRecord() writes a record into the database. The
    * record is held in an internal work area, i.e. the record control. 
    * If the record has previously been read into the record control (by
    * calling one of the TdbGetRecord(), TdbGetRecordInBase(), 
    * TdbGetRecordInSearch() or TdbGetRecordInSearchRis() functions) this
    * routine will update the record in the BAF file, otherwise a new record
    * will be added, and the \p recordId parameter will receive the number
    * that was assigned to the record by the TRIPkernel. A requirement for
    * updating the database is that the function TdbSetBase() was previously
    * called with an access mode of \c MODE_WRITE specified.
    *
    * Alternatively, a copy of an existing record may be added to the database.
    * The function TdbSetBase() should then be called to specify access mode
    * \c MODE_READ before loading the existing record into the record control
    * via one of the aforementioned routines. Prior to storing the record,
    * however, TdbSetBase() must be called again, this time with the access
    * mode of \c MODE_COPY specified.
    *
    * Alternatively specifying the access mode \c MODE_COPY indicates to
    * TdbPutRecord() that a record is to be added, not updated as would
    * normally be the case when an existing record is used.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a handle to a record control.
    * The record control object must have been previously created by a call to
    * the function TdbCreateRecordControl() and attached to a database by a
    * call to the function TdbSetBase() (with the access mode set to either of
    * \c MODE_WRITE or \c MODE_COPY depending upon the intended usage).
    *
    * The \p recordId parameter is passed a pointer to an \c int variable as 
    * the record number identifying the record to be updated. For storage of a 
    * new record, set this variable to 0 on input. For update of a record, set
    * this variable to the number of the record to update. On output this parameter 
    * always receives the number of the record that was written.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 130 | Record control structure not created. |  |
    * | 290 | Invalid BAF,BIF,VIF or DBL file name. |  |
    * | 1218 | Field 1 cannot be copied; different field types. |  |
    * | 2882 | Non-existing record: {1} |  |
    * | 7042 | Missing mandatory field: {1} |  |
    * | 7074 | Already existing record name: {1} |  |
    * | 7202 | Disk quota exceeded for database {2} |  |
    * | 7298 | Record name field must not be blank. |  |
    * | 7330 | Record name field must not be changed. |  |
    * | 7522 | Database {2} is OS write protected. |  |
    * | 9090 | Database {2} is locked by another user, try again later. |  |
    * | 9122 | Database log for {2} is locked by another user, try again later. |  |
    * | 9538 | No write access to the database {2}. |  |
    * | 14402 | Field 1 cannot be copied; main/part record conflict. |  |
    * | 15202 | The BAF file of 2 needs reorganization (See PACKIT). |  |
    * | 17218 | The {1} file of {2} is no {1} file. |  |
    * | 17634 | Minimum no of subfields for {1} is {3}. |  |
    * | 17666 | Minimum no of paragraphs for {1} is {3}. |  |
    * | 18338 | No database connection. |  |
    * | 14754 | Database {2} is open for read only. |  |
    * | 14818 | No write access to this record. |  |
    * | 15714 | Field {1} conflict, {2} {1} paragraph only |  |
    * | 19490 | Ill-formed or invalid integer. |  |
    * | 19522 | Ill-formed or invalid number. |  |
    * | 19554 | Ill-formed or invalid date. |  |
    * | 19586 | Ill-formed or invalid time. |  |
    * | 19618 | Invalid phrase. |  |
    * | 19650 | The phrase does not match the specified pattern. |  |
    * | 19746 | Subfield contains more than 255 characters. |  |
    * | 19778 | Subfield contains just blank characters. |  |
    * | 22722 | Old version of the {1} file, reindex {2}. |  |
    * | 25954 | Cannot load foreign key link database's CONTROL record (1) |  |
    * | 25986 | The operation on record {1} has been aborted because it would cause an integrity violation |  |
    * | 26082 | Some records affected by the update could not be modified -- database integrity has been compromised |  |
    * | 33539 | Record {1} added. | The function completed its operations without error. |
    * | 33571 | Record {1} modified. | The function completed its operations without error. |
    * | 33891 | Empty record, not stored. | The function completed its operations without error. |
    * | 33923 | No changes made to this record. | The function completed its operations without error. |
    *
    * @see TdbCreateRecordControl()
    * @see TdbGetRecord()
    */
   int TdbPutRecord(TdbHandle recordControl, int* recordId);
   

   /**
    * @brief Load a TRIP record with TFORM-formatted data.
    * @param cursor cursor identifier
    * @param buffer_address Buffer address
    * @param buffer_len Length of buffer
    * @param err_partnr Number of first part record with error
    * @param err_fieldnr Number of first field with error
    * @param err_itemnr Number of first item with error
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbPutRecordBuffer() is used to load a TRIP record with
    * TFORM-formatted data. Remember to call TdbPutRecord() to commit the
    * changes to the database.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a handle to a cursor for the record.
    *
    * The \p buffer_address parameter is passed the address of the buffer with
    * TFORM data to write to the database.
    *
    * The \p buffer_len parameter is passed an \c int as the number of bytes
    * in the buffer ( \p buffer_address ) to write to the database.
    *
    * The \p err_partnr parameter is passed a pointer to an \c int variable to 
    * receive the number of the first part record that contains a field failing 
    * a constraint in the database design.
    *
    * The \p err_fieldnr parameter is passed a pointer to an \c int variable to 
    * receive the number of the first field that fails to comply with a constraint 
    * in the database design.
    *
    * The \p err_itemnr parameter is passed a pointer to an \c int variable to 
    * receive the number of the first item that fails to comply with a constraint 
    * in the database design.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 610 | Database {1} not found. |  |
    * | 962 | File name error, check logical names for {1}. |  |
    * | 1314 | Database {1} is OS read protected. |  |
    * | 2178 | Invalid {NUM}. |  |
    * | 2210 | Invalid {DAT}. |  |
    * | 2242 | Invalid {TIM}. |  |
    * | 2530 | No access to database {1}. |  |
    * | 3106 | Old version of BIF/VIF files. |  |
    * | 9538 | No write access to the database {2}. |  |
    * | 11362 | Max 250 databases can be opened during a TRIP session. |  |
    * | 12130 | Attempt to open a previously deleted database. |  |
    * | 13762 | No write access to field: {1} |  |
    * | 15010 | {1} is a database cluster. | The database name passed to the function refers to a database cluster and not a single database. |
    * | 16706 | Invalid {INT}. |  |
    * | 17314 | No access to DB cluster 1. |  |
    * | 18178 | Use TdbCheckPartName() for checking/storing part names. |  |
    * | 19490 | Ill-formed or invalid integer. |  |
    * | 19522 | Ill-formed or invalid number. |  |
    * | 19554 | Ill-formed or invalid date. |  |
    * | 19586 | Ill-formed or invalid time. |  |
    * | 19618 | Invalid phrase. |  |
    * | 19650 | The phrase does not match the specified pattern. |  |
    * | 19746 | Subfield contains more than 255 characters. |  |
    * | 19778 | Subfield contains just blank characters. |  |
    * | 25602 | Out of memory |  |
    *
    * @see TdbPutRecord()
    * @see TdbSetCursor()
    */
   int TdbPutRecordBuffer(TdbHandle cursor, char* buffer_address,
                          int buffer_len, int* err_partnr, int* err_fieldnr,
                          int* err_itemnr);

   
   /**
    * @brief Returns information about a record.
    * @param recordControl Record control for record to get information about.
    * @param mode Type of information to retrieve.
    * @param stringValue Output parameter for information of string type.
    * @param intValue Output parameter for information of integer type.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbRecordInfo() function provides a way to obtain various
    * information about a TRIP database record.
    *
    * #### Parameter Values
    * The \p recordControl parameter is a record control that refers to a
    * TRIP database record. The record control is created using
    * TdbCreateRecordControl() and initialized to a specific record using
    * TdbGetRecord() or one of the deprecated functions such as 
    * TdbGetRecordInSearchRis() or TdbGetRecordInBase().
    *
    * The \p mode parameter is used to tell the function what information to
    * return and must be set to one of the following values:
    *
    * | Value | Output Parameter | Description |
    * | ----- | ---------------- | ----------- |
    * | RECORDINFO_RID | intValue | Return the record identifier (RID) of the record. |
    * | RECORDINFO_DBNAME | stringValue | Return the name of the database the record is stored in. |
    * | RECORDINFO_RNAME | stringValue | Return the record name value of the record. |
    * | RECORDINFO_TSTAMP_CODED | intValue | Returns the modification timestamp of the record in a numeric representation. |
    * | RECORDINFO_TSTAMP_ASCII | stringValue | Returns the modification timestamp of the record in the current date form. |
    * | RECORDINFO_CLASS_NAME | stringValue | Returns the name of the class assigned to this record. Requires that the database is associated with a classification scheme. |
    * | RECORDINFO_HITS | intValue | Returns the total number of hits in the record. The record must be retrieved from the latest search set using TdbGetRecordInSearchRis. |
    * | RECORDINFO_TSTAMP_UTC | intValue | Returns the modification timestamp of the record as a Unix time value in UTC. |
    * | RECORDINFO_TSTAMP_UTCDAY | intValue | As RECORDINFO_TSTAMP_UTC, but with the hour, minute and second set to midnight (00:00:00). |
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 7298 | Record name field must not be blank. |  |
    * | 7394 | No record name field defined in {1}. |  |
    * | 17826 | No of subfields in 3 exceeds the limit of {1}. |  |
    *
    * @see TdbCreateRecordControl()
    * @see TdbGetRecord()
    * @see TdbGetRecordInSearchRis()
    * @see TdbGetRecordInBase()
    */
   int TdbRecordInfo(TdbHandle recordControl, int mode, char* stringValue,
                     int* intValue);

   
   /**
    * @brief Returns information about a cursor.
    * @param cursor Cursor from which to retrieve information
    * @param mode Type of information to retrieve.
    * @param stringValue Output parameter for information of string type.
    * @param intValue Output parameter for information of integer type.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbCurosrInfo() function provides a way to obtain various information
    * about a TRIP cursor.
    *
    * #### Parameter Values
    * The \p cursor parameter is a handle to a cursor that may refer to a
    * TRIP database record, part record or field. The cursor is created using
    * TdbSetCursor() .
    *
    * The \p mode parameter is used to tell the function what information to
    * return and must be set to one of the following values:
    *
    * | Value | Output Parameter | Description |
    * | ----- | ---------------- | ----------- |
    * | CURSORINFO_DB | stringValue | Return the name of the database associated with the cursor |
    * | CURSORINFO_RID | intValue | Return the record identifier (RID) of the record to which the cursor refer. |
    * | CURSORINFO_PARTID | intValue | Return number of the part record associated with the cursor. |
    * | CURSORINFO_FIELDNR | intValue | Return number of the field associated with the cursor. |
    * | CURSORINFO_PARANR | intValue | Return number of the text field paragraph associated with the cursor. |
    * | CURSORINFO_ITEMNR | intValue | Return number of the field item associated with the cursor. |
    * | CURSORINFO_HITS | intValue | Returns the total number of hits in the field. The cursor must refer to a field in a record that has been retrieved from the latest search set using TdbGetRecordInSearchRis. |
    *
    * #### Since
    * This function was introduced in TRIP version 6.2-9.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 18466 | Unrecognised option: {1}. |  |
    *
    * @see TdbSetCursor()
    * @see TdbRecordInfo()
    */
   int TdbCursorInfo(TdbHandle cursor, int mode, char* stringValue,
                     int* intValue);

   
   /**
    * @brief Release a lock on a record
    * @param recordControl Record control identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbReleaseRecord() releases the lock on the record that is
    * currently in the record control area. The record could have been locked
    * by a call to one of the functions TdbLockRecord(), TdbgGetRecord(), 
    * TdbGetRecordInBase(), TdbGetRecordInSearch() or TdbGetRecordInSearchRis(),
    * if the preceding call to the function TdbSetBase() had an access mode of 
    * \c MODE_WRITE specified. A record is automatically locked if it is loaded
    * in write-access mode.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a handle to a record control.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 16418 | TRIPdaemon failed to respond (1), notify your System Mgr. |  |
    * | 16450 | TRIPdaemon failure (1), notify your System Mgr. |  |
    * | 16482 | Your process is unknown to the TRIP daemon, notify your System Mgr. |  |
    *
    * @see TdbLockRecord()
    * @see TdbGetRecord()
    * @see TdbGetRecordInBase()
    * @see TdbGetRecordInSearch()
    * @see TdbGetRecordInSearchRis()
    * @see TdbSetBase()
    */
   int TdbReleaseRecord(TdbHandle recordControl);

   
   /**
    * @brief Rename a record part
    * @param cursor Cursor identifier
    * @param part_name Name of record part
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbRenameRecordPart() is used to rename the record part
    * that the \p cursor parameter is currently addressing. The cursor must
    * have been attached to the record control area by using the function
    * TdbSetRecordPart() prior to attempting to call this function.
    *
    * The routine can only be used to update databases that include a part
    * record name field.
    *
    * The function TdbPutRecord() must be called in order to save record.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a handle to a cursor identifying a
    * part record.
    *
    * The \p part_name parameter is passed a character string specifying 
    * the new name of the record part.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 17986 | Database {1} has no part name field. |  |
    *
    * @see TdbPutRecord()
    * @see TdbSetRecordPart()
    */
   int TdbRenameRecordPart(TdbHandle cursor, const char* part_name);

   
   /**
    * @brief Replace a data item in a record control area
    * @param cursor Cursor identifier
    * @param data_item Data item to replace existing data
    * @param item_length Number of bytes in data item buffer
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbReplaceItem() replaces the value of the sentence or
    * subfield indicated by the cursor with the value contained in the buffer.
    * This routine may be called directly after a call to the TdbSetCursor()
    * has set the domain of the supplied cursor. If the domain covers a field
    * or a paragraph, all of the domain is replaced by the buffer contents.
    *
    * In order to save the updated record the function TdbPutRecord() must be
    * called.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a handle to a cursor. This argument
    * will have been set with a prior call to the TdbSetCursor() function.
    *
    * The \p data_item parameter is passed a character string as the data item
    * to replace the existing data in the domain of the \p cursor argument.
    *
    * The \p item_length parameter is passed an \c int as the count of the
    * number of bytes in the data item buffer to use. The entire item in the
    * domain of \p cursor will be replaced irrespective of the value of this
    * argument.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 2178 | Invalid {NUM}. |  |
    * | 2210 | Invalid {DAT}. |  |
    * | 2242 | Invalid {TIM}. |  |
    * | 13762 | No write access to field: {1} |  |
    * | 16706 | Invalid {INT}. |  |
    * | 18178 | Use TdbCheckPartName() for checking/storing part names. |  |
    * | 19490 | Ill-formed or invalid integer. |  |
    * | 19522 | Ill-formed or invalid number. |  |
    * | 19554 | Ill-formed or invalid date. |  |
    * | 19586 | Ill-formed or invalid time. |  |
    * | 19618 | Invalid phrase. |  |
    * | 19650 | The phrase does not match the specified pattern. |  |
    * | 19746 | Subfield contains more than 255 characters. |  |
    * | 19778 | Subfield contains just blank characters. |  |
    *
    * @see TdbPutRecord()
    * @see TdbSetCursor()
    */
   int TdbReplaceItem(TdbHandle cursor, char* data_item, int item_length);

   
   /**
    * @brief Set the access mode for a database
    * @param recordControl Record control identifier
    * @param base Database name
    * @param access_mode Access mode for database
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbSetBase() is used to make a database accessible for input
    * and/or update operations via the TdbGetRecord(), TdbGetRecordInBase(),
    * TdbGetRecordInSearch(), TdbGetRecordInSearchRis() and TdbPutRecord()
    * functions.
    *
    * If a copy of an existing record is to be added to the database, the
    * function TdbSetBase() must be called twice. The first time (before
    * loading the record into the record control) an access mode of
    * \c MODE_READ should be specified. The second time (before adding the
    * record to the database) an access mode of \c MODE_COPY must be set.
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a handle to a record control.
    * This record control object must have been previously created by a call
    * to the TdbCreateRecordControl() function.
    *
    * The \p base parameter is passed a character string as the name of the
    * database to be associated with the record control argument. On a successful
    * call, this parameter receives the normalized name of the database.
    *
    * The \p access_mode parameter is passed an \c int as the access mode for
    * the database. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | MODE_READ | bit 0 | Require read access to the database |
    * | MODE_WRITE | bit 1 | Require modify access to the database |
    * | MODE_DELETE | bit 2 | Require delete access to the database |
    * | MODE_COPY | bit 3 and bit 1 | Require copy access to the database |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 1218 | Field {1} cannot be copied; different field types. |  |
    * | 10594 | No delete rights to database {1}. |  |
    * | 14402 | Field {1} cannot be copied; main/part record conflict. |  |
    * | 15714 | Field {1} conflict, {2} {1} paragraph only |  |
    * | 16738 | No database specified for data entry. |  |
    * | 17826 | No of subfields in {3} exceeds the limit of {1}. |  |
    *
    * @see TdbGetRecord()
    * @see TdbGetRecordInBase()
    * @see TdbGetRecordInSearch()
    * @see TdbGetRecordInSearchRis()
    * @see TdbPutRecord()
    */
   int TdbSetBase(TdbHandle recordControl, char* base, int access_mode);

   
   /**
    * @brief Set a cursor domain
    * @param cursor Cursor identifier
    * @param recordControl Record control identifier
    * @param field Field number
    * @param paragraph Paragraph number
    * @param item Item number
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbSetCursor() is used to create and/or set a cursor
    * indicating a record or record substructure. The domain of the cursor is
    * defined by field, paragraph and subfield or sentence numbers within the
    * record contained in the record control area.
    *
    * The cursor domain can be made to cover more than a single
    * sentence/subfield by setting the item argument to zero. For any non-TExt
    * field, this will cause the entire field to be included in the domain,
    * whereas for a TExt field, the paragraph argument must also be set to
    * zero, or else the domain will be limited to the specified paragraph.
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a pointer to a TdbHandle variable that
    * either specifies the cursor to set or receives a newly created cursor.
    * The first time the function TdbSetCursor() is called for a specific
    * cursor, the variable to which the parameter points must be set to NULL to
    * indicate that the cursor has to be created before it is positioned.
    *
    * The \p recordControl parameter is passed a handle to a record control.
    * The record control specified is that which the cursor is to address. The
    * record control must already exist, typically with a call to the
    * TdbCreateRecordControl() function.
    *
    * The \p field parameter is passed an integer as the field number to set as
    * the cursor domain.
    *
    * The \p paragraph parameter is passed an integer as the paragraph number
    * to set as the cursor domain. This argument is only meaningful if the
    * field currently addressed by the cursor is a TExt type field. If a value
    * of zero is provided then the domain will be the entire field.
    *
    * The \p item parameter is passed an integer as the item number to set as
    * the cursor domain. If the field addressed by the cursor is a TExt field,
    * the item corresponds to the sentence within the paragraph. If the field
    * is of any type other than TExt, the item corresponds to the subfield
    * within the field. If a value of zero is provided then the domain will be
    * the entire paragraph (for TExt fields) or the entire field for all other
    * types.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 2146 | Non-existing field: {1} |  |
    *
    * @see TdbCreateRecordControl()
    * @see TdbDeleteCursor()
    */
   int TdbSetCursor(TdbHandle* cursor, TdbHandle recordControl, int field,
                    int paragraph, int item);


   /**
    * @brief Make a database accessible for data entry linkage
    * @param recordControl Record control
    * @param base_name Name of linked database
    * @param base_id TRIP internal identification number for database.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbSetLinkBase() opens a database for searching and makes it
    * accessible for linkage during data entry. The database identifier must
    * be set to zero on the first call. The \p base_id parameter will contain
    * the actual identifier on return from the function.
    *
    * The function TdbGetLinkRecord() is used to get at records from the link
    * database by the record name. If a record is found, the data can be
    * fetched by setting a cursor with TdbSetCursor() using the link record
    * record control and then call TdbGetLine() .
    *
    * #### Parameter Values
    * The \p recordControl parameter is passed a handle to a record control.
    *
    * The \p base_name parameter is passed a character string as the name of
    * the linked database.
    *
    * The \p base_id parameter is passed a pointer to a TdbHandle variable that
    * will receive the TRIP identification for the database.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 610 | Database {1} not found. |  |
    * | 962 | File name error, check logical names for {1}. |  |
    * | 1314 | Database {1} is OS read protected. |  |
    * | 2530 | No access to database {1}. |  |
    * | 3106 | Old version of BIF/VIF files. |  |
    * | 7010 | Database LOG file cannot be opened. |  |
    * | 7522 | Database {2} is OS write protected. |  |
    * | 9538 | No write access to the database {2}. |  |
    * | 11362 | Max 250 databases can be opened during a TRIP session. |  |
    * | 12130 | Attempt to open a previously deleted database. |  |
    * | 15010 | {1} is a database cluster. | The database name passed to the function refers to a database cluster and not a single database. |
    * | 17314 | No access to DB cluster {1}. |  |
    * | 17858 | Chinese vocabulary files not found. |  |
    * | 25602 | Out of memory |  |
    *
    * @see TdbGetLinkRecord()
    * @see TdbGetLinkBase()
    * @see TdbSetCursor()
    * @see TdbGetLine()
    */
   int TdbSetLinkBase(TdbHandle recordControl, char* base_name,
                      TdbHandle* base_id);
   
   
   /**
    * @brief Specify the record part
    * @param cursor Cursor identifier
    * @param recordControl Record control identifier
    * @param part_number Part number
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbSetRecordPart() is used to specify the record part to
    * which a cursor refers. If the part number is zero, the cursor will be
    * set to refer to a new, not yet created record part at the end of the part
    * list.
    *
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a pointer to a TdbHandle variable that
    * specifies a cursor handle. If the handle is set to NULL on input, then a
    * new cursor is created; otherwise, the value is taken as that of an
    * existing cursor and it is modified to refer to the specified record part.
    *
    * The \p recordControl parameter is passed a handle to the record control
    * to associate with the \p cursor argument.
    *
    * The \p part_number parameter is passed an integer as the part number. A
    * value of zero indicates a that a new part is to be created. Passing zero
    * also indicates the head record in the context of reading data from the
    * record.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 | Failure | Operation failed |
    * | 1 | Cursor was set to refer to an existing part | The function completed its operations without error. |
    * | 3 | Non-existing part, i.e. new part record to be created. | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    *
    * @see TdbSetCursor()
    */
   int TdbSetRecordPart(TdbHandle* cursor, TdbHandle recordControl,
                        int part_number);

   
   /**
    * @brief Initialize reading of CONTROL records
    * @param mode Operation mode
    * @param all_flag All flag
    * @param name1 Name string
    * @param name2 Name string
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbShowControl() initializes a series of calls to
    * TdbNextControl() . This routine will provide the same functionality as
    * the CCL commands for showing contents from the CONTROL file, sorted
    * alphabetically. It must be called before the first call the 
    * TdbNextControl() function in order to specify which records are to be 
    * transferred. Also see TdbStartControl(), which can be used to set a 
    * starting point in the list of selected records.
    *
    * #### Parameter Values
    * The \p mode parameter is passed an \c int to indicate the type of
    * information to retrieve from the CONTROL file. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | DATABASE_LIST | 1 | Return database list (equivalent to CCL: order BASE) |
    * | USER_LIST | 2 | Return user list (equivalent to CCL order SHOW USER) |
    * | DATABASE_STATUS | 3 | Return status information equivalent to CCL order SHOW BASE) |
    * | PROCEDURE_LIST | 4 | Return procedure list (equivalent to CCL order SHOW PROC) |
    * | DATABASE_ACCESS | 5 | Return database access list (equivalent to CCL order SHOW ACCESS) |
    * | OUTPUT_FORMAT_LIST | 6 | Return output format list (equivalent to CCL order SHOW FORMAT) |
    * | ENTRY_FORM_LIST | 7 | Return entry form list (equivalent to CCL order SHOW EFORM) |
    * | SEARCH_FORM_LIST | 8 | Return search form list (equivalent to CCL order SHOW SFORM) |
    * | THESAURUS_STATUS | 9 | Return thesauri status information (equivalent to CCL order SHOW THES) |
    * | GROUP_LIST | 10 | Returns group information (equivalent to CCL order SHOW GROUP) |
    * | CURRENT_STATUS | 11 | Returns single database/cluster status (equivalent to CCL order STATUS) |
    * | USERS_PER_GROUP | 14 | Return information on all members of one group |
    * | GROUPS_PER_USER | 15 | Return information on all groups for one user |
    *
    * The \p all_flag parameter is passed an \c int as the indicator of scope
    * of information to be returned. Valid values are:
    *
    * | Constant Value | Purpose |
    * | ---- | ---- |
    * | 0 | Only possible value for a normal user. Limits scope of information to that which they can retrieve using the equivalent CCL order. |
    * | 1 | System manager can set this value to examine all records of the current mode |
    *
    * The \p name1 parameter is passed a character string as the first name
    * string. This argument is used to specify the first or only argument, e.g,
    * database name , user name . A blank string indicates all records of the
    * current mode.
    *
    * The \p name2 parameter is passed a character string as the second name
    * string. A blank string indicates all records that are connected to the
    * \p name1 argument.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 610 | Database {1} not found. |  |
    * | 930 | Syntax error. |  |
    * | 6626 | No database is open, supply database name. |  |
    * | 8866 | Missing argument. |  |
    * | 9762 | You are not the creator of this record. |  |
    * | 9794 | You are not a file manager. |  |
    * | 9826 | You are not a user manager. |  |
    * | 9922 | You are not a SYSTEM manager. |  |
    * | 15746 | System Manager rights is needed for the ALL argument |  |
    * | 15778 | The ALL argument must not be used with name arguments |  |
    *
    * @see TdbNextControl()
    * @see TdbStartControl()
    */
   int TdbShowControl(int mode, int all_flag, char* name1, char* name2);
  
  
   /**
    * @brief Sorts a display order or term set.
    * @param termSetNumber The number of the term set to sort, or 0 (zero) to sort the latest regular term list.
    * @param sortFlags Sort specification bit mask
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbSortTerms() function is used to apply a new sort order to
    * the latest executed DISPLAY order or registered term set.
    *
    * #### Parameter Values
    * Pass 0 (zero) to the \p termSetNumber parameter to re-sort the latest
    * DISPLAY order, or a positive number identifying the term set to re-sort.
    * 
    * The \p sortFlags parameter is a bit mask that specifies how to sort the 
    * term list. The mask must contain exactly one of the following values:
    * 
    * | Symbol | Description |
    * | ------ | ----------- |
    * | TERM_SORT_TEXT | Sort alphabetically, ascending |
    * | TERM_SORT_FREQ | Sort on frequency, descending |
    * | TERM_SORT_TEXT_DESC | Sort alphabetically, descending |
    * | TERM_SORT_FREQ_ASC | Sort on frequency, descending |
    * 
    * When sorting a baseline term set, the \p sortFlags bit mask may also 
    * include the following:
    * 
    * | Symbol | Description |
    * | ------ | ----------- |
    * | TERM_SORT_BL_ATEND | Sort zero-sized baseline terms at end |
    *
    * #### Return Values
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbCreateTermSet()
    * @see TdbExecuteCcl()
    */
   int TdbSortTerms(int termSetNumber, int sortFlags);

   
   /**
    * @brief Sort an existing search set.
    * @param searchNumber The number of the search set to sort.
    * @param sortSpecification This is the SORT specification.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbSortRecords() function is used to sort search sets according to
    * a SORT specification. To iterate through the sorted search set, use the
    * TdbGetRecord() function with mode \c RECORD_FROM_SORTED_SEARCH .
    *
    * #### Parameters
    * The \p sortSpecification parameter specifiess only the part that is to
    * the right of the '=' sign in the CCL "SHOW SORT=" statement.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 66 | Unbalanced parenthesis. | The SORT clause contained an unbalanced parenthesis. |
    * | 994 | There is no search for implicit use. |  |
    * | 1698 | Non-existing field name: {2} |  |
    * | 3970 | Sort order syntax error. |  |
    * | 4002 | Too many records {1} for sort, current limit is {2}. |  |
    * | 4034 | {1} field type not allowed in sort order. |  |
    * | 4066 | Sort key creation error. |  |
    * | 13186 | Set is empty. | The sort failed because the search set did not contain any records. |
    * | 19490 | Ill-formed or invalid integer. |  |
    * | 22722 | Old version of the {1} file, reindex {2}. | Index structure is too old. Reindex the database. |
    *
    * @see TdbGetRecord()
    */
   int TdbSortRecords(int searchNumber, char* sortSpecification);


   /**
    * @brief Sort an existing search set with clusters and/or record parts.
    * @param searchNumber The number of the search to sort.
    * @param sortSpecification This is the SORT specification.
    * @param mode Search mode bitmask 
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbSortRecordsEx() function is used to sort search sets according
    * to a SORT specification. When sorting a search set from a cluster or 
    * if sorting should consider record parts, this function should be used
    * instead of the similarly named \p TdbSortRecords function. To iterate
    * through the sorted  search set, use the TdbGetRecord() function with 
    * mode \c RECORD_FROM_SORTED_SEARCH .
    *
    * #### Parameters
    * The \p sortSpecification parameter specifiess only the part that is to
    * the right of the '=' sign in the CCL "SHOW SORT=" statement.
    *
    * The \p mode parameter is passed an integer as a bitmask 
    * indicating additional control parameters for the sort operation.
    * Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | SORT_MERGE | 1 | Sort records from all databases in cluster together. Mutually exclusive with SORT_NOMERGE. |
    * | SORT_NOMERGE | 2 | Sort records from database in cluster separately. Mutually exclusive with SORT_MERGE. |
    * | SORT_PARTS | 3 | Perform part-record oriented sorting |
    *
    * If the \p mode parameter does not include either SORT_MEGRGE or 
    * SORT_NOMERGE, the sort operation will use the DEFINE [NO] MERGE
    * value currently set in the session.
    *
    * #### Since
    * This function was introduced in TRIP version 8.1-1:3
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 66 | Unbalanced parenthesis. | The SORT clause contained an unbalanced parenthesis. |
    * | 994 | There is no search for implicit use. |  |
    * | 1698 | Non-existing field name: {2} |  |
    * | 3970 | Sort order syntax error. |  |
    * | 4002 | Too many records {1} for sort, current limit is {2}. |  |
    * | 4034 | {1} field type not allowed in sort order. |  |
    * | 4066 | Sort key creation error. |  |
    * | 13186 | Set is empty. | The sort failed because the search set did not contain any records. |
    * | 19490 | Ill-formed or invalid integer. |  |
    * | 22722 | Old version of the {1} file, reindex {2}. | Index structure is too old. Reindex the database. |
    *
    * @see TdbGetRecord()
    */
   int TdbSortRecordsEx(int searchNumber, const char* sortSpecification, int mode);

   
   /**
    * @brief Set the initial value for reading CONTROL records
    * @param name Name of first CONTROL record
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbStartControl() loads the name of the CONTROL database
    * record that is to be used as the start point for a sequence of calls to
    * the TdbShowControl() and TdbNextControl() functions.
    *
    *
    * #### Parameter Values
    * The \p name parameter is passed a character string as the name of the
    * first CONTROL record to show.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Failure | No CONTROL records could not be found at or after the specified starting point. |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 3 | No exact match was found, but the next CONTROL record (in alphabetical order) was selected. | The function completed its operations without error. |
    *
    * @see TdbNextControl()
    * @see TdbShowControl()
    */
   int TdbStartControl(char* name);

   
   /**
    * @brief Undelete the previously deleted record part
    * @param cursor Cursor identifier
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The function TdbUndeleteRecordPart() is used to restore the record part
    * that was previously deleted by a call to the function
    * TdbDeleteRecordPart(), the mode argument to that routine set to one. The
    * record part will be inserted before the part to which the cursor is
    * currently referring.
    *
    * The routine can only be used to update databases that include a part
    * record name field.
    *
    * The function TdbPutRecord() must be called in order to save the record.
    *
    *
    * #### Parameter Values
    * The \p cursor parameter is passed a handle to a cursor. The cursor is
    * used as a reference to data in a TRIP internal work area.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 130 | Record control structure not created. |  |
    * | 16066 | There is no deleted part record to undelete. |  |
    *
    * @see TdbDeleteRecordPart()
    */
   int TdbUndeleteRecordPart(TdbHandle cursor);

   
   /**
    * @brief Produce term lists in combination with link databases
    * @param field_number Number of field(s) to check
    * @param base_id Link database id
    * @param term Term to display
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * The TdbValidValues() function is used to produce term lists in
    * combination with link databases. This call will fill the display window
    * (if defined), and selection of terms must be carried out by the
    * application.
    *
    * After the call to TdbValidValues(), do the following to retrive the terms.
    *
    * The list is from the current database or a restriction database:
    *     + Successive calls to TdbGetDisplayTerm() will return all terms matching the query input term.
	*
    * The list is from a link database:
    *     + Call TdbGetDisplayTerm() to get the term value.
    *
    * Call TdbGetLinkRecord() to read the corresponding record from the link
    * database. Then use standard methods to read desired fields from this
    * database.
    *
    * @note Make sure to use a negative term number for all calls to
    * TdbGetDisplayTerm() in order to get the terms in its original form.
    *
    * #### Parameter Values
    * The content of the \p field_number parameter depends on what kind of
    * list is to be produced. The choice of list is made by TRIP in the
    * priority order shown here. A list from a link database; one or two
    * field numbers. A list from a restriction database; one field number.
    * A list from the current database; one field number.
    *
    * For a link database, the field numbers should be those which are to
    * be loaded from the link database. If two fields are to be shown,
    * \p field_number must have the value: f2*65536+f1, where f1 and f2 are
    * the field names.
    *
    * The \p base_id parameter is passed a \c TdbHandle as the link database id.
    * This is the value returned from TdbSetLinkBase(). If a link database is
    * not used, this parameter must be NULL.
    *
    * The \p term parameter is an input string that defines what to display.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 194 | No terms! |  |
    * | 578 | No index found for {1}. |  |
    *
    * @see TdbSetLinkBase()
    * @see TdbGetLinkRecord()
    * @see TdbGetDisplayTerm()
    */
   int TdbValidValues(int field_number, TdbHandle base_id, char* term);

   /**
    * @brief Returns data with original formatting from a field of any kind.
    * @param cursor [IN] Cursor set to a field
    * @param buf_len [OUT] Return length of data in \p rec_buf
    * @param rec_buf [OUT] Receives a pointer to the field value
    * @param hit_cnt [OUT] Number of hits in the returned value
    * @param hit_vec [OUT] Hit vector (array with offset/length pairs)
    * @param para_cnt [OUT] Number of paragraphs returned
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * When called for a TEXT field, the function will return each subfield
    * separated by an LF (line feed) character.
    * 
    * The memory returned in the output parameters \p rec_buf and \p hit_vec 
    * must be released by the caller using the TdbDeallocate() function.
    * 
    * @see TdbSetCursor()
    * @see TdbDeallocate()
    */
    int TdbGetFieldInfo(TdbHandle cursor, int* buf_len, char** rec_buf,
                        int* hit_cnt, int** hit_vec, int* para_cnt);
   
   /**
    * @brief Assign a value to a TEXT field
    * @param cursor Cursor set to a text field
    * @param buf_len Length of the value in \p rec_buf
    * @param rec_buf Value to store
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    */
   int TdbPutFieldInfo(TdbHandle cursor, int buf_len, char* rec_buf);
   

   /**
    * @brief Initialize the text summarization engine.
    * @param handle Receives a handle to the summarizer as a void pointer
    * @param engine_id ID of the summarization engine to use. Reseverd for future use. Always pass 1 as value.
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * Initializes the specified summarization engine for a summarization run.
    * The \p engine_id parameter currently only accepts 1 as valid value.
    *
    * After having called this function, call TdbAccumulateSummarization() any
    * number of times to accumulate data to summarize. Generate the summary by
    * calling TdbGenerateSummary(), and close the summarization run by a call
    * to the TdbEndSummarization() function.
    *
    * @see TdbAccumulateSummarization()
    * @see TdbGenerateSummary()
    * @see TdbEndSummarization()
    */
   int TdbStartSummarization(void** handle, int engine_id);
   
   /**
    * @brief Add data to the summarizer.
    * @param handle Handle to the summarizer
    * @param cursor Cursor referring to a field value to add
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * Adds data to the summarizer referred to by \p handle (obtained by a call
    * to TdbStartSummarization()).
    *
    * @see TdbStartSummarization()
    * @see TdbGenerateSummary()
    * @see TdbEndSummarization()
    */
   int TdbAccumulateSummarization(void* handle, TdbHandle cursor);
   
   /**
    * @brief Generate a summary.
    * @param handle Handle to the summarizer
    * @param summary Receievs a null-terminated string containing a summary
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * This function generates a summary based on the data added to the
    * summarizer via calls to the TdbAccumulateSummarization() function.
    * The returned value in \p summary is owned by the TRIP kernel and must
    * not be deallocated by the application.
    *
    * @see TdbStartSummarization()
    * @see TdbAccumulateSummarization()
    * @see TdbEndSummarization()
    */
   int TdbGenerateSummary(void* handle, Char** summary);
   
   /**
    * @brief Ends a summarization run
    * @param handle Handle to the summarizer
    * @return 1 or other odd value on success, even value on error
    * @ingroup datamanipulation
    *
    * #### Description
    * Closes a summarization run as opened by a call to TdbStartSummarization().
    *
    * @see TdbStartSummarization()
    * @see TdbAccumulateSummarization()
    * @see TdbGenerateSummary()
    */
   int TdbEndSummarization(void* handle);

   
   
   
   // ------------------------------------------------------------------------

   /**
    * @defgroup infocommands Information Commands
    * @brief Information request commands
    *
    * The information commands are functions that offer functionality for
    * obtaining miscellaneous information of various kind, such as about the
    * active session, the logged on user, command symbols, etc.
    */


   /**
    * @brief Return the mode in which the current record is being processed
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The TdbBaffitMode() function returns the type of operation performed on
    * the record as is it being processed by the BAFFIT utility program. It is
    * consequently only useful when called from within a LOAD or INDEX ASE.
    *
    * #### Return Codes
    * | Symbol | Constant Value | Description |
    * | ------ | -------------- | ----------- |
    * | ADD_MODE | 1 | A record is being added to the database |
    * | MODIFY_MODE | 2 | An existing record is being modified |
    * | DELETE_MODE | 3 | An existing record is being deleted |
    *
    * @see TdbBatchLoadIndex()
    */
   int TdbBaffitMode(void);
   

   /**
    * @brief Retrieve the current date
    * @param date Date string
    * @param length Length of date string
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * This function TdbCurrentDate() returns the current date as provided by
    * the system clock. Format of the return date will be according to the
    * format set in the user's profile. A TRIPapi based application can
    * override user profile settings by calling the TdbShellDefDateForm()
    * function.
    *
    *
    * #### Parameter Values
    * The \p date parameter is passed a character string that will receive the
    * current date (as provided by the operating system). This is a write-only
    * parameter.
    *
    * The \p length parameter is passed a pointer to an \c int variable that will 
    * receive the number of characters in the date string returned in the \p date 
    * parameter.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * @see TdbCreateTimeStamp()
    * @see TdbCurrentTime()
    * @see TdbShellDefDateForm()
    */
   int TdbCurrentDate(char* date, int* length);

   
   /**
    * @brief Retrieve the current time
    * @param timestr Time string
    * @param length Length of time string
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The function TdbCurrentTime() returns the current time, derived from the
    * system clock. Results are always in the format {HH:MM:SS} .
    *
    * #### Parameter Values
    * The \p timestr parameter is passed a character string to receive the
    *  current time derived from the operating system.
    *
    * The server-side version of the \p length parameter is passed a pointer
    * to an integer that receives a count of the number of data characters in
    * the \p timestr parameter.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * @see TdbCreateTimeStamp()
    * @see TdbCurrentDate()
    */
   int TdbCurrentTime(char* timestr, int* length);


   /**
    * @brief Returns various data about a field in a database.
    * @param cursor Cursor pointing at field to retrieve info about
    * @param mode Mode of operation
    * @param cvalue Information as character string
    * @param ivalue Information as integer
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * Returns various data about a field in a database.
    *
    * #### Parameter Values
    * The \p cursor parameter is a cursor that has been set to refer to the
    * field which to retrieve information about.
    *
    * The \p mode parameter is a flag used to request information about the
    * field specified by the \p cursor parameter. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose | Returned In |
    * | ------------- | -------------- | ------- | ----------- |
    * | FIELDINFO_SIZE | 1 | Size of field in characters | ivalue |
    * | FIELDINFO_TYPE | 2 | Type of field. | ivalue |
    * | FIELDINFO_MAX_SUBFIELDS | 3 | Max allowed subfield count | ivalue |
    * | FIELDINFO_IS_PARTFIELD | 4 | Indicates if field is a partfield | ivalue |
    *
    * The \p cvalue is passed a character string to receive the requested
    * information in character format.
    *
    * The \p ivalue is passed a pointer to an int that receives the requested
    * information in integer format.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 |  | Requested information is not available. |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 14658 | Undefined cursor. |  |
    * | 18464 | Unrecognized option: mode | The option value specified in the mode parameter is not valid. |
    * | 21634 | {1} is not a blob field. |  |
    */
   int TdbFieldInfo(TdbHandle cursor, int mode, char* cvalue, int* ivalue);

   
   /**
    * @brief Retrieve a job status message from the TRIPdaemon message queue
    * @param message Retrieved message
    * @param length Length of message
    * @param logfile Name of log file for batch job
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The function TdbGetQueuedMessage() retrieves a status message from the
    * TRIPdaemon when a batch job has finished executing. An application that
    * submits batch jobs (such as print, load, index, load/index, or global
    * update) should call this function at regular intervals to prevent the
    * message queue from eventually overflowing.
    *
    * If the \p logfile argument is non-null, the routine will also retrieve
    * the name of the log file that was created for the batch job.
    *
    * #### Parameter Values
    * The \p message parameter is passed a character string to contain the
    * message retrieved from the message queue.
    *
    * The \p length parameter is passed a pointer to an \c int variable that will
    * receive the length of the returned message, in number of bytes.
    *
    * The \p logfile parameter is passed a character string, which the call of
    * this function, will contain the name of the log file created for the
    * batch job.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | A queued message was retrieved. |
    * | 0 | Operation failed | There are no messages on queue. |
    */
   int TdbGetQueuedMessage(char* message, int* length, char* logfile);

   
   /**
    * @brief Retrieve current language code
    * @param language Buffer to receieve current language code.
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The function TdbLanguage() returns the three character language code that
    * is currently being used by the CCL parser. This code is initially derived
    * from the configuration setting TDBS_LANG as specified in the configuration
    * file (tdbs.conf) or in the current process' environment.
    *
    * To modify the language used by the CCL parser, use the
    * TdbShellModifyCcl() function.
    *
    * #### Parameter Values
    * The \p language parameter is passed a character string to receive the
    * code for the currently selected dialect of the CCL language.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbShellModifyCcl()
    */
   int TdbLanguage(char* language);


   /**
    * @brief Returns information about the the TRIP license
    * @param mode Type of information to retrieve.
    * @param strarg Mode-specific extra input parameter
    * @param stringValue Output parameter for information of string type.
    * @param strvalsz Allocated size of the stringValue buffer.
    * @param intValue Output parameter for information of integer type.
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The TdbLicenseInfo() function provides a way to obtain various
    * information about a TRIP session.
    *
    * #### Parameter Values
    * The \p mode parameter is used to tell the function what information to
    * return and must be set to one of the following values:
    *
    * | Value | Output Parameter | Description |
    * | ----- | ---------------- | ----------- |
    * | LICINFO_PRODUCT | intValue | Returns a boolean value (0/1) indicating if the installed license includes the product specified by the \p strarg parameter. |
    * | LICINFO_PRODUCT_SYSTEM | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPsystem product. |
    * | LICINFO_PRODUCT_HIGHWAY | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPhighway product. |
    * | LICINFO_PRODUCT_NXP | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPnxp product. |
    * | LICINFO_PRODUCT_COF | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPcof product. |
    * | LICINFO_PRODUCT_XML | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPxml product. |
    * | LICINFO_PRODUCT_JXP | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPjxp product. |
    * | LICINFO_PRODUCT_SQL | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPsql product. |
    * | LICINFO_PRODUCT_CLIENT | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPclient product. |
    * | LICINFO_PRODUCT_JTK | intValue | Returns a boolean value (0/1) indicating if the installed license includes the TRIPjtk product. |
    * | LICINFO_EXPIRATION_DATE | intValue and/or stringValue | Returns the expiration date as string in YYYY-MM-DD format, and/or as an integer in YYYYMMDD format. |
    * | LICINFO_MAX_CONCURRENT_USERS | intValue | Returns the maxium concurrent users limit. |
    * | LICINFO_MAX_REGISTERED_USERS | intValue | Returns the maxium registered users limit. |
    * | LICINFO_MAX_OPEN_DATABASES | intValue | Returns the maxium open databases limit. |
    * | LICINFO_MAX_CLUSTER_SIZE | intValue | Returns the maxium cluster size limit. |
    * | LICINFO_MAX_SEARCH_SETS | intValue | Returns the maxium search set size limit. |
    * | LICINFO_MAX_DBRECORDS | intValue | Returns the maxium allowed number of records in a database. |
    * | LICINFO_INCLUDES_GRAPH_SEARCH | intValue | Returns a boolean value (0/1) indicating if graph search is enabled. |
    * | LICINFO_INCLUDES_XPATH_SEARCH | intValue | Returns a boolean value (0/1) indicating if XPath search is enabled. |
   *
    * #### Since
    * This function was introduced in TRIP version 8.0-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 8866 | Missing argument | A required argument was not supplied to the function. |
    * | 18466 | Unrecognised option: value. | This error occurs when the mode parameter has an unsupported value. |
    *
    * @see TdbSessionInfo()
    */
   int TdbLicenseInfo(int mode, const char* strarg,
      char* stringValue, size_t strvalsz, int* intValue);


   /**
    * @brief Returns information about the current session.
    * @param mode Type of information to retrieve.
    * @param stringValue Output parameter for information of string type.
    * @param intValue Output parameter for information of integer type.
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The TdbSessionInfo() function provides a way to obtain various
    * information about a TRIP session.
    *
    * #### Parameter Values
    * The \p mode parameter is used to tell the function what information to
    * return and must be set to one of the following values:
    *
    * | Value | Output Parameter | Description |
    * | ----- | ---------------- | ----------- |
    * | SESSIONINFO_RUNTIME | intValue | Returns a boolean value (0/1) indicating if the installed license is a runtime license. |
    * | SESSIONINFO_DATEFORM | intValue | Return the number of the current TRIP date form. |
    * | SESSIONINFO_DATEFORM | stringValue | The date separator characters used with the current TRIP date form. |
    * | SESSIONINFO_OPENBASES | intValue | Return a count of the present number of open databases. |
    * | SESSIONINFO_CURRENTBASE | stringValue | Returns the name of the current database. |
    * | SESSIONINFO_CLUSTER | intValue, stringValue | Returns a list of cluster members. Call with the stringValue argument set to NULL to get the required minimum size of the stringValue buffer in the intValue parameter as output. Requirement: the current open database must be a cluster. |
    * | SESSIONINFO_DBS_IN_SEARCH | stringValue | Returns a comma-separated list of databases in a specified search set. The search number is specified in the intValue parameter. NOTE: this mode is only supported in the server-side version of this API. |
    * | SESSIONINFO_SIF_NAME | stringValue | Returns the name of SIF file currently in use. |
    * | SESSIONINFO_CHARSET | intValue | Returns the number of the character set in use by the session. This can be one of the values: CHSET_LA1CHSET_LA2CHSET_LA3CHSET_UTF8CHSET_CHICHSET_GBKCHSET_EUCCHSET_SJIS |
    * | SESSIONINFO_SESSION_STATUS | intValue | Return one of the following values: 0 - session is not started, 1 - kernel is initialized, but user is not logged in, 2 - session is active and user is logged in, 3 - session is inactive, user has logged out, 4 - user has been temporarily logged out w/o any saved session state ( see TdbChangeUser() ), 5 - user has been temporarily logged out with session state intact ( see TdbChangeUser() ). |
    * | SESSIONINFO_CJK_MODE | intValue | Return the pictogram character set in use: 0 - not in use, 1 - GB 2312 80, 2 - Apple Chinese, 3 - EUC, 4 - Shift/JIS, 5 = GBK |
    * | SESSIONINFO_SUPERMAN | intValue | Non-zero if the current user is SYSTEM and has SUPERMAN rights. |
    * | SESSIONINFO_COMMERCIAL | intValue | Non-zero if a commercial license is installed. |
    * | SESSIONINFO_DBS_INITIATED | intValue | Returns the number of databases opened during the current session. |
    * | SESSIONINFO_DB_IN_SEARCH | stringValue | Returns the name of the database or cluster assosicated with a particular search set. Specify the search set number in the intValue parameter as input. |
    * | SESSIONINFO_HIGHEST_SEARCH | intValue | Returns the highest search set number in use in the session. |
    * | SESSIONINFO_VERSION | stringValue | Returns the version information as a string in the format MAJOR.MINOR-SERVICE[:PATCH] |
    * | SESSIONINFO_VERSION_MAJOR | intValue | Returns the major version number. |
    * | SESSIONINFO_VERSION_MINOR | intValue | Returns the minor version number. |
    * | SESSIONINFO_VERSION_SERVICE | intValue | Returns the service release number. |
    * | SESSIONINFO_VERSION_PATCH | intValue | Returns the patch number. |
    * | SESSIONINFO_CURRENTTHES | stringValue | Returns the name of the currently defined thesaurus. |
    * | SESSIONINFO_THESDEF | stringValue | Returns the definition used to declare the current thesaurus. |
    * | SESSIONINFO_AUTOLIC | intValue | One or more licenses have been automatically installed. Specific to container deployments. |
	*
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 18466 | Unrecognised option: value. | This error occurs when the mode parameter has an unsupported value. |
    *
    * @see TdbStartTrip()
    * @see TdbLicenseInfo
    * @see TdbShellDefDateForm()
    * @see TdbChangeUser()
    */
   int TdbSessionInfo(int mode, char* stringValue, int* intValue);

   /**
    * @brief Enumerate currently defined views
    * @param enum_cursor Controls the enumeration; set to zero for first call
    * @param view_name Buffer to receive the name of a view
    * @param name_size Allocated size of the view_name buffer
    * @param view_def Pointer to a char pointer that receives a string holding the view definition
    * @param def_size Receives the length of the string returned via view_def
    * @return 1 on success, 0 when no more views exist, or a positive even value on error
    * @ingroup infocommands
    * 
    * #### Description
    * Call this function in a loop with the \p enum_cursor parameter set to zero
    * for the first call, and continue looping while the function returns an
    * odd value, typically 1.
    * 
    * The \p view_def parameter is optional. If defined, it will receive a pointer
    * to a character string that contains a comma-separated list the field names that
    * the view is comprised of. The received pointer must be deallocated before the
    * using the function TdbDeallocate(). If the view is based on a field group in
    * a database currently open for search, the field names in the \p view_def 
    * parameter will be prefixed by the database name (e.g. "ALICE.TXT,ALICE.TXT2" 
    * instead of just "TXT,TXT2"). If the view is defined using the DEFINE VIEW 
    * command, the field names will not have database name prefixes. Also note that
    * using a database name as a field name prefix is not valid in the DEFINE VIEW
    * command.
    * 
    * @see TdbDeallocate()
    */
   int TdbEnumViews(int64_t* enum_cursor, char* view_name, size_t name_size,
                    char** view_def, size_t* def_size);

   /**
    * @brief Check or retrieve CCL keywords
    * @param symbol Symbol string
    * @param symbol_length Length of symbol
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The function TdbShellSymbol() can be used to either check the validity,
    * or to retrieve the correct form of a CCL command/modifier in the current
    * language.
    *
    * To get the correct CCL form of a DAO symbol, put the DAO symbol for the
    * command in the \p symbol argument and pad the rest of the string with
    * blank space up to 16 bytes. On return from the function, the argument
    *  will contain the corresponding CCL symbol. The actual CCL and DAO
    * symbols can be found in the file ENGLISH.CCL (for English language) on
    * the directory referred to by the environment variable or configuration
    * setting TDBS_SYS.
    *
    * The DAO symbol can be entered in different ways in order to get different
    * forms of the CCL symbol. The case of the three DAO characters determine
    * how the CCL symbol will be returned. Assuming the current language to be
    * English, the following forms of the DAO symbol for Find would yield these
    * forms of the CCL command:
    *
    *  + \c fin [rarr  ]
    *  + \c f (the shortest possible form of the CCL command in lower case)
    *  + \c fiN [rarr  ]
    *  + \c F (the shortest form in upper case)
    *  + \c fIn or \c fIN [rarr  ]
    *  + \c find (the full command in lower case)
    *  + \c Fin or \c FiN [rarr  ]
    *  + \c Find (the full command with the shortest form capitalized)
    *  + \c FIn or \c >FIN [rarr  ]
    *  + \c FIND (the full command in upper case).
    *
    * To check a CCL symbol, put the three character long DAO form first in
    * the symbol string, a blank space in position four, and then the CCL
    * symbol. The return code from the function indicates success or failure,
    * and for a success, the CCL symbol will also be returned in the string
    * as described above.
    *
    * The routine can also be used to access the current definition of the
    * mask characters or the comma separator. To check all mask characters,
    * put the string '#:![amp   ]' in the symbol argument. To check one single
    * mask character, put that mask character in the first position of the
    * string.
    *
    * The 16-character wide symbol string must always be padded with blank
    * space.
    *
    * #### Parameter Values
    * The \p symbol parameter is passed a character string as the CCL symbol
    * to verify. The symbol buffer must be at least 16 characters long, not
    * including any terminating null character.
    *
    * The \p symbol_length parameter is passed an integer as the length of
    * the symbol string.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 | Operation failed | The specified symbol does not exist or is not valid. |
    */
   int TdbShellSymbol(char* symbol, int* symbol_length);

  
   /**
    * @brief This routine returns information about the currently logged on user.
    * @param mode Type of information to retrieve.
    * @param stringValue Output parameter for information of string type.
    * @param intValue Output parameter for information of integer type.
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The TdbUserInfo() function provides a way to obtain various information
    * about the currently logged on user.
    *
    * #### Parameter Values
    * The \p mode parameter is used to tell the function what information to
    * return and must be set to one of the following values:
    *
    * | Value | Output Parameter | Description |
    * | ----- | ---------------- | ----------- |
    * | USERINFO_FM | intValue | Returns 1 if the user has file manager rights and 0 otherwise. |
    * | USERINFO_UM | intValue | Returns 1 if the user has user manager rights and 0 otherwise. |
    * | USERINFO_SM | intValue | Returns 1 if the user has system manager rights and 0 otherwise. |
    * | USERINFO_STARTPROC | stringValue | Returns the name of the start procedure for this user. |
    * | USERINFO_NAME | stringValue | Returns the name of the user. |
    * | USERINFO_EXT_NAME | stringValue | Returns the external OS or LDAP name of the user. |
    * | USERINFO_GROUPS | intValue, stringValue | Returns the list of TRIP groups for the user. The list is delimited by the pipe character. Pass NULL for the stringValue parameter, and the intValue will on return contain the minimum allocated size of the buffer to pass into stringValue. If stringValue and intValue are both non-NULL, the intValue parameter will on return contain the number of groups. |
    * | USERINFO_EXT_GROUPS | intValue, stringValue | Returns the list of external OS or LDAP groups for the user. The list is delimited by the pipe character. Pass NULL for the stringValue parameter, and the intValue will on return contain the minimum allocated size of the buffer to pass into stringValue. If stringValue and intValue are both non-NULL, the intValue parameter will on return contain the number of groups. |
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 18466 | Unrecognised option: (value}. | This error occurs when the mode parameter has an unsupported value. |
    *
    * @see TdbStartTrip()
    */
   int TdbUserInfo(int mode, char* stringValue, int* intValue);

   
   /**
    * @brief Return the current TRIP version
    * @param version TRIP version string
    * @param version_length Length of version string
    * @return 1 or other odd value on success, even value on error
    * @ingroup infocommands
    *
    * #### Description
    * The function TdbVersion() returns the TRIPsystem version identifier for
    * the current TRIP session.
    *
    * The returned version string is specified in the form \c M.m-s[:p] where:
    *   + \c M is the major version number
    *   + \c m is the minor version number
    *   + \c s is the service release version number
    *   + \c p is the optional patch version number
    *
    * Examples:
    *  + 7.2-0 for the first release of version 7.2
    *  + 7.2-1 for the first service patch release for version 7.2
    *  + 7.2-1:1 for the first patch to version 7.2-1
    *
    * #### Parameter Values
    * The \p version parameter is passed a character string to receive the
    * version identifier for the current TRIPkernel installation.
    *
    * The \p version length parameter is passed a pointer to an integer that
    * receives the count of the number of characters in version string.
    *
    * #### Return Codes
    * This function always succeeds.
    */
   int TdbVersion(char* version, int* version_length);

   
   // ------------------------------------------------------------------------

   /**
    * @defgroup control TDBS Control
    * @brief Control functions
    *
    * The TDBS control functions provides access to ASEs, filters, etc.
    */


   /**
    * @brief Get the return code from the most recently called ASE routine
    * @ingroup control
    *
    * #### Description
    * The TdbAseRetCode() function returns the result value of the most
    * recently called ASE routine. Note that this value is distinct from that
    * signalled by TdbCallAse() which denotes the success of actually calling
    * the routine.
    *
    * #### Privileges Required
    * If the called ASE itself calls privileged functions then the calling
    * user must possess those privileges for correct operation.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | The ASE failed to complete successfully. |
    *
    * The return values 1 (success) and 0 (failure) may be combined with
    * boolean indicators to affect the calling application. Typically these
    * indicators are used with data entry forms in the TRIP Classic
    * environment. Valid values are:
    *
    * | Bit number | Purpose (when set) |
    * | ---------- | ------------------ |
    * | Bit 1 | Refresh screen upon return from the ASE |
    * | Bit 2 | Continue with the previous keyboard action |
    * | Bit 4 | Flush any pending messages |
    * | Bit 5 | Keep the data entry cursor in the last field used |
    * | Bit 6 | Do not allow any data entry to fields on the current page |
    *
    * @see TdbCallAse()
    * @see TdbLoadAse()
    * @see TdbRetCode()
    */
   int TdbAseRetCode(void);


   /**
    * @brief Allow an application to detect a user-initiated interrupt signal
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The TdbBreak() function returns a value which signifies whether the
    * user has caused an interrupt, for example by hitting \c CTRL-C in a host
    * application. The TRIPkernel itself handles interrupts during searching
    * and sorting, but if the user application is involved in a lengthy
    * process, this function should be used within code loops to allow that
    * process to abort cleanly.
    *
    * The control character is set up by TRIPclassic to be \c CTRL-C, but this
    * can be altered by a call to the TdbSetBreakChars() function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 |  | No interrupt signal detected |
    * | 1 |  | User interrupt signal detected |
    *
    * @see TdbSetBreakChars()
    */
   int TdbBreak(void);
   
   
   /**
    * @brief Call an ASE routine
    * @param aseID Identifier of an ASE function.
    * @param argstr The argument passed to/from the ASE.
    * @param arglen Length of the argument string.
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The TdbCallAse() function calls an ASE routine that has previously been
    * identified and loaded by the TdbLoadAse() function. All ASE routines
    * expect two arguments, a string and the string's length, which are
    * specified by \p argstr and \p arglen. The argstr argument must reference
    * a buffer of at least \c 256 bytes.
    *
    * The TdbLoadAse() function has to be called before TdbCallAse().
    *
    * #### Parameter Values
    * The \p aseID parameter is an operating system specific handle as
    * returned by the TdbLoadAse() function.
    *
    * The \p argstr parameter is a string passed as the first argument of the
    * ASE routine. As this argument can be read and written by the ASE routine
    * care should be exercised when calling this function.
    *
    * The \p arglen parameter is the second argument to the ASE routine and
    * contains an integer value as the length of the argument string being
    * passed.
    *
    * #### Return Codes
    * The return codes in the table below are what may be returned by the
    * kernel routine calling the ASE. If the ASE has successfully been called,
    * the return code from this function is implementation-specific to the ASE.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 10402 | ASE routine {1} not found (OS status: {2}). |  |
    *
    * @see TdbAseRetCode()
    * @see TdbLoadAse()
    */
   int TdbCallAse(void* aseID, char* argstr, int* arglen);


   /**
    * @brief Calls a filter ASE for export of data from a TRIP database.
    * @param fd Filter ASE argument structure
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * Calls a user defined filter ASE function to export data from a TRIPsystem
    * database. The FilterData structure indentifies both the filter to call
    * and the data to be exported. Although a filter is similar to an ASE, the
    * TdbLoadAse() function does not have to be called.
    *
    * Filter functions are like ASEs, but takes a wider range of in- and output
    * parameter which allows for a greater range of usage. 
    *
    * #### Parameter Values
    * The \p fd parameter is passed a pointer to a FilterData structure that
    * identifies the ASE function that will called and other relevant
    * information. The fields \c buffer and \c filter_arguments in the
    * FilterData structure may on output be allocated by the filter ASE
    * routine. The caller is responsible for releasing this memory; take
    * care to use the appropriate function for this (normally free(), but
    * is on Windows likely to be HeapFree instead). Refer to the documentation
    * of the ASE you are calling for information on which to use.
    *
    * #### Return Codes
    * The return codes in the table below are what may be returned by the
    * kernel routine calling the filter. If the filter has successfully been
    * called, the return code from this function is implementation-specific to
    * the filter.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 10402 | ASE routine {1} not found (OS status: {2}). |  |
    *
    * @see TdbImport()
    */
   int TdbExport(FilterData* fd);

   
   /**
    * @brief Calls a filter for import of data into a TRIP database.
    * @param fd Filter ASE argument structure
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The function TdbImport() calls a user defined filter ASE function to
    * import data into a TRIPsystem database. The FilterData structure defines
    * both the filter to call and the data to be imported.
    *
    * Filter functions are like ASEs, but takes a wider range of in- and output
    * parameter which allows for a greater range of usage.
    *
    * #### Parameter Values
    * The \p fd parameter is passed a pointer to a FilterData structure that
    * identifies the ASE function that will called and other relevant
    * information. In contrast to the TdbExport() function, the fields in 
    * the TdbImport structure are expected to not be allocated by the ASE, but
    * by the caller before making the call. An import ASE is not expected to
    * reassign any fields in the FilterData structure.
    *
    * #### Return Codes
    * The return codes in the table below are what may be returned by the
    * kernel routine calling the filter. If the filter has successfully been
    * called, the return code from this function is implementation-specific to
    * the filter.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 10402 | ASE routine {1} not found (OS status: {2}). |  |
    *
    * @see TdbExport()
    */
   int TdbImport(FilterData* fd);

   
   /**
    * @brief Load an ASE routine
    * @param ase_name Name of ASE
    * @param ase_id ASE handle
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The function TdbLoadAse() checks that the ASE routine exists. The
    * routine returns the address to be used in TdbCallAse().
    *
    * The specifics of calling the ASE using the returned address are system
    * dependent.
    *
    * #### Parameter Values
    * The \p ase_name parameter is passed a character string as the ASE name.
    *
    * The \p ase_id parameter receives the ASE identity. The variable to which
    * the \p ase_id parameter points must be set to NULL upon entry to the
    * routine. The value returned is the run-time address of the ASE function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 10402 | ASE routine {1} not found (OS status: {2]). |  |
    *
    * @see TdbCallAse()
    */
   int TdbLoadAse(char* ase_name, void** ase_id);

   
   /**
    * @brief Reset the TRIP session status
    * @param mode Mode of status restoration
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The function TdbLoadStatus() will reset the search status of the current
    * TRIP  session to that current when a previous call to TdbSaveStatus() was
    * made. The restoration can be done to various degrees. If mode one is
    * specified, the database or database cluster which was open at the time
    * TdbSaveStatus() was called will be reopened. If mode two is specified,
    * all search sets created after the call to TdbSaveStatus() will be
    * deleted, but the currently open database will remain open so that
    * subsequent searches are still performed on the same database. For a
    * complete restoration of the status, the mode must be set to three,
    * causing both of the above actions to be taken.
    * 
    * This function must only be called if a call to TdbSaveStatus() has
    * previously been made that has not yet been matched by a TdbLoadStatus()
    * call.
    *
    * #### Parameter Values
    * The \p mode parameter is passed an ineger as the indication of how
    * previous search status is restored. Valid values are:
    *
    * | Constant Value | Purpose |
    * | -------------- | ------- |
    * | 1 | Reopen the databases that were open when the function SAVE STATUS was last called. |
    * | 2 | Delete all search sets created since the last call to the function SAVE STATUS, but leave the current databases open. |
    * | 3 | Delete all search sets created since the last call to the function SAVE STATUS, and reopen the databases that were open immediately prior to that call. |
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * @see TdbSaveStatus()
    */
   int TdbLoadStatus(int mode);


   /**
    * @brief Save the current TRIP session status
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The function TdbSaveStatus() will save the current search status of the
    * TRIP session, defined by a) the last search made, and b) the set of
    * currently opened databases. 
    *
    * Each call to TdbSaveStatus() must always be matched by a call to the
    * TdbLoadStatus() function. If an application has called TdbSaveStatus()
    * three times in a row, then it must also call TdbLoadStatus() three 
    * times.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbLoadStatus()
    */
   int TdbSaveStatus(void);

   
   /**
    * @brief Set the Command Break Characters
    * @param mask Break character mask
    * @param routine User-written procedure
    * @param mode Procedure AST mode
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The function TdbSetBreakChars() allows the TRIP user running a Linux or
    * UNIX server-side console application to define one or more control
    * characters that can be used to interrupt the processing of a CCL command.
    * The characters are indicated by the bits set in the \p mask parameter
    * which is an integer where bit 1 corresponds to CTRL-A , bit 2 to CTRL-B,
    * etc. On receiving the break character, the user-supplied routine defined
    * by the \p routine parameter is called.
    *
    * If TRIP Classic is used in conjunction with other software that also
    * intercepts control characters from the terminal, the mode argument to
    * TdbSetBreakChars() should be set to 1.
    *
    * #### Parameter Values
    * The \p mask parameter is passed an \c int as a bit-vector for the break
    * character mask. The \p break characters argument consists of a bit mask
    * for the break characters to be set; \c bit \c 1 corresponds to CTRL-A,
    * \c bit \c 2 to CTRL-B , etc.
    *
    * The \p routine parameter is passed the address of a user-written procedure. 
    * The \p routine argument is the entry mask for the user-supplied procedure 
    * invoked whenever one of the break characters set in the \p mask is received.
    *
    * The \p mode parameter is passed an \c int as the AST mode for the
    * user-written procedure.
    *
    * This function does not have any effect on Windows.
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * @see TdbBreak()
    * @see TdbSetBreakFlag()
    */
   int TdbSetBreakChars(int mask, void(*routine)(int), int mode);


   /**
    * @brief Tells TRIPsystem to abort an ongoing search.
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The function TdbSetBreakFlag() can be used to tell TRIPsystem to abort
    * an ongoing search.
    *
    * #### Return Codes
    * This function always succeeds.
    *
    * @see TdbBreak()
    * @see TdbSetComforterInterval()
    */
   int TdbSetBreakFlag(void);

   
   /**
    * @brief Set the Comforter Interval
    * @param seconds Number of seconds between calls
    * @param proc User-written procedure
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The function TdbSetComforterInterval() is used to specify the number of
    * seconds that are allowed to elapse between call to the user-supplied
    * routine defined by the \p proc parameter.
    *
    * #### Parameter Values
    * The \p seconds parameter is passed an \c int as the number of seconds
    * between successive calls to the user-supplied comforter routine.
    *
    * The \p proc parameter is passed a the address of a user-written procedure. 
    * The \p proc argument is the entry mask for the user-supplied procedure 
    * invoked after each period of elapsed seconds supplied in the \p seconds
    * parameter. When called, the user-written procedure will receive one argument 
    * which specifies the current type of activity (1 = searching, 2 = sorting).
    *
    * #### Since
    * This function was introduced in TRIP version 3.4-0.
    *
    * #### Return Codes
    * This function always succeeds.
    */
   int TdbSetComforterInterval(int seconds, void(*proc)(int));
   
   
   /**
    * @brief Normalizes a string
    * @param string Character string
    * @param length Length of string
    * @param mode Normalization flag
    * @return 1 or other odd value on success, even value on error
    * @ingroup control
    *
    * #### Description
    * The TdbStrNormalize() function is used to normalize a text string
    * according to one of TRIP's normalization schemes.
    *
    * #### Parameter Values
    * The \p string parameter is passed a character string to normalize. The
    * string is normalized in place.
    *
    * The \p length parameter is passed an integer as the count of the number
    * of characters in the string.
    *
    * The \p mode parameter is passed a value to control the normalization.
    * Valid values are:
    *
    * | Constant Value | Purpose |
    * | -------------- | ------- |
    * | 1 | To uppercase without compression of blank equivs. |
    * | 2 | To uppercase with compression of blank equivs. |
    * | 3 | CONTROL normalization |
    * | 4 | Current database(s) normalization |
    * | 5 | Translation according to printer table |
    * | 6 | Translation according to local printer table |
    * | 7 | Translation for sorting |
    *
    * #### Return Codes
    * This routine returns 1 on success and 0 on failure (mode value out of
    * range).
    */
   int TdbStrNormalize(char* string, int* length, int mode);


   // ------------------------------------------------------------------------

   /**
    * @defgroup graphdb Graph Search and Management
    * @brief Functions for use of graph datbases
    *
    * The graph functions provides access to manipulation and querying of
    * TRIP graph databases.
    */


   /**
    * @brief Locate the direct relationships between two sets of records.
    * @param resultSet The resulting search set, consisting of a set of edges
    *        from the graph database
    * @param sourceSet The set of vertex records that the edges to locate have
    *        as origin. Pass 0 (zero) to indicate that all vertices are to be
    *        included in the search. This set can be from any database, graph
    *        or otherwise.
    * @param edgeSet A set of edge records from the current graph database,
    *        indicating the sub-graph to query. Pass 0 (zero) to use all edges
    *        in the graph database.
    * @param targetSet The set of vertex records that the edges to locate have
    *       as destination. Pass 0 (zero) to indicate that all vertices are to
    *       be included in the search. This set can be from any database, graph
    *       or otherwise.
    * @param exclusionSet A set of edge records to explicitly not include in
    *        the search result. Pass 0 (zero) to indicate that no exclusion set
    *        is to be used.
    * @param condition An extra CCL condition to apply to the graph edges. Must
    *        be specified without the command keyword (e.g. FIND). If no extra
    *        condition is needed, pass an empty string.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * The TdbGraphFind() function queries a graph database for direct
    * (non-transitive) relationships between any two records. The database
    * currently open for search must be a graph database.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * The number of steps this function performs is such that a significant
    * number of return codes may be expected. They are therefore not listed
    * here. Please refer to the error codes list in this reference manual for
    * explanations to any code returned.
    *
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphFind(int* resultSet, int sourceSet, int edgeSet, int targetSet,
                    int exclusionSet, const char* condition);


   /**
    * @brief Create a search set of the vertices associated by the edges in a
    *        graph search set.
    * @param resultSet The resulting search set, consisting of a set of vertex
    *       records.
    * @param mode Indicates what to resolve. Pass either GRAPH_RESOLVE_SOURCE
    *       (1) to resolve the origin records of the edges, or
    *       GRAPH_RESOLVE_TARGET (2) to resolve the destination records of
    *       the edges.
    * @param edgeSet A search set with edge records from a graph database.
    *        The vertex records to resolve are the ones that these edges
    *        associate.
    * @param database Restrict resolution to vertex records in this database
    *        only. Pass an empty string to resolve all records regardless of
    *        database.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * The TdbGraphResolve() resolves an edge set ( as produced by a function
    * such as TdbGraphFind() ) into a set of vertex records. The resulting set
    * is formed from either the origins of the edges or the destinations of the
    * edges.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * The number of steps this function performs is such that a significant
    * number of return codes may be expected. They are therefore not listed
    * here. Please refer to the error codes list in this reference manual for
    * explanations to any code returned.
    *
    * @see TdbGraphFind()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphResolve(int* resultSet, int mode, int edgeSet,
                       const char* database);
   

   /**
    * @brief Perform a transitive search in a graph datanase.
    * @param resultSet The resulting search set, consisting of a set of edges
    *        from the graph database.
    * @param maxDepth The maximum traversal depth to which to perform the
    *        search.
    * @param reverse Pass true to perform the search in the opposite to the
    *        defined direction of the edges. Pass false to follow edges in
    *        their defined direction.
    * @param allEdges Pass true return a set of all traversed edges, and false
    *        to return a set that only consists of the most distant edges
    *        found.
    * @param edgeSet The edge set to query. Pass 0 (zero) to use all all the
    *        edges in the current graph database.
    * @param sourceSet The set of vertex records at which the transitive
    *        search is to start.
    * @param targetSet The set of vertex records to which to limit found edges
    *        to. If specified, only edges that have these vertices as
    *        destination (if \p reverse is false) or as origin (if \p reverse
    *        is true) will be considered. Note that this set restricts all the
    *        vertices processed by the transitive search - not only the
    *        vertices at the very end of the search. Pass 0 (zero) to use all
    *        possible vertices.
    * @param condition An extra CCL condition to apply to the graph edges. Must
    *        be specified without the command keyword (e.g. FIND). If no extra
    *        condition is needed, pass an empty string.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * The TdbGraphTransitive() function queries a graph database for
    * relationships such that zero or more vertices may have to be traversed
    * to get from origin to destination. The database currently open for search
    * must be a graph database.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * The number of steps this function performs is such that a significant
    * number of return codes may be expected. They are therefore not listed
    * here. Please refer to the error codes list in this reference manual for
    * explanations to any code returned.
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphTransitive(int* resultSet, int maxDepth, boolean reverse,
                          boolean allEdges, int edgeSet, int sourceSet,
                          int targetSet, const char* condition);
   
   
   /**
    * @brief Initialize a graph path analysis operation.
    * @param context Output parameter. Receives a handle to the initialized
    *        analysis.
    * @param algorithm Determine how the path analysis is to be performed on a
    *        general level. Valid values are GRAPH_PATH_ALL (1) to find all
    *        possible routes, and GRAPH_PATH_SHORTEST (2) that locates the
    *        shortest (or least expensive) route.
    * @param maxDepth Determine how far away from the vertices in the origin
    *        set that the analysis is allowed to traverse.
    * @param originSet A set of vertex records that are to be used as starting
    *        points for the analysis.
    * @param edgeSet The edge set to query. Pass 0 (zero) to use all all the
    *        edges in the current graph database.
    * @param destinationSet A set of vertex records that are to be used as
    *        destination points for the analysis.
    * @param reverse Pass false to perform the analysis in the defined
    *        direction of the edges, or true to perform the analysis in reverse
    *        (from destination to origin).
    * @param condition An extra CCL condition to apply to the graph edges. Must
    *        be specified without the command keyword (e.g. FIND). If no extra
    *        condition is needed, pass an empty string.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * The TdbGraphStartPathAnalysis() function prepares the TRIP kernel for a
    * graph path analysis operation. Use this function to find any path or the
    * shortest path between two sets of vertices. Execute the analysis by
    * calling TdbGraphPathAnalysisStep() until it reports that the analysis is
    * complete. To examine intermediary states during analysis, call the
    * TdbGraphAnalysisSnapshot() function. To examine the results of a
    * completed analysis, use the TdbGraphNextPath() and TdbGraphGetPath()
    * functions. When all done, release the analysis context by calling the
    * TdbGraphClosePathAnalysis() function.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * The number of steps this function performs is such that a significant
    * number of return codes may be expected. They are therefore not listed
    * here. Please refer to the error codes list in this reference manual for
    * explanations to any code returned.
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphStartPathAnalysis(int** context, int algorithm, int maxDepth,
                                 int originSet, int edgeSet,
                                 int destinationSet, boolean reverse,
                                 const char* condition);

   
   /**
    * @brief Perform one iteration of a graph path analysis operation.
    * @param context Context handle referring to the graph path analysis for
    *        which to execute another iteration.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * This function executes a single iteration of a graph path analysis. An
    * iteration involves finding edges level "down" into the graph and evaluate
    * the paths available so far. Execute this function repeatedly while
    * returns 1 or any other success code. The error code 226 (no hits) is
    * returned when the analysis is completed.
    *
    * The analysis is initialized by a call to the TdbGraphStartPathAnalysis()
    * function. Use this function to find any path or the shortest path between
    * two sets of vertices. Execute the analysis by calling
    * TdbGraphPathAnalysisStep() until it reports that the analysis is
    * complete. To examine intermediary states during analysis, call the
    * TdbGraphAnalysisSnapshot() function. To examine the results of a
    * completed analysis, use the TdbGraphNextPath() and TdbGraphGetPath()
    * functions. When all done, release the analysis context by calling the
    * TdbGraphClosePathAnalysis() function.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * The number of steps this function performs is such that a significant
    * number of return codes may be expected. They are therefore not listed
    * here. Please refer to the error codes list in this reference manual for
    * explanations to any code returned.
    *
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 226 | No hits! | The graph analysis is completed. |
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphPathAnalysisStep(int* context);

   
   /**
    * @brief Terminate a graph path analysis operation and release any resources
    *        allocated to it.
    * @param context Context handle referring to the graph path analysis to close.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * This function terminates a graph path analysis operation as initialized
    * by TdbGraphStartPathAnalysis() function. Failure to call this function
    * after a completed path analysis will cause memory leaks.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * The number of steps this function performs is such that a significant
    * number of return codes may be expected. They are therefore not listed
    * here. Please refer to the error codes list in this reference manual for
    * explanations to any code returned.
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphClosePathAnalysis(int** context);


   /**
    * @brief Returns path count statistics from an on-going analysis.
    * @param context Context handle referring to an on-going graph path analysis.
    * @param count Output parameter. Receives the number of paths found so far.
    * @param mode Pass non-zero to retrieve the count of candidate paths and
    *        zero to retrieve the count of completed paths.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * This function reports the current analysis state in terms of counts of
    * candidate paths (for non-zero mode) and completed paths (if mode is set
    * to zero).
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphAnalysisSnapshot(int* context, int* count, int mode);
   

   /**
    * @brief Advances to the next path from a completed analysis result.
    * @param context Context handle referring to a completed graph path analysis.
    * @param database Output parameter. Receives a pointer to the name of the
    *        database in which the start vertex of the path is stored.
    * @param recordId Output parameter. Receives the record ID number for the
    *        start vertex of the path.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * This function advances the context to report the next path from a
    * completed analysis result and returns information on the first vertex of
    * the path. To read data for the rest of the path, use the
    * TdbGraphGetPath() function.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | No more paths are available in the analysis context. |
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphNextPath(int* context, char** database, int* recordId);


   /**
    * @brief Returns information on a path found by a graph analysis operation.
    * @param context Context handle referring to a completed graph path analysis.
    * @param edge Output parameter. Receives a pointer to the name of the next
    *        edge in the path.
    * @param weight Output parameter. Receives the weight value for the edge.
    * @param database Output parameter. Receives a pointer to the name of the
    *        database in which the current edge's destination vertex is stored.
    * @param recordId Output parameter. Receives the record ID number for the
    *        current edge's destination vertex.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * This function reports info on an edge and its destination vertex from a
    * path and advances the location one step. This function is called
    * repatedly until all the info on the current path has been returned. A
    * call to the TdbGraphNextPath() function is required to advance to the
    * next path.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | No more data is available on the current path. |
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphRecAssoc()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphGetPath(int* context, char** edge, double* weight,
                       char** database, int* recordId);
   
   
   /**
    * @brief Create a graph edge between two existing TRIP records.
    * @param recordControl [IN/OUT] Supply a record control set with write
    *        permissions to the graph database in which to create the edge, or
    *        pass a pointer to a NULL record control in order to use the graph
    *        database currently open for search. The parameter may also be set
    *        to NULL (in contrast to a pointer to a variable holding NULL) if
    *        the edge is to be committed by this function and the record control
    *        is not required as output. On output, this parameter receives the
    *        record control was supplied or (in case a pointer to a NULL
    *        variable was supplied) created by this function. If the parameter
    *        was set to NULL on input, nothing is returned. In case a record
    *        control is returned, the caller is responsible for its release.
    * @param recordName [IN] Pass the record name to assign the edge record to
    *        create, or an empty string in order to automatically generate a
    *        unique ID if a record name is not already defined.
    * @param edgeName [IN] Name of the edge to create
    * @param sourceRecord [IN] Record control referring to the record to use
    *        as source vertex
    * @param targetRecord [IN] Record control referring to the record to use
    *        as target vertex
    * @param weight [IN] Receives the weight value for the edge.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * This function creates a graph edge record that associates two existing
    * TRIP records. Unless the recordControl parameter is set to NULL,
    * the caller will be responsible for committing the record using
    * TdbPutRecord() and for the release of the record control using
    * TdbDeleteRecordControl().
    *
    * #### Privileges Required
    * The current TRIPsystem user needs write access to the graph database in
    * which the edge record is to be created.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    *
    * #### Return Codes
    * The number of steps this function performs is such that a significant
    * number of return codes may be expected. They are therefore not listed
    * here. Please refer to the error codes list in this reference manual for
    * explanations to any code returned.
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphAddVertex()
    */
   int TdbGraphRecAssoc(TdbHandle* recordControl, const char* recordName,
                        const char* edgeName, TdbHandle sourceRecord,
                        TdbHandle targetRecord, double weight);

   
   /**
    * @brief Create a simple in-graph vertex record.
    * @param recordControl [IN/OUT] Supply a record control set with write
    *        permissions to the graph database in which to create the vertex,
    *        or pass a pointer to a NULL record control in order to use the
    *        graph database currently open for search. The parameter may also
    *        be set to NULL (in contrast to a pointer to a variable holding
    *        NULL) if the vertex is to be committed by this function and the
    *        record control is not required as output. On output, this
    *        parameter receives the record control was supplied or (in case a
    *        pointer to a NULL variable was supplied) created by this function.
    *        If the parameter was set to NULL on input, nothing is returned.
    *        In case a record control is returned, the caller is responsible
    *        for its release.
    * @param recordName [IN] Pass the record name to assign the vertex record
    *        to create, or an empty string in order to automatically generate
    *        a unique ID if a record name is not already defined.
    * @param label [IN] The label to assign the vertex record. This label is
    *        required but does not necessarily have to be unique.
    * @param referenceRecord [IN] Optional. Record control referring to another
    *        record that this in-graph record will reference. Such reference
    *        records can be used to hold additional data for the vertex that do
    *        not fit within the defined structure of a graph database.
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * This function creates a simple in-graph vertex record, optionally by
    * referencing another (non-graph) record with additional information.
    *
    * Unless the recordControl parameter is set to NULL, the caller will be
    * responsible for committing the record using TdbPutRecord() and for the
    * release of the record control using TdbDeleteRecordControl().
    *
    * #### Privileges Required
    * The current TRIPsystem user needs write access to the graph database in
    * which the vertex record is to be created.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    *
    * #### Return Codes
    * The number of steps this function performs is such that a significant
    * number of return codes may be expected. They are therefore not listed
    * here. Please refer to the error codes list in this reference manual for
    * explanations to any code returned.
    *
    * @see TdbGraphFind()
    * @see TdbGraphResolve()
    * @see TdbGraphTransitive()
    * @see TdbGraphStartPathAnalysis()
    * @see TdbGraphPathAnalysisStep()
    * @see TdbGraphClosePathAnalysis()
    * @see TdbGraphAnalysisSnapshot()
    * @see TdbGraphNextPath()
    * @see TdbGraphGetPath()
    * @see TdbGraphRecAssoc()
    */
   int TdbGraphAddVertex(TdbHandle* recordControl, const char* recordName,
                         const char* label, TdbHandle referenceRecord);
   
   
#ifndef DOXYGEN_SHOULD_SKIP_THIS
   /**
    * @brief Perform a graph query
    * @param resultSet Output parameter. Receives the search result set number.
    * @param setType Output parameter. Receives the graph set type, which is
    *        either GRAPH_SETTYPE_REGULAR (0) for a vertex set, or
    *        GRAPH_SETTYPE_GRAPH (1) for an edge set.
    * @param recordCount Number of records in the result set.
    * @param query Graph query language expression to execute
    * @return 1 or other odd value on success, even value on error
    * @ingroup graphdb
    *
    * #### Description
    * @note This is a HIGHLY EXPERIMENTAL function that is not part of the
    * public TRIP kernel API. It is likely to change or be removed without
    * notice.
    */
   int TdbGraphQuery(int* resultSet, int* setType, int* recordCount,
                     const char* query);
#endif


   // ------------------------------------------------------------------------

  /**
  * @defgroup jsonxml JSON/XML Functions
  * @brief Functions for interaction with JSON/XML databases
   *
   * Commands to programmatically interact with JSON/XML databases, 
   * primarily to load JSON and XML documents into TRIPxml databases, 
   * and to read JSON and XML documents from TRIPxml databases.
  */

  /**
    * Store an in-memory XML document into a JSON/XML database.
    * @param recordControl Record control associated with a JSON/XML database
    * @param storeCopy Pass true to store a binary copy in the D_XMLDOC field
    * @param validate Pass true to validate the XML according to DTD or Schema
    * @param xmldata XML document contents
    * @param length The length of the data in \p xmldata.
    * @return 1 or other odd value on success, even value on error
    * @ingroup jsonxml
    *
    * #### Description
    * If this function succeeds, the record is ready for further customization
    * by the application, and when/if no further customization is needed, to
    * commit to the database using the TdbPutRecord() function.
    *
    * @see TdbPutRecord()
    */
   int TdbPutXmlBuffer(TdbHandle recordControl, boolean storeCopy,
      boolean validate, const char* xmldata, int length);


   /**
    * Store an in-memory JSON document into a JSON/XML database.
    * @param recordControl Record control associated with a JSON/XML database
    * @param jsondata JSON document contents
    * @param length The length of the data in \p jsondata.
    * @return 1 or other odd value on success, even value on error
    * @ingroup jsonxml
    *
    * #### Description
    * If this function succeeds, the record is ready for further customization
    * by the application, and when/if no further customization is needed, to
    * commit to the database using the TdbPutRecord() function.
    *
    * @see TdbPutRecord()
    */
   int TdbPutJsonBuffer(TdbHandle recordControl, const char* jsondata, int length);


   /**
    * Store an XML document file into a JSON/XML database.
    * @param recordControl Record control associated with a JSON/XML database
    * @param storeCopy Pass true to store a binary copy in the D_XMLDOC field
    * @param validate Pass true to validate the XML according to DTD or Schema
    * @param xmlfile Name of a local XML file to import
    * @return 1 or other odd value on success, even value on error
    * @ingroup jsonxml
    *
    * #### Description
    * If this function succeeds, the record is ready for further customization
    * by the application, and when/if no further customization is needed, to
    * commit to the database using the TdbPutRecord() function.
    *
    * @see TdbPutRecord()
    */
   int TdbPutXmlFile(TdbHandle recordControl, boolean storeCopy,
      boolean validate, const char* xmlfile);


   /**
    * Store a JSON document file into a JSON/XML database.
    * @param recordControl Record control associated with a JSON/XML database
    * @param jsonfile Name of a local JSON file to import
    * @return 1 or other odd value on success, even value on error
    * @ingroup jsonxml
    *
    * #### Description
    * If this function succeeds, the record is ready for further customization
    * by the application, and when/if no further customization is needed, to
    * commit to the database using the TdbPutRecord() function.
    *
    * @see TdbPutRecord()
    */
   int TdbPutJsonFile(TdbHandle recordControl, const char* jsonfile);


   /**
    * Get a JSON/XML record formatted as an XML document.
    * @param recordControl Record control associated with a JSON/XML database
    * @param xmldata Receives a buffer containing the XML document
    * @param length Receives the length of the data returned in \p xmldata
    * @return 1 or other odd value on success, even value on error
    * @ingroup jsonxml
    *
    * #### Description
    * The buffer returned in \p xmldata must be deallocated by the caller.
    * Use the free() function for this purpose on Linux/UNIX. On Windows,
    * it is necessary to use HeapFree() Win32 API function on the heap returned
    * by the GetProcessHeap() Win32 API function. Using the regular free() function
    * to release this memory will on Windows cause a crash or put the application
    */
   int TdbGetXmlBuffer(TdbHandle recordControl, char** xmldata, int* length);


   /**
    * Get a JSON/XML record formatted as a JSON document.
    * @param recordControl Record control associated with a JSON/XML database
    * @param jsondata Receives a buffer containing the JSON document
    * @param length Receives the length of the data returned in \p jsondata
    * @return 1 or other odd value on success, even value on error
    * @ingroup jsonxml
    *
    * #### Description
    * The buffer returned in \p jsondata must be deallocated by the caller.
    * Use the free() function for this purpose on Linux/UNIX. On Windows,
    * it is necessary to use HeapFree() Win32 API function on the heap returned 
    * by the GetProcessHeap() Win32 API function. Using the regular free() function 
    * to release this memory will on Windows cause a crash or put the application
    * into an unstable state.
    */
   int TdbGetJsonBuffer(TdbHandle recordControl, char** jsondata, int* length);


   /**
    * Execute an XPath based query on a JSON/XML database.
    * @param xpath XPath query expression
    * @param searchSet Receives the number of the resulting serch set.
    * @return 1 or other odd value on success, even value on error
    * @ingroup jsonxml
    *
    * #### Description
    * The use of this function requries that the session is started in
    * Unicode (UTF-8) mode, and that a JSON/XML database ís currently
    * open for search.
    *
    * Refer to the TRIPsystem product document "JSON and XML Databases"
    * for information about the supported XPath syntax.
    *
    * Use the value returned in the \p searchSet parameter with the
    * TdbSearchInfo() function to obtain information about record
    * and hit count.
    */
   int TdbExecuteXPath(const char* xpath, int* searchSet);


   /**
    * Get a JSON/XML search result formatted as an XML fragment set document.
    * @param searchSetNo Number of a search in a JSON/XML database
    * @param xpath XPath expression to use for the selection of document fragments
    * @param startRis The ordinal number of the record in the search at which to start extracting fragments
    * @param endRis The ordinal number of the record in the search at which to stop extracting fragments
    * @param highlight Pass true to apply search set highlighting markup to the resulting document
    * @param buffer Receives a buffer containing the requested fragment set
    * @param length Receives the length of the data returned in \p buffer
    * @return 1 or other odd value on success, even value on error
    * @ingroup jsonxml
    *
    * #### Description
    * This function uses an XPath expression to transform a search set of
    * TRIPxml records containing XML or JSON documents into a single XML document
    * with fragments from each document. The XPath expression is used to specify
    * the fragment to include from each document in the output.
    *
    * Refer to the TRIPsystem product document "JSON and XML Databases"
    * for information about the supported XPath syntax.
    *
    * The buffer returned in \p buffer must be deallocated by the caller.
    * Use the free() function for this purpose on Linux/UNIX. On Windows,
    * it is necessary to use HeapFree() Win32 API function on the heap returned
    * by the GetProcessHeap() Win32 API function. Using the regular free() function
    * to release this memory will on Windows cause a crash or put the application
    * into an unstable state.
    */
   int TdbGetXmlFragments(int searchSetNo, const char* xpath,
      int startRis, int endRis, boolean highlight,
      void** buffer, int* length);



   // ------------------------------------------------------------------------
   
   /**
    * @defgroup misc Miscellaneous functions
    * @brief Functions of miscellaneous kinds
    */
   

   /**
    * @brief Release member returned by various other kernel API functions.
    * @param buf Memory to release
    * @return 1 or other odd value on success, even value on error
    * @ingroup misc
    *
    * #### Description
    * Functions that return memory that should be deallocated using this
    * function include:
    *
    *  + TdbGetFieldInfo()
    *  + TdbGetFocus()
    *  + TdbGetFocus2()
    */
   int TdbDeallocate(void* buf);
   

   /**
    * @brief Make a database known to TRIP
    * @param base Name of database
    * @return 1 or other odd value on success, even value on error
    * @ingroup misc
    *
    * #### Description
    * The function TdbOpenBase() makes a database known to TRIP without
    * opening it for searching. The database will however still be included in
    * the list of databases opened in the current session.
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string as the name of the
    * database to open. On a successful call, this parameter receives the 
    * normalized name of the database.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 7010 | Database LOG file cannot be opened. |  |
    * | 3106 | Old version of BIF/VIF files. | The database must be reindexed before it can be opened. |
    * | 11362 | Max 250 databases can be opened during a TRIP session. |  |
    * | 2530 | No access to database name. | The logged on user did not have access to the specified database. |
    * | 9538 | No write access to database name. | The logged on user did not have write access to the specified database. |
    * | 15010 | name is a database cluster. |  |
    * | 17314 | No access to DB cluster name. |  |
    * | 1934 | Database name not found. | The specified database did not exist. |
    * | 962 | File name error, check logical names for database. | The BAF/BIF/VIF files could not be found at the specified location. |
    * | 1314 | Database name is OS read protected. | The protection on the files for this database prohibits read access. |
    */
   int TdbOpenBase(char* base);
   
   
   /**
    * @brief Retrieve the number of bases which can be opened
    * @param bases Number of bases to open in current session
    * @return 1 or other odd value on success, even value on error
    * @ingroup misc
    *
    * #### Description
    * The function TdbOpenBaseQuota() returns the number of databases that can
    * be opened in addition to those currently open in the session.
    *
    * One base has to been set.    *
    *
    * #### Parameter Values
    * The \p bases parameter is passed a pointer to an \c int variable to receive 
    * the number of bases that can be opened in the current session.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    */
   int TdbOpenBaseQuota(int* bases);


   /**
    * @brief Call TRIPcof to extract text from file data stored in a STRING field.
    * @param recordControl A record control for the record into which to store the extracted text
    * @param partNumber The part number for the fields identified by \p targetFieldName, \p propertyNameField and \p propertyValueField
    * @param targetFieldName The name of the TEXT field to receive the extracted text
    * @param propertyNameField The name of a PHRASE field to receieve a list of extractyed property names (typled with \p propertyValueField), or NULL to skip property extarction.
    * @param propertyValueField The name of a PHRASE field to receieve a list of extractyed property values (typled with \p propertyNameField), or NULL to skip property extarction
    * @param sourceFieldName The name of the STRING field in which the file is stored
    * @param fileName The name of the file stored in the source field
    * @return 1 or other odd value on success, even value on error
    * @ingroup misc
    *
    * #### Description
    * This function extracts text and, optionally, document properties from a file
    * stored in a STRING field. Upon successful completion of this function, the
    * text and optional properties have been stored in the named fields in the
    * record referenced by the \p recordControl. Use TdbPutRecord() to commit the
    * results.
    *
    * @see TdbCreateRecordControl
    * @see TdbPutRecord
    * @see TdbSetCursor
    */
   int TdbExtractTextFromField(TdbHandle recordControl,
      int partNumber,
      const char* targetFieldName,
      const char* propertyNameField,
      const char* propertyValueField,
      const char* sourceFieldName,
      const char* fileName);

   /**
    * @brief Call TRIPcof to extract text from a file on the local file system
    * @param recordControl A record control for the record into which to store the extracted text
    * @param partNumber The part number for the fields identified by \p targetFieldName, \p propertyNameField and \p propertyValueField
    * @param targetTextFieldName The name of the TEXT field to receive the extracted text
    * @param propertyNameField The name of a PHRASE field to receieve a list of extractyed property names (typled with \p propertyValueField), or NULL to skip property extarction.
    * @param propertyValueField The name of a PHRASE field to receieve a list of extractyed property values (typled with \p propertyNameField), or NULL to skip property extarction
    * @param targetStringFieldName The name of a STRING field into which to store a copy of the file, or NULL to not store the file.
    * @param fileName The name of a local file to extract text and properties from
    * @return 1 or other odd value on success, even value on error
    * @ingroup misc
    *
    * #### Description
    * This function extracts text and, optionally, document properties from a file
    * on the local file system. Upon successful completion of this function, the
    * text and optional properties have been stored in the named fields in the
    * record referenced by the \p recordControl. A copy of the file can optionally 
    * be stored in the record as well. Use TdbPutRecord() to commit the results.
    *
    * @see TdbCreateRecordControl
    * @see TdbPutRecord
    */
   int TdbExtractTextFromFile(TdbHandle recordControl,
      int partNumber,
      const char* targetTextFieldName,
      const char* propertyNameField,
      const char* propertyValueField,
      const char* targetStringFieldName,
      const char* fileName);

   /**
    * @brief Call TRIPcof to extract text from in-memory file data.
    * @param recordControl A record control for the record into which to store the extracted text
    * @param partNumber The part number for the fields identified by \p targetFieldName, \p propertyNameField and \p propertyValueField
    * @param targetFieldName The name of the TEXT field to receive the extracted text
    * @param propertyNameField The name of a PHRASE field to receieve a list of extractyed property names (typled with \p propertyValueField), or NULL to skip property extarction.
    * @param propertyValueField The name of a PHRASE field to receieve a list of extractyed property values (typled with \p propertyNameField), or NULL to skip property extarction
    * @param targetStringFieldName The name of a STRING field into which to store a copy of the file, or NULL to not store the file.
    * @param filedata The file data to extract text from
    * @param fileDataSize The length of the data in the \p filedata buffer
    * @param fileName The name of the file contained in the \p filedata buffer
    * @return 1 or other odd value on success, even value on error
    * @ingroup misc
    *
    * #### Description
    * This function extracts text and, optionally, document properties from a file
    * on the local file system. Upon successful completion of this function, the
    * text and optional properties have been stored in the named fields in the
    * record referenced by the \p recordControl. A copy of the file can optionally
    * be stored in the record as well. Use TdbPutRecord() to commit the results.
    *
    * @see TdbCreateRecordControl
    * @see TdbPutRecord
    */
   int TdbExtractTextFromMem(TdbHandle recordControl,
      int partNumber,
      const char* targetFieldName,
      const char* propertyNameField,
      const char* propertyValueField,
      const char* targetStringFieldName,
      unsigned char* filedata,
      size_t fileDataSize,
      const char* fileName);



	// ------------------------------------------------------------------------

	/**
	 * @defgroup obsoleted Deprecated Functions
	 * @brief Functions no longer to be used.
	 *
	 * These functions are no longer maintained. Replacement functions are
	 * available where possible. Application code should be modified to use
	 * the named alternate functions as soon as possible, since the deprecated
	 * functions may be removed in a future release.
	 */

   
   /**
    * @brief Initialize TRIP for a new session or resume a previous session
    * @deprecated Deprecated by TdbStartTrip()
    * @param shell_dao Address of the shell dao interface record application
    *        will use.
    * @param restart Start mode of the session.
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * NOTE: This function has been superseded by TdbStartTrip() .
    *
    * The TdbBeginTdbs() function is the first API function that should be
    * called by an application program. Upon return, and assuming a success
    * return code, the environment for further API functions will have been
    * initialised. However, this function should never be called by an
    * application software exit (ASE) as all required initialisation will have
    * been successfully completed prior to the ASE being activated.
    *
    * The value passed as the \p restart argument specifies whether or not
    * this session is to be restarted from a SIF file that was retained from
    * some previous session. If the value of the \p restart argument is TRUE
    * then some internal initialization is bypassed and the context is read
    * from the SIF file upon calling the TdbCheckUser() function. If the value
    * of the \p restart argument is FALSE then a new SIF file will always be
    * generated and any existing file ignored.
    *
    * The TdbBeginTdbs() function initializes the TRIP kernel for use by the
    * calling process. It performs consistency checks upon the operating system
    * environment, TRIPsystem installation, licence conditions, and initialises
    * essential status variables and reporting mechanisms in the TRIP kernel.
    * If the site running the application is controlled by a concurrent user
    * license, the TRIP Daemon is consulted to gain a session token, if
    * possible. It checks for the existence and accessibility of the data
    * dictionary (the CONTROL database) and of the error message store (the
    * TDBSERR database) that the kernel needs during execution.
    *
    * The length of CCL statements in sessions started with TdbBeginTdbs() is
    * limited to 400 bytes. The superseding function TdbStartTrip() does not
    * impose any such limit.
    *
    * #### Parameter Values
    * The \p shell_dao parameter is passed the address of a pointer to a shell
    * dao structure. This is used by the system during the execution of DAO
    * commands (see the TdbDaoOrder() function). It is also used to return
    * additional information, which cannot be contained within a return code,
    * from other information gathering functions such as the TdbGetSearch()
    * function. Refer to the TRIP Data Structures Reference Guide for a
    * complete description of the structure. The record must be initialized
    * with null-bytes before calling the routine. Only one shell_dao record
    * can be used by an application.
    *
    * The \p restart parameter is passed a flag to control the (re)start mode
    * of a session during initialization of TRIP. Valid values are:
    * | Value | Purpose |
    * | ----- | ------- |
    * | False | Start the session with a new SIF file. |
    * | True | Restart the session from an existing SIF, if possible. |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 162 | System file {1} not found or OS read protected. |  |
    * | 8290 | OS open file limit of {1} reached. |  |
    * | 8418 | TRIP/TDBS availability has expired. |  |
    * | 12098 | Cannot find TDBS access record. |  |
    * | 16898 | Concurrent user quota (N) exceeded | The maximum number of licenced concurrent TRIP users has been reached. When this error code is returned the accompanying error text will include the maximum number of concurrent users licenced for the machine. |
    * | 16418 | TRIPdaemon failed to respond (1), notify your System Mgr. |  |
    * | 16450 | TRIPdaemon failure (1), notify your System Mgr. |  |
    * | 17570 | Invalid TRIPdaemon process name | The TRIPkernel cannot locate the process for the TRIPdaemon. |
    * | 18370 | The ShellDao input argument is NULL. |  |
    * | 34435 | TRIP/TDBS availability expires {1}. | The function completed its operations without error. |
    *
    * @see TdbStartTrip()
    * @see TdbCheckUser()
    * @see TdbDaoOrder()
    * @see TdbEndTdbs()
    */
   TDB_DEPRECATED("Deprecated by TdbStartTrip")
   int TdbBeginTdbs(shell_dao_rec* shell_dao, boolean restart);
   
   
   /**
    * @brief Terminates a TRIP session started with TdbBeginTdbs.
    * @deprecated Deprecated by TdbStopTrip()
    * @param restart Mode of session reinitialization
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbEndTdbs() will end the current TRIP session. The session
    * must have been started with TdbBeginTdbs() . When TRIP exits an option
    * exists to save the current SIF file for future use. This function must be
    * the last TRIP toolkit routine called by the application. It must be
    * called only once per image activation.
    *
    * No calls to this function should be made from an application software
    * exit (ASE).
    *
    * #### Parameter Values
    * The \p restart parameter is passes a byte ( char ) a flag for the mode
    * of a subsequent session's initialization. Valid values are:
    *
    * | Constant Value | Purpose |
    * | -------------- |--- ---- |
    * | 0 | Delete the SIF file |
    * | 1 | Retain the SIF file for possible re-use in a later session |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 10434 | Background printing still in progress. | This may occur when a PRINT command is still executing on the server. |
    * | 16322 | Disk quota exceeded when writing to the SIF file. |  |
    *
    * @see TdbBeginTdbs()
    * @see TdbStartTrip()
    * @see TdbStopTrip()
    */
   TDB_DEPRECATED("Deprecated by TdbStopTrip")
   int TdbEndTdbs(boolean restart);

   
   /**
    * @brief Create a copy of a database definition
    * @deprecated Deprecated by TdbCopyBaseDesign()
    * @param old_base Name of (existing) database
    * @param new_base Name of (new) database
    * @param base_specification Database specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbCopyBaseDef() initiates the copying of one database
    * design to another of a different name. Upon a successful return the
    * \p base_specification parameter pointer will reference a base
    * specification record containing all the general properties of the new
    * database as copied from the old.
    *
    * In order to commit the new database to CONTROL the process must
    * subsequently call the TdbPutBaseDef() function.
    *
    * Note that the filename properties are not copied by this operation.
    *
    * #### Parameter Values
    * The \p old_base parameter is passed a character string as the name of the
    * existing database whose design is to be copied.
    *
    * The \p new_base parameter is passed a character string as the name of
    * the new database.
    *
    * The \p base_specification parameter is passed a base specification
    * record for the new database. Values will be copied form the equivalent
    * record for the old database.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 34019 | Database design copied. | Operation completed successfully. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 7586 | Already existing database design: name. | Target database must not exist. |
    *
    * @see TdbGetBaseDef()
    * @see TdbPutBaseDef()
    */
   TDB_DEPRECATED("Deprecated by TdbCopyBaseDesign")
   int TdbCopyBaseDef(char* old_base, char* new_base,
                      base_spec_rec* base_specification);

   /**
    * @brief Create a copy of a thesaurus definition
    * @deprecated Deprecated by TdbCopyThesDesign()
    * @param old_thes Name of source thesaurus
    * @param new_thes Name of new thesaurus
    * @param thes_specification Thesaurus specification record for new thesaurus.
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbCopyThesDef() starts the copying of a thesaurus to
    * different one of a new name. The calling process must subsequently call
    * TdbPutThesDef() to commit the new thesaurus to the CONTROL file.
    *
    * #### Parameter Values
    * The \p old_thes parameter is passed a character string as the name of the
    * thesaurus design to be copied.
    *
    * The \p new_thes parameter is passed a character string as the name of the
    * thesaurus to be created.
    *
    * The \p thes_specification parameter is passed a thesaurus specification
    * record for the newly created thesaurus.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * @see TdbCopyThesDesign()
    * @see TdbPutThesDef()
    */
   TDB_DEPRECATED("Deprecated by TdbCopyThesDesign")
   int TdbCopyThesDef(const char* old_thes, const char* new_thes,
                      base_spec_rec* thes_specification);
   

   /**
    * @brief Execute a DAO order
    * @deprecated Deprecated by TdbExecuteCcl()
    * @param order Type of command executed
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbDaoOrder() attempts to execute the DAO order stored in
    * the field \p DAO of the shell dao interface record passed as an argument
    * to the call of the TdbBeginTdbs() function.
    *
    * Normally, the DAO order will be created from a CCL order using the
    * function TdbShellToDao() but the calling process can simply write DAO
    * statements to the shell dao interface record and calling the
    * TdbDaoOrder() function.
    *
    * The syntax and semantics of DAO orders is not documented.
    *
    * Upon return from this function the value of the \p order argument can be
    * used to branch to perform command specific post processing such as
    * reporting on the number of records hit by a search, or running each line
    * of a procedure. As most commands result in some kind of output, a simple
    * generalized kernel window output handler will ensure that most commands
    * are handled correctly.
    *
    * #### Parameter Values
    * The \p order parameter is passed a pointer to an \c int variable that will 
    * receive a value that corresponds to the type of DAO command which was executed
    * the TdbDaoOrder() function. The returned values are:
    *
    * | Symbolic Name | Constant Value | Explanation |
    * | ------------- | -------------- | ----------- |
    * | DAO_CUT | 1 | A CUT order was executed; equivalent to the keystrokes KP6 in TRIPclassic. |
    * | DAO_SELECT | 2 | A SELect order was executed. |
    * | DAO_FIND | 3 | A Find order was executed. |
    * | DAO_SHOW | 4 | A Show order was executed. |
    * | DAO_MORE | 5 | A More order was executed. |
    * | DAO_BASE | 6 | A BASe order was executed. |
    * | DAO_CALL | 7 | A CALl order was executed. |
    * | DAO_DISPLAY | 8 | A Display order was executed. |
    * | DAO_IMPORT | 9 | A IMPOrt order was executed. |
    * | DAO_EXPORT | 10 | A EXPOrt order was executed. |
    * | DAO_MACRO | 11 | A RUN macro order was executed. |
    * | DAO_STOP | 12 | A STOP order was executed. |
    * | DAO_DEFINE | 13 | A DEfine order was executed. |
    * | DAO_LIST | 14 | A List order was executed. |
    * | DAO_NEXT | 15 | A Next order was executed. |
    * | DAO_PREV | 15 | A PREVious order was executed. |
    * | DAO_HELP | 16 | A HELP order was executed. |
    * | DAO_EXPAND | 17 | A EXPand order was executed. |
    * | DAO_PRINT | 18 | A PRint order (but a not print local order) was executed. |
    * | DAO_RUN | 19 | A RUN procedure order was executed. |
    * | DAO_SAVE | 20 | A SAVe order was executed. |
    * | DAO_DELETE | 21 | A DELete order was executed. |
    * | DAO_TRACE | 22 | A TRace order was executed. |
    * | DAO_CONTINUE | 23 | A CONTinue order was executed. |
    * | DAO_STATUS | 24 | A STatus order was executed. |
    * | DAO_PRINT_LOCAL | 25 | A PRint LOCal order was executed. |
    * | DAO_EDIT | 26 | A EDit order was executed. |
    * | DAO_INSERT | 27 | A INSert order was executed. |
    * | DAO_UPDATE | 28 | A UPDate order was executed. |
    * | DAO_BACK | 29 | A Back order was executed. |
    * | DAO_TOP | 30 | A Top order was executed. |
    * | DAO_MEASURE | 31 | A MEasure order was executed. |
    * | DAO_FREQ | 32 | A FReq order was executed. |
    * | DAO_INDEX | 33 | A INDex order was executed. |
    * | DAO_CONT_EDIT | 34 | A CONTinue EDit order was executed. |
    * | DAO_CLOSE | 35 | A CLOSE order was executed. |
    * | DAO_HIDE | 36 | A HIDe order was executed. |
    * | DAO_REVEAL | 37 | A REVeal order was executed. |
    * | DAO_CONT_SHOW | 38 | A CONTinue Show order was executed. |
    * | DAO_FUZZ | 39 | A FUZz order was executed. |
    * | DAO_SFORM | 40 | A SFOrm order was executed. |
    * | CMD_LEAVE | 41 | A LEAVE order was executed. |
    * | CMD_LOAD | 42 | A LOAD order was executed. |
    * | CMD_RENUM | 43 | A RENUM order was executed. |
    * | CMD_FIND_SAVE | 44 | A FIND SAVE order was executed. |
    *
    * #### Return Codes
    * The return codes from this function depends on the nature of the CCL
    * command given. Since most of TRIP operations can be executed via CCL,
    * this function may return almost any return code.
    *
    * @see TdbCurrentBase()
    * @see TdbGetDisplay()
    * @see TdbGetSearch()
    * @see TdbGetWindowInfo()
    * @see TdbGetWindowHits()
    * @see TdbNextRunCommand()
    * @see TdbShellSymbol()
    * @see TdbShellToDao()
    */
   TDB_DEPRECATED("Deprecated by TdbExecuteCcl")
   int TdbDaoOrder(int* order);

   
   /**
    * @brief Delete a field specification from a database design
    * @deprecated Deprecated by TdbDeleteBaseField()
    * @param fldspec specification Field specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbDeleteFieldSpec() deletes a given field from a database
    * design in memory, which must have previously been loaded using
    * TdbGetBaseDef(). Either of the functions TdbGetFieldName() or
    * TdbGetFieldSpec() should also have been called to put the proper values in
    * the field specification record. Upon a subsequent call to TdbPutBaseDef(),
    * the field will be deleted from the data dictionary. This routine can only
    * be used successfully when the database is empty.
    *
    * #### Parameter Values
    * The \p fldspec parameter is passed a pointer to a field specification
    * record structure.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 7906 | Field name is missing. |  |
    * | 14978 | Thesaurus fields may not be deleted. | Predefined fields in a thesaurus may not be deleted. |
    * | 18146 | XML fields may not be deleted. | Predefined fields in an XML database may not be deleted. |
    * | 1698 | Non-existing field name: name |  |
    * | 7810 | Deletion of fields is legal only when the database is empty. |  |
    *
    * @see TdbDeleteBaseField
    * @see TdbGetBaseDef()
    * @see TdbGetFieldSpec()
    * @see TdbPutBaseDef()
    */
   TDB_DEPRECATED("Deprecated by TdbDeleteBaseField")
   int TdbDeleteFieldSpec(field_spec_rec* fldspec);
   
    
    /**
     * @brief Get record name field and other information about the database
     * @deprecated Deprecated by TdbBaseInfo()
     * @param base Name of database
     * @param recordNameField Name of record name field
     * @param flags Field specific flags
     * @return 1 or other odd value on success, even value on error
     * @ingroup obsoleted
     *
     * #### Description
     * The function TdbGetBaseInfo() returns the name of the record name field
     * and other data about the database.
     *
     * #### Parameter Values
     * The \p base parameter is passed a character string as the name of the
     * database.
     *
     * The \p recordNameField is passed a character string as the record name
     * field.
     *
     * The \p flag parameter is passed a pointer to an \c int variable that 
     * receives a bit mask. The current acceptable values are:
     *
     * | Constant Value | Description |
     * | -------------- | ----------- |
     * | 0 | Bit0 is set if the database is an XML database. |
     *
     * #### Since
     * This function was introduced in TRIP version 3.3-1.
     *
     * #### Return Codes
     * | Code | Description | Explanation |
     * | ---- | ----------- | ----------- |
     * | 1 | Operation completed successfully | The function completed its operations without error. |
     * | 15234 | Missing database name. |  |
     * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
     * | 610 | Database name not found. | The specified database did not exist. |
     * | 3042 | Name is not a database. |  |
     * | 2530 | No access to database name. | The logged on user did not have access to the specified database. |
     * | 2498 | You have no FM access to the database. |  |
     *
     * @see TdbBaseInfo()
     * @see TdbGetBaseDesign()
     * @see TdbPutBaseDesign()
     */
    int TdbGetBaseInfo(char* base, char* recordNameField, int* flags);
    
   
   /**
    * @brief Load a database definition
    * @deprecated Deprecated by TdbGetBaseDesign()
    * @param base Name of database
    * @param baseSpecification Database specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbGetBaseDef() loads the definition of the specified TRIP
    * database into the base specification record passed via the
    * \p baseSpecification parameter. The TRIPsystem user ID of the calling
    * process must be the owner of the database being examined.
    *
    * #### Parameter Values
    * The \p base parameter is passed a character string as the name of the
    * database.
    *
    * The \p baseSpecification is passed a base specification record as the
    * database specification record.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the (file manager) FM privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 611 | Database name not found. | The specified database did not exist. |
    * | 15234 | Missing database name. |  |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 3042 | Name is not a database. |  |
    * | 2530 | No access to database name. | The logged on user did not have access to the specified database. |
    * | 9538 | No write access to database name. | The logged on user did not have write access to the specified database. |
    * | 10594 | No delete rights to database name. | The logged on user did not have delete access to the specified database. |
    * | 2498 | You have no FM access to the database. |  |
    *
    * @see TdbGetBaseDesign()
    * @see TdbPutBaseDef()
    */
   TDB_DEPRECATED("Deprecated by TdbGetBaseDesign")
   int TdbGetBaseDef(char* base, base_spec_rec* baseSpecification);

   
    /**
     * @brief Check the specified database files for need of defragmentation.
     * @param base Name of database to check
     * @param baf Receives true if the BAF file needs defragmentation else false
     * @param bif Receives true if the BIF file needs defragmentation else false
     * @param vif Receives true if the VIF file needs defragmentation else false
     * @return 1 or other odd value on success, even value on error
     * @ingroup obsoleted
     * @deprecated Deprecated without replacement.
     *
     * #### Description
     * This function has been been deprecated without replacement and
     * will now always return false for all files. Use the file system tools
     * available for your operating system to perform defragmentation checks
     * instead.
     */
    TDB_DEPRECATED("Deprecated without replacement")
    int TdbGetDefragStatus(const char* base, boolean* baf, boolean* bif,
                           boolean* vif);
    
    

   /**
    * @brief Load the name of a field
    * @deprecated Deprecated by TdbGetBaseField()
    * @param field_specification Field specification record
    * @param ordinal Ordinal field number
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbGetFieldName() loads the name of a field into the field
    * specification record passed via the \p field specification parameter.
    *The field selected is specified by the ordinal field number in the
    * database definition.
    *
    * A call to TdbGetBaseDef() must be made before any attempts to load the
    * field specification record are made with this function.
    *
    * To obtain a complete field specification TdbGetFieldSpec() should
    * subsequently be called in which that function reads the field name in
    * the specification record to determine which field's specification is
    * being requested.
    *
    * #### Parameter Values
    * The \p field_specification parameter is passed a field specification
    * record to be filled in with the name of the field.
    *
    * The \p ordinal number parameter is passed an integer as the ordinal
    * field number. Acceptable values range from one to the number of fields
    * in the database design. If no fields have been deleted from the database
    * design, the ordinal field number will be equal to the actual field
    * number.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Could not get field name. |
    *
    * @see TdbGetBaseDef()
    */
   TDB_DEPRECATED("Deprecated by TdbGetBaseField")
   int TdbGetFieldName(field_spec_rec *field_specification, int ordinal);
   
   /**
    * @brief Load a field specification record
    * @deprecated Deprecated by TdbGetBaseField()
    * @param fieldSpec Field specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbGetFieldSpec() loads the specification of a database
    * field into the field specification record passed via the \p fieldSpec
    * parameter. Before calling the routine, the name of the desired field
    * must be put into the \p field_name member of the record by either
    * calling TdbGetFieldName() or explicitly copying the field name into the
    ' record.
    *
    * A call to the function TdbGetBaseDef() has to have preceded a call to
    * this function.
    *
    * #### Parameter Values
    * The \p fieldSpec parameter is passed a field specification record to be
    * filled in with details of the request field.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 0 |  | Could not get field specification. |
    *
    * @see TdbGetBaseField()
    * @see TdbGetBaseDef()
    * @see TdbGetFieldName()
    */
   TDB_DEPRECATED("Deprecated by TdbGetBaseField")
   int TdbGetFieldSpec(field_spec_rec* fieldSpec);
   
   
   /**
    * @brief Store a database definition
    * @deprecated Deprecated by TdbPutBaseDesign()
    * @param baseSpec Base specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbPutBaseDef() writes the database definition contained in
    * the base specification record provided into the CONTROL file. A call to
    * TdbPutBaseDef() must have been preceded by a call to either of the
    * TdbGetBaseDef() or TdbCopyBaseDef() functions.
    *
    * #### Parameter Values
    * The \p baseSpec parameter is passed a base specification recordto write
    * to the CONTROL file.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 32515 | Database design for name altered. | Database design successfully altered. |
    * | 32547 | Database design for name created. | Database design successfully created. |
    * | 33411 | Database design for name altered (check logical names). | Database design successfully altered, but there may be something wrong with the BAF/BIF/VIF file names. |
    * | 33443 | Database design for name created (check logical names). | Database design successfully created, but there may be something wrong with the BAF/BIF/VIF file names. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 11842 | Unbalanced parenthesis in database description. |  |
    * | 14786 | id is not a valid character folding class. |  |
    *
    * @see TdbPutBaseDesign()
    * @see TdbCopyBaseDesign()
    * @see TdbGetBaseDesign()
    * @see TdbCopyBaseDef()
    * @see TdbGetBaseDef()
    */
   TDB_DEPRECATED("Deprecated by TdbPutBaseDesign")
   int TdbPutBaseDef(base_spec_rec* baseSpec);
   
   
   /**
    * @brief Store a field specification record
    * @deprecated Deprecated by TdbPutBaseField() and TdbCloseBaseField()
    * @param fieldSpec Field specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbPutFieldSpec() stores the field specification record into
    * the internal database definition buffer for later writing to the CONTROL
    * file by a call to the TdbPutBaseDef() function. This routine can only be
    * called after a database definition has been loaded by the TdbGetBaseDef()
    * and TdbGetFieldSpec() function have been called.
    *
    * #### Parameter Values
    * The \p fieldSpec parameter is passed a field specification record with
    * updated details of a field in the relevant database. ´
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 2146 | Non-existing field: field. |  |
    * | 6786 | Already existing field name. |  |
    * | 6850 | Invalid field type. |  |
    * | 7106 | Field name is already the unique record name field. | There can be only one record name field in a database. |
    * | 7266 | Record name field must be of type {PHR}. |  |
    * | 7426 | field is already copyright protected by field. |  |
    * | 7458 | Field name must be: letter followed by alphanums or underscores. |  |
    * | 7618 | Changing of field type is legal only when the database is empty. |  |
    * | 7650 | Removal of record id is legal only when the database is empty. |  |
    * | 7714 | Pattern restriction is only valid for field type {PHR}. |  |
    * | 7746 | File reference restriction is only valid for field type {PHR}. |  |
    * | 8226 | Field name name coincides with the database name. | A field can not have the same name as the database. |
    * | 8258 | Record id field must not be specified for non-empty databases. |  |
    * | 8482 | Field field is already the record number field. |  |
    * | 8514 | Record number field must be of type {INT}. |  |
    * | 8546 | Removal of rec no field is legal only when the database is empty. |  |
    * | 8578 | Record number field must not be specified for non-empty databases. |  |
    * | 9058 | Invalid interval. |  |
    * | 11810 | Unbalanced parenthesis in field comment. |  |
    * | 12674 | Changing part field status is allowed only for empty databases. |  |
    * | 14946 | The fieldtype of a thesaurus field may not be changed. |  |
    * | 15426 | Changing from/to one paragraph is only allowed for empty databases. |  |
    * | 15650 | Only the CTX field in a thesaurus can be the record name field. |  |
    * | 15810 | Field field is already the unique part name field. |  |
    * | 15842 | Part name field must be of type {PHR}. |  |
    * | 15874 | Removal of part name field is legal only when the database is empty. |  |
    * | 15906 | Part name field must not be specified for non-empty databases. |  |
    * | 15938 | Part name field must be specified as part field. |  |
    * | 17154 | A part record field cannot be made the record name field. |  |
    * | 17186 | A part record field cannot be made the record number field. |  |
    * | 17378 | Word indexing is only valid for {PHR} fields. |  |
    * | 18018 | The fieldtype of an XML field may not be changed. | The predefined fields of an XML database may not be altered. |
    *
    * @see TdbPutBaseField()
    * @see TdbCloseBaseField()
    * @see TdbGetFieldSpec()
    * @see TdbGetBaseDef()
    * @see TdbPutBaseDef()
    */
   TDB_DEPRECATED("Deprecated by TdbPutBaseField and TdbCloseBaseField")
   int TdbPutFieldSpec(field_spec_rec* fieldSpec);

   
   /**
    * @brief Load a thesaurus definition
    * @deprecated Deprecated by TdbGetThesDesign()
    * @param thesaurus Name of thesaurus
    * @param thesaurusSpec Thesaurus specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbGetThesDef() loads the definition of the specified
    * TRIPsystem thesaurus into the base specification record provided. The
    * TRIPsystem user ID of the calling process must be the owner of the
    * thesaurus being examined.
    *
    * #### Parameter Values
    * The \p thesaurus parameter is passed a character string as the name of
    * the thesaurus.
    *
    * The \p thesaurusSpec parameter is passed a base specification record as
    *the thesaurus specification record.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 17474 | A TRIP runtime version cannot create databases. | This function cannot be used with a runtime TRIP license. |
    * | 1934 | Database name not found. | The specified database did not exist. |
    * | 3010 | name is not a thesaurus. |  |
    * | 2498 | You have no FM access to the database. |  |
    *
    * @see TdbGetThesDesign()
    * @see TdbCopyThesDef()
    * @see TdbDeleteThesDef()
    * @see TdbPutThesDef()
    */
   TDB_DEPRECATED("Deprecated by TdbGetThesDesign")
   int TdbGetThesDef(char* thesaurus, base_spec_rec* thesaurusSpec);

   /**
    * @brief XML enable a database design
    * @deprecated Deprecated by TdbPutBaseProperty()
    * @param xml XML-enabled flag
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * Defines a database loaded by a call to TdbGetBaseDef() to be XML enabled.
    *
    * Once a database has been XML enabled this cannot be revoked.
    *
    * #### Parameter Values
    * The \p xml parameter is passed a boolean as a flag that database is
    * XML-enabled
    *
    * #### Since
    * This function was introduced in TRIP version 3.3-1.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 34019 | Database design copied. | Operation completed successfully. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 18082 | An existing database cannot be XML enabled. | Only a database without any pre-existing fields can be XML-enabled. |
    * | 18114 | XML enabling of a database cannot be reset. | A database cannot be XML-disabled, once it has been XML-enabled. |
    *
    * @see TdbPutBaseProperty()
    * @see TdbGetBaseDef()
    */
   TDB_DEPRECATED("Deprecated by TdbPutBaseProperty")
   int TdbDefineXml(boolean xml);

   
   /**
    * @brief Graph-enable a database design
    * @deprecated Deprecated by TdbPutBaseProperty()
    * @param base_spec Database design specification as returned by TdbGetBaseDef.
    * @param enable Pass true to declare this database as a graph, and false to declare it as a regular database.
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * Defines a database loaded by a call to TdbGetBaseDef() to be
    * graph-enabled.
    *
    * Once a database has been graph-enabled this cannot be revoked.
    *
    * #### Since
    * This function was introduced in TRIP version 7.1-0.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 18114 | XML enabling of a database cannot be reset. | A database cannot be graph-enabled if it is already XML-enabled. |
    * | 23106 | Graph enabling of a database cannot be reset. | A database cannot be graph-enabled if it is already graph-enabled. |
    * | 23138 | An existing database cannot be graph enabled. | A database cannot be graph-enabled if its design already contains fields. |
    *
    * @see TdbPutBaseProperty()
    * @see TdbGetBaseDef()
    */
   TDB_DEPRECATED("Deprecated by TdbPutBaseProperty")
   int TdbDefineGraph(base_spec_rec* base_spec, boolean enable);


   /**
    * @brief Store a thesaurus definition
    * @deprecated Deprecated by TdbPutThesDesign()
    * @param thesaurusSpec Base specification record for thesaurus
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbPutThesDef() writes the thesaurus definition in the base
    * specification record into the CONTROL file. A call to TdbPutThesDef()
    * must be preceded by a call to the TdbGetThesDef() function.
    *
    *
    * #### Parameter Values
    * The \p thesaurusSpec parameter is passed a base specification record for
    * the thesaurus.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 34275 | Thesaurus design for {name} altered (Check logical names). | Operation completed successfully. |
    * | 34307 | Thesaurus design for {name} created (Check logical names). | Operation completed successfully. |
    * | 34339 | Thesaurus design for {name} altered. | Operation completed successfully. |
    * | 34371 | Thesaurus design for {name} created. | Operation completed successfully. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 11842 | Unbalanced parenthesis in database description. |  |
    * | 14786 | {id} is not a valid character folding class. |  |
    *
    * @see TdbGetThesDef()
    */
   TDB_DEPRECATED("Deprecated by TdbPutThesDesign")
   int TdbPutThesDef(base_spec_rec* thesaurusSpec);


   /**
    * @brief Obtain search result information
    * @deprecated Deprecated by TdbSearchInfo()
    * @param search Search number
    * @param record_count Number of records for search
    * @param hit_count Number of hits for search
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbGetSearch() returns information about the results of a
    * previous CCL Find order. It returns the number of records found by the
    * search, and the number of occurrences of the search terms found within
    * these records.
    *
    * #### Parameter Values
    * The \p search parameter is passed a pointer to an \c int variable that
    * represents the number of the search for which details are to be retrieved. 
    * If the number is set to zero on input, details for the current search will 
    * be returned and the actual search number will be returned in this parameter
    * on succesful output. Otherwise, this parameter must be set to a positive
    * number on input, which must correspond to an existing search set within
    * the session.
    *
    * The \p record_count parameter is passed a pointer to an \c int variable to 
    * receive the number of records found by the search.
    *
    * The \p hit_count parameter is passed a pointer to an \c int variable to 
    * receive the count of terms actually hit by the search.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 2274 | Non-existing search number: {1} |  |
    *
    * @see TdbSearchInfo()
    * @see TdbSearchResult()
    */
   TDB_DEPRECATED("Deprecated by TdbSearchInfo")
   int TdbGetSearch(int* search, int* record_count, int* hit_count);

   
   /**
    * @brief Returns information about a conducted search.
    * @deprecated Deprecated by TdbSearchInfo()
    * @param searchNumber The number of the search to request information
    *        about. Use 0 (zero) for the last search. The real search number
    *        is returned in this parameter.
    * @param recordCount This output parameter contains the number of records
    *        found in the specified search.
    * @param hitCount This output parameter contains the total number of hits
    *        in the specified search.
    * @param cclOrder The server version takes a pointer to an (unallocated)
    *        character buffer as input, and on output receives the CCL
    *        statement that generated the search set. The client version of
    *        this parameter is a by the application allocated character buffer
    *        that will receieve the CCL statement that generated the search
    *        set.
    * @param length  Takes a pointer to an integer that recieves the length of
    *        the cclOrder on output.
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The TdbSearchResult() function returns information about a conducted
    * search such as search number, record count, hit count, and the CCL order
    * that generated the search set.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 2274 | Non-existing search number: number. | The specified search number did not correspond to a search set. |
    *
    * @see TdbSearchInfo
    */
   TDB_DEPRECATED("Deprecated by TdbSearchInfo")
   int TdbSearchResult(int* searchNumber, int* recordCount, int*hitCount,
                       char** cclOrder, int* length);

   
   /**
    * @brief Load a database cluster specification
    * @deprecated Deprecated by TdbGetClusterDesign()
    * @param cluster name Cluster name
    * @param clusterSpecification Cluster Specification Record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbGetDbCluster() loads and makes current the specification
    * of a stored database cluster.
    *
    * The cluster specification can only be loaded by the owner of the cluster.
    *
    *
    * #### Parameter Values
    * The \p cluster name parameter is passed a character string as the name
    * of the cluster whose specification record is to be retrieved.
    *
    * The \p cluster specification parameter is passed a cluster specification
    * record to receive the predefined database cluster specification record.
    * Refer to the TRIP Data Structures Reference Guide for a description of
    * the record.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 706 | You have no rights to create or alter database designs. | The current user needs FM (file manager) rights to access this function. |
    * | 8802 | Only alphanumerics/underscores allowed in database/cluster names. | The specified cluster name contained invalid characters. |
    * | 17442 | name is not a database cluster. |  |
    * | 16770 | You are not the owner of this DB cluster. |  |
    *
    * @see TdbGetClusterdesign()
    */
   TDB_DEPRECATED("Deprecated by TdbGetClusterDesign")
   int TdbGetDbCluster(char* cluster, cluster_spec_rec* clusterSpecification);

   
   /**
    * @brief Store a database cluster specification
    * @deprecated Deprecated by TdbPutClusterDesign()
    * @param clusterSpec Cluster specification record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbPutDbCluster() will store the specification of a database
    * cluster defined in the cluster specification record.
    *
    * The cluster specification record must previously have been obtained by a
    * call to the TdbGetDbCluster() function.
    *
    * #### Parameter Values
    * The \p clusterSpec parameter is passed a cluster specification record,
    * with changes to the named predefined cluster of databases.
    *
    * #### Privileges Required
    * The current TRIPsystem user must possess the file manager (FM) privilege
    * to execute this function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 39235 | Database cluster design for name altered. | Operation completed successfully. |
    * | 39267 | Database cluster design for name created. | Operation completed successfully. |
    * | 9794 | Database name name is the same as the DB cluster name. |  |
    *
    * @see TdbPutClusterDesign()
    * @see TdbGetDbCluster()
    */
   TDB_DEPRECATED("Deprecated by TdbPutClusterDesign")
   int TdbPutDbCluster(cluster_spec_rec* clusterSpec);

   
   /**
    * @brief Get information about the current TRIP session
    * @deprecated Deprecated by TdbUserInfo() and TdbSessionInfo()
    * @param run_time Run-time indicator
    * @param logged_in Flag for successful call to CHECK USER
    * @param FM File manager flag
    * @param UM User manager flag
    * @param SM System manager flag
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbGetSessionInfo() retrieves miscellaneous information
    * about the current TRIP session and TRIP user. If the function
    * TdbBeginTdbs() has not been called it will return an error code.
    *
    * #### Parameter Values
    * The \p run_time parameter is passed a byte to receive the run-time
    * indicator. The returned values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | False | 0 | Current TRIPsystem session is not a run-time session |
    * | True | 1 | The session is a run-time session and database creation/modification is not allowed. |
    *
    * The \p logged_in parameter is passed a byte to receive the flag
    * indicating whether a TRIP user identity has been established by a
    * successful call to TdbCheckUser() . The returned values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | False | 0 | No user identity |
    * | True | 1 | A TRIPkernel user has successfully logged in. |
    *
    * The \p FM parameter is passed a byte to receive the flag indicating the
    * user's status as a File manager. This parameter is only valid if
    * \p logged_in is returned as True. The returned values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | False | 0 | The current TRIP user is not a file manager |
    * | True | 1 | The TRIP user is a file manager and may create databases |
    *
    * The \p UM parameter is passed a byte to receive the flag indicating the
    * user's status as a User manager. This parameter is only valid if
    * \p logged_in is returned as True. The return values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | False | 0 | The current TRIP user is not a user manager |
    * | True | 1 | The TRIP user is a user manager and may create other users |
    *
    * The \p SM parameter is passed a byte to receive the flag indicating the
    * user's status as System manager. This parameter is only valid if
    * \p logged_in is returned as True. The returned values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | False | 0 | The current TRIP user is not the system manager |
    * | True | 1 | The TRIP user is the system manager |
    *
    * @see TdbCheckUser()
    * @see TdbSessionInfo()
    * @see TdbUserInfo()
    */
   TDB_DEPRECATED("Deprecated by TdbSessionInfo and TdbUserInfo")
   int TdbGetSessionInfo(char* run_time, Char* logged_in,
                         Char* FM, Char* UM, Char* SM);

   
   /**
    * @brief Load the next command in a stored procedure
    * @deprecated Deprecated by TdbGetNextProcLine()
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    *
    * The function TdbNextRunCommand() loads the next line from a stored TRIP
    * procedure or macro into the \p shin shell input field in the
    * \c shell_dao interface record that is currently being used by TRIPsystem.
    *
    * This function should be called after a call to TdbDaoOrder() that has
    * returned a \c DAO_RUN ( \c 19 ) status indicating that a CCL command
    * command has been executed. Repeated calls to TdbNextRunCommand() will
    * cause each line of the stored procedure to be copied into the
    * \c shell_dao interface record, for subsequent parsing (by the function
    * TdbShellToDao() ) and execution (by TdbDaoOrder() ).
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbDaoOrder()
    * @see TdbShellToDao()
    * @see TdbGetNextProcLine()
    */
   TDB_DEPRECATED("Deprecated by TdbGetNextProcLine")
   int TdbNextRunCommand(void);


   /**
    * @brief Retrieve the address of a shell dao record
    * @deprecated Deprecated by TdbExecuteCcl()
    * @param shell_dao Address of SHELL/DAO record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * The function TdbShellDaoPtr() returns the address of the SHELL/DAO
    * record area for the current TRIP session, as supplied to the
    * TdbBeginTdbs() function.
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    * | 9954 | Missing mandatory argument. |  |
    */
   TDB_DEPRECATED("Deprecated by TdbExecuteCcl")
   int TdbShellDaoPtr(shell_dao_rec** shell_dao);

   
   /**
    * @brief Initialize the CCL parser
    * @deprecated Deprecated by TdbStartTrip() and TdbShellModifyCcl()
    * @param language CCL locale language to use
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * @note This function is deprecated. The command shell is initialized when
    * calling TdbStartTrip(). The CCL language used also can be changed by
    * calling the TdbShellModifyCcl() function.
    *
    * The function TdbShellInit() initializes the CCL parser used in the TRIP
    * search mode and is used to specify the language file which is to be used.
    * The default language file is specified by the TRIPrcs setting TDBS_LANG.
    * If no such setting is specified, the default will be English.
    *
    * #### Parameter Values
    * The \p language parameter is passed an integer that indicates which CCL
    * locale language to use. Valid values are:
    *
    * | Symbolic Name | Constant Value | Purpose |
    * | ------------- | -------------- | ------- |
    * | LANGUAGE_DEFAULT | 1 | Use the system default language |
    * | LANGUAGE_ENGLISH | 9 | Use English dialect |
    * | LANGUAGE_SWEDISH | 10 | Use Swedish dialect |
    * | LANGUAGE_CHINESE | 11 | Use Chinese dialect |
    * | LANGUAGE_GERMAN |  12 | Use German dialect |
    * | LANGUAGE_FRENCH |  13 | Use French dialect |
    * | LANGUAGE_FINNISH | 14 | Use Finnish dialect |
    * | LANGUAGE_NORWEGIAN | 15 | Use Norwegian dialect |
    *
    * #### Return Codes
    * | Code | Description | Explanation |
    * | ---- | ----------- | ----------- |
    * | 0 | Operation failed |  |
    * | 1 | Operation completed successfully | The function completed its operations without error. |
    *
    * @see TdbStartTrip()
    * @see TdbShellModifyCcl()
    */
   TDB_DEPRECATED("Deprecated by TdbStartTrip and TdbShellModifyCcl")
   int TdbShellInit(int language);

   
   /**
    * @brief Parse a CCL command
    * @deprecated Deprecated by TdbExecuteCcl() and TdbGetParsedCcl()
    * @param shell_dao Address of SHELL TO DAO record
    * @return 1 or other odd value on success, even value on error
    * @ingroup obsoleted
    *
    * #### Description
    * @note This function is superseded by the functions TdbExecuteCcl()
    * and TdbGetParsedCcl() .
    *
    * The function TdbShellToDao() will parse the CCL command contained in the
    * shell input field of the SHELL/DAO interface record. This routine is the
    * interface to the embedded CCL parser within the TRIP system. The modified
    * CCL command produced during this parsing process is stored in the shout
    * shell output field of the interface record, and the DAO (internal) form
    * of the command is stored in the dao DAO input field of the record.
    *
    * To execute the order, call TdbDaoOrder() after successful completion of this function.
    *
    * #### Parameter Values
    * The \p shell_dao parameter is passed a pointer to the SHELL/DAO record
    * that was originally passed to the TdbShellInit() function. Refer to the
    * TRIP Data Structures Reference Guide for a description of the SHELL/DAO
    * interface record.
    *
    * #### Return Codes
    * The return codes from this function depends on the nature of the CCL
    * command given. Since most of TRIP operations can be executed via CCL,
    * this function may return almost any return code.
    *
    * @see TdbExecuteCcl()
    * @see TdbGetParsedCcl()
    */
   TDB_DEPRECATED("Deprecated by TdbExecuteCcl and TdbGetParsedCcl")
   int TdbShellToDao(shell_dao_rec* shell_dao);

   
   
#if defined(__cplusplus)
}
#endif

#endif

