/*******************************************************************************
 *
 * Filename : tripapi_defs.h
 * Copyright: infinIT Services GmbH
 * Purpose  : Macros and constants for the TRIP API
 *
 */

#ifndef __TRIPAPI_DEFS_H__
#define __TRIPAPI_DEFS_H__

/**
 * @defgroup macros Macros and constants
 * @brief Macro and constant definitions
 */

/**
 * @brief Length of CCL commands when using the legacy DAO mode
 * @ingroup macros
 */
#define MAXORDERLEN     400

/* Manifest constants for TdbBeginTdbs() / TdbEndTdbs() */
#define NO_RESTART              0
#define DISCARD_SESSION         0
#define START_NEW               (char)0     /* Obsolete */
#define ALLOW_RESTART           1
#define SAVE_SESSION            1
#define RESTART                 (char)1     /* Obsolete */

/**
 * @name Window types
 * @brief Manifest constants for window types used in TdbDefineWindow()
 * @ingroup macros
 *  @{
 */
#define CURRENT_WINDOW          0 /*!< @ingroup macros Current output window */
#define SHOW_WINDOW             1 /*!< Output window for SHOW commands */
#define EXPAND_WINDOW           2 /*!< EXPAND output window  */
#define DISPLAY_WINDOW          3 /*!< Output window for DISPLAY commands */
#define HISTORY_WINDOW          4 /*!< Output window for search history */
#define SYSINFO_WINDOW          5 /*!< Output window for system info */
#define PRINT_WINDOW            6 /*!< PRINT output window */
/** @} */

/**
 * @name DAO Completion Codes
 * @brief TdbDaoOrder() completion codes for end-user programs
 * @ingroup macros
 *  @{
 */
#define DAO_SELECT              2      /*!< @ingroup macros SELECT command */
#define DAO_FIND                3
#define DAO_SHOW                4
#define DAO_MORE                5
#define DAO_BASE                6
#define DAO_CALL                7
#define DAO_DISPLAY             8
#define DAO_IMPORT              9
#define DAO_EXPORT              10
#define DAO_MACRO               11
#define DAO_STOP                12
#define DAO_DEFINE              13
#define DAO_LIST                14
#define DAO_NEXT                15
#define DAO_PREV                15
#define DAO_HELP                16
#define DAO_EXPAND              17
#define DAO_PRINT               18
#define DAO_RUN                 19
#define DAO_SAVE                20
#define DAO_DELETE              21
#define DAO_TRACE               22
#define DAO_CONTINUE            23
#define DAO_STATUS              24
#define DAO_PRINT_LOCAL         25
#define DAO_EDIT                26
#define DAO_INSERT              27
#define DAO_UPDATE              28
#define DAO_BACK                29
#define DAO_TOP                 30
#define DAO_MEASURE             31
#define DAO_FREQ                32
#define DAO_INDEX               33
#define DAO_CLOSE               35
#define DAO_FUZZ                39
#define DAO_LOAD                42
#define DAO_RENUM               43
#define DAO_FIND_SAVE           44
/** @} */

/**
 * @name Classic DAO completion codes
 * @brief TdbDaoOrder() completion codes used by TRIPclassic / TRIPclient
 * @ingroup macros
 *  @{
 */
#define DAO_CUT                 1      /*!< @ingroup macros CUT command */
#define DAO_CONT_EDIT           34
#define DAO_HIDE                36
#define DAO_REVEAL              37
#define DAO_CONT_SHOW           38
#define DAO_SFORM               40
#define DAO_LEAVE               41
/** @} */

/**
 * @name CCL completion codes
 * @brief TdbExecuteCcl() completion codes for end-user programs
 * @ingroup macros
 *  @{
 */
#define CMD_ERROR               0      /*!< @ingroup macros CCL error */
#define CMD_CUT                 1
#define CMD_SELECT              2
#define CMD_FIND                3
#define CMD_SHOW                4
#define CMD_MORE                5
#define CMD_BASE                6
#define CMD_CALL                7
#define CMD_DISPLAY             8      /*!< DISPLAY command */
#define CMD_IMPORT              9
#define CMD_EXPORT              10
#define CMD_MACRO               11
#define CMD_STOP                12
#define CMD_DEFINE              13
#define CMD_LIST                14
#define CMD_NEXT                15
#define CMD_PREV                15
#define CMD_HELP                16
#define CMD_EXPAND              17
#define CMD_PRINT               18
#define CMD_RUN                 19
#define CMD_SAVE                20
#define CMD_DELETE              21
#define CMD_TRACE               22
#define CMD_CONTINUE            23
#define CMD_STATUS              24
#define CMD_PRINT_LOCAL         25
#define CMD_EDIT                26
#define CMD_INSERT              27
#define CMD_UPDATE              28
#define CMD_BACK                29
#define CMD_TOP                 30
#define CMD_MEASURE             31
#define CMD_FREQ                32
#define CMD_INDEX               33
#define CMD_CONT_EDIT           34
#define CMD_CLOSE               35
#define CMD_HIDE                36
#define CMD_REVEAL              37
#define CMD_CONT_SHOW           38
#define CMD_FUZZ                39
#define CMD_SFORM               40
#define CMD_LEAVE               41
#define CMD_LOAD                42
#define CMD_RENUM               43
#define CMD_FIND_SAVE           44
#define CMD_DELETE_SEARCH       45
/** @} */

/**
 * @name Language constants
 * @brief Manifest constants for use in TdbStartTrip(), TdbShellInit(),
 *       TdbShellModifyCcl(), etc
 * @ingroup macros
 *  @{
 */
#define LANGUAGE_DEFAULT        0   /*!< @ingroup macros Doesn't read dialect file  */
#define LANGUAGE_ENGLISH        9   /*!< Reads TDBS_SYS:ENGLISH.CCL */
#define LANGUAGE_SWEDISH        10  /*!< Reads TDBS_SYS:SWEDISH.CCL   */
#define LANGUAGE_CHINESE        11  /*!< Reads TDBS_SYS:CHINESE.CCL   */
#define LANGUAGE_GERMAN         12  /*!< Reads TDBS_SYS:GERMAN.CCL    */
#define LANGUAGE_FRENCH         13  /*!< Reads TDBS_SYS:FRENCH.CCL    */
#define LANGUAGE_FINNISH        14  /*!< Reads TDBS_SYS:FINNISH.CCL   */
#define LANGUAGE_NORWEGIAN      15  /*!< Reads TDBS_SYS:NORWEGIAN.CCL */
#define LANGUAGE_JAPANESE       16  /*!< Reads TDBS_SYS:JAPANESE.CCL  */
/** @} */

/**
 * @name Character set constants
 * @brief Manifest character set constants for use in TdbStartTrip() and
 *        TdbSessionInfo().
 * @ingroup macros
 *  @{
 */
#define CHSET_NONE               0  /*!< @ingroup macros No charset */
#define CHSET_LA1                1  /*!< ISO-8859-1 */
#define CHSET_LA2                2  /*!< ISO-8859-2 */
#define CHSET_LA3                3  /*!< ISO-8859-3 */
#define CHSET_UTF8               16 /*!< UTF-8 */
#define CHSET_CHI                17 /*!< GBK */
#define CHSET_GBK                18 /*!< GBK */
#define CHSET_EUC                19
#define CHSET_SJIS               20
/** @} */

/**
 * @brief Macro for converting a character set constant value into a value
 * suitable for the TdbStartTrip mode parameter.
 * @ingroup macros
 */
#define CHARSET_MODE(charset) (charset << 8)

/**
 * @brief Value for the TdbStartTrip mode parameter when using a login
 * ticket or access token to log in instead of username and password.
 * @ingroup macros
 */
#define STARTTRIP_TICKET 0x80

/**
 * @brief Value for the TdbStartTrip mode parameter when using an
 * access token for login and an API key is supplied in the username
 * parameter.
 * @ingroup macros
 */
#define STARTTRIP_APIKEY 0x2000000

/**
 * @brief Value for the TdbStartTrip mode parameter when starting from
 * an application that runs across the network. Always used when started
 * via tbserver.
 * @ingroup macros
 */
#define STARTTRIP_NETWORK 0x1000000

/**
 * @name CONTROL list information
 * @brief Manifest constants for use in TdbShowControl() for output of CONTROL file.
 * @ingroup macros
 *  @{
 */
#define DATABASE_LIST           1   /*!< @ingroup macros Show BASe LIST */
#define USER_LIST               2   /*!< Show USer */
#define DATABASE_STATUS         3   /*!< Show BASe R= */
#define PROCEDURE_LIST          4   /*!< Show PRoc */
#define DATABASE_ACCESS         5   /*!< Show BASe ACCess */
#define OUTPUT_FORMAT_LIST      6   /*!< Show Format */
#define ENTRY_FORM_LIST         7   /*!< Show EFOrm */
#define SEARCH_FORM_LIST        8   /*!< Show SFOrm */
#define THESAURUS_STATUS        9   /*!< STatus Xxx */
#define GROUP_LIST              10  /*!< Show Group */
#define CURRENT_STATUS          11  /*!< Status */
#define USERS_PER_GROUP         14
#define GROUPS_PER_USER         15
#define EXTENDED_USER_LIST      17  /*!< Show USer */
#define DATABASE_LIST_NOBAF     18  /*!< Show BASe LIST, info from BIF not BAF */
#define TEMPLATE_LIST           19
 /** @} */

/**
 * @name Database access mode flags
 * @brief Manifest constants for use in TdbSetBase() access modes
 * @ingroup macros
 *  @{
 */
#define MODE_BIT_READ_ONLY      1 /*!< @ingroup macros Read access */
#define MODE_BIT_READ_WRITE     2
#define MODE_BIT_DELETE_CHECK   4
#define MODE_BIT_COPY           8
#define MODE_BIT_EDIT_PREP      16
#define MODE_BIT_CHANGE_EFO     32
#define MODE_BIT_REVERT_EFO     64

#define MODE_READ               ( MODE_BIT_READ_ONLY )
#define MODE_WRITE              ( MODE_BIT_READ_WRITE | MODE_BIT_READ_ONLY )
#define MODE_DELETE             ( MODE_BIT_DELETE_CHECK | MODE_BIT_READ_WRITE )
#define MODE_COPY               ( MODE_BIT_COPY | MODE_BIT_READ_WRITE )
#define MODE_EDIT               ( MODE_BIT_EDIT_PREP | MODE_BIT_READ_WRITE )
#define MODE_NEW_EFO            ( MODE_BIT_CHANGE_EFO | MODE_BIT_EDIT_PREP | MODE_BIT_READ_WRITE )
#define MODE_OLD_EFO            ( MODE_BIT_REVERT_EFO | MODE_BIT_EDIT_PREP | MODE_BIT_READ_WRITE )
/** @} */


/**
 * @name Access check mode types
 * @brief Manifest constants for use in TdbCheckBase(), TdbCheckDbCluster(),
 *        and TdbCheckThes()
 * @ingroup macros
 *  @{
 */
#define CHECK_READ              1   /*!< @ingroup macros Verify READ access */
#define CHECK_UPDATE            2
#define CHECK_OWNERSHIP         3
#define CHECK_DELETE            4
/** @} */

/**
 * @name Baffit  mode types
 * @brief Manifest constants for use in TdbBaffitMode()
 * @ingroup macros
 *  @{
 */
#define ADD_MODE                1 /*!< @ingroup macros */
#define MODIFY_MODE             2
#define DELETE_MODE             3
/** @} */

/**
 * @name Batch job types
 * @brief Manifest constants for use in TdbBatchLoadIndex()
 * @ingroup macros
 *  @{
 */
#define LOADIX                  1   /*!< Mode for TdbBatchLoadIndex() - load TFORM into database an and index it */
#define INDEX                   2   /*!< Mode for TdbBatchLoadIndex() - index database */
#define LOAD                    3   /*!< Mode for TdbBatchLoadIndex() - load TFORM into database */
#define REINDEX                16   /*!< Mode for TdbBatchLoadIndex() - reindex an entire database, use with INDEX or LOADIX */
 /** @} */

/**
 * @name Synchronous 
 * @brief Manifest constants for use in TdbIndex()
 * @ingroup macros
 *  @{
 */
#define SYNCH_INDEX_ALL       -17   /*!< Mode for TdbIndex() - index all new and modified records synchronously */
#define SYNCH_REINDEX_ALL     -21   /*!< Mode for TdbIndex() - reindex the entire database synchronously */
 /** @} */

/**
 * @name Break constants
 * @brief Manifest constants for use with TdbBreak()
 * @ingroup macros
 *  @{
 */
#define BREAK_NONE              0   /*!< @ingroup macros */
#define BREAK_USER              1
/** @} */

/**
 * @name Field types
 * @brief Manifest constants for use with TdbShellType() / TdbCheckField()
 * @ingroup macros
 * @{
 */
#define FIELD_TEXT              1    /*!< @ingroup macros TEXT field */
#define TEXT_FIELD              1       /* Deprecated */
#define FIELD_PHRASE            3
#define PHRASE_FIELD            3       /* Deprecated */
#define FIELD_INTEGER           9
#define INTEGER_FIELD           9       /* Deprecated */
#define FIELD_NUMBER            10
#define NUMBER_FIELD            10      /* Deprecated */
#define FIELD_DATE              11
#define DATE_FIELD              11      /* Deprecated */
#define FIELD_TIME              12
#define TIME_FIELD              12      /* Deprecated */
#define FIELD_STRING            14
#define STRING_FIELD            14      /* Deprecated */
/** @} */

/**
 * @name Part record name check modes
 * @brief Manifest constants for return codes from TdbCheckPartName()
 * @ingroup macros
 * @{
 */
#define PART_EXIST              16034  /*!< @ingroup macros */
#define PART_NOT_EXIST          16002
/** @} */

/**
 * @name Record name check modes
 * @brief Manifest constants for return codes from TdbCheckRecordName()
 * @ingroup macros
 * @{
 */
#define RECORD_EXIST            7074      /*!< @ingroup macros */
#define RECORD_NOT_EXIST        2882
/** @} */

/**
 * @name SELECT record modes
 * @brief Manifest constants for return codes from TdbCheckSelectRecord()
 * @ingroup macros
 * @{
 */
#define NO_SELECTION_MADE       0         /*!< @ingroup macros */
#define SELECTION_MADE          1
/** @} */

/**
 * @name Symbol check modes
 * @brief Manifest constants for use with TdbCheckSymbol() classes
 * @ingroup macros
 * @{
 */
#define CHECK_BASENAME          2          /*!< @ingroup macros */
#define CHECK_VIEWNAME          3
#define CHECK_TEXT              4
#define CHECK_PHRASE            5
#define CHECK_NUMBER            6
#define CHECK_DATE              7
#define CHECK_TIME              8
#define CHECK_ALL               9
#define CHECK_FIELDS            10
#define CHECK_MAPS              14
/** @} */

/**
 * @name Timestamp options
 * @brief Manifest constants for use with TdbCreateTimeStamp() / TdbGetTimeStamp()
 * @ingroup macros
 * @{
 */
#define DATE_AND_TIME           0          /*!< @ingroup macros */
#define DATE_ONLY               1
/** @} */

/**
 * @name Current form modes
 * @brief Manifest constants for use with TdbCurrentForm()
 * @ingroup macros
 * @{
 */
#define CURRENT_OUTPUT          1          /*!< @ingroup macros */
#define CURRENT_ENTRY           2
/** @} */

/**
 * @name DISPLAY options
 * @brief Manifest constants for use with TdbGetDisplay()
 * @ingroup macros
 * @{
 */
#define DISPLAY_NORMAL          1          /*!< @ingroup macros */
#define DISPLAY_THES            2
#define DISPLAY_INDIRECT        3
/** @} */


/**
 * @name Term list/set sorting options
 * @brief Manifest constants for use with TdbSortTerms()
 * @ingroup macros
 * @{
 */
#define TERM_SORT_TEXT   1        /*!< Sort alphabetically ascending */
#define TERM_SORT_FREQ  2         /*!< Sort on frequency, descending */
#define TERM_SORT_BL_ATEND   8    /*!< Put zero-sized baseline terms at end */
#define TERM_SORT_TEXT_DESC  16   /*!< descending alpha sort direction */
#define TERM_SORT_FREQ_ASC   32   /*!< ascending freq sort direction  */

 /** @} */


/**
 * @name Term list/set sorting options
 * @brief Manifest constants for use with TdbTermSetInfo()
 * @ingroup macros
 * @{
 */
#define TERM_VALUE                1     /*!< Term value */
#define TERM_RECORDCOUNT          2     /*!< No. of records the term matches */
#define TERM_TERMCOUNT            3     /*!< Total term count */
#define TERM_BASELINE_TERMCOUNT   4     /*!< Baseline term count */
#define TERM_BASELINE_RECORDCOUNT 5     /*!< Baseline term record count */
/** @} */

/**
 * @name TdbNextWindow modes
 * @brief Manifest constants for use with TdbNextWindow()
 * @ingroup macros
 * @{
 */
#define FIELD_LEVEL             0         /*!< @ingroup macros */
#define PARAGRAPH_LEVEL         1
#define ALL_FIELDS              0
#define TEXT_FIELDS             1
#define TEPH_FIELDS             2
#define PHRASE_FIELDS           3
/** @} */

/**
 * @name TdbProcessMode modes
 * @brief Manifest constants for use with TdbProcessMode()
 * @ingroup macros
 * @{
 */
#define INTERACTIVE_MODE        1         /*!< @ingroup macros */
#define TTY_MODE                2
#define BATCH_MODE              3
/** @} */

/**
 * @name Field index modes
 * @brief Manifest bitmask value constants for field indexing mode, for
 *        use with TdbGetFieldProperty() and TdbPutFieldProperty().
 * @ingroup macros
 *
 * The bitmask values FIELD_INDEX_WORD, FIELD_INDEX_SEPARATE and
 * FIELD_INDEX_CLASS can be combined by doing a bitwise or
 * operation on them (e.g. FIELD_INDEX_SEPARATE | FIELD_INDEX_CLASS).
 *
 * @{
 */
#define FIELD_INDEX_NONE        0   /*!< @ingroup macros  Not indexed */
#define FIELD_INDEX_NORMAL      1   /*!< Regular indexing */
#define FIELD_INDEX_WORD        5   /*!< Word indexed */
#define FIELD_INDEX_SEPARATE    17  /*!< Separate indexing */
#define FIELD_INDEX_CLASS       33  /*!< Field used in classification */
/** @} */

#ifndef DOXYGEN_SHOULD_SKIP_THIS
/* Manifest constants for use with TedCheckMode() */
#define RECORD_MODE             1
#define PART_MODE               2        /* not yet implemented */
#define FIELD_MODE              3        /* not yet implemented */
#endif

/**
 * @name BlobBlock status codes
 * @brief Manifest constants for use with TdbGetBlobBlock() / TdbPutBlobBlock()
 * @ingroup macros
 * @{
 */
#define NO_SUCH_BLOCK           0      /*!< @ingroup macros */
#define LAST_BLOCK              1
#define OK_BLOCK                3
#define NO_SUCH_CURSOR          14658
#define NO_SUCH_BLOB            21634
/** @} */

/* A generic lower-limit constant for TRIPapi return codes which may be
 *  interpreted by TdbMessage()
 */
#ifndef MSG_HAS_TEXT
#   define MSG_HAS_TEXT         32
#endif

/* Manifest constants for TdbMessage() */

#ifndef MSG_SET_ERROR
#define MSG_SET_ERROR          -1
#endif

#ifndef MSG_SET_INFO
#define MSG_SET_INFO           -2
#endif

/* Manifest constants for use with TdbImport() and TdbExport() */
#define IEOPT_FILENAME          1
#define IEOPT_FILEPTR           2
#define IEOPT_MEMORY            4
#define EXPORT_BLOCK            16
#define EXPORT_ALLOC            32
#define EXPORT_MKFILE           64
#define EXPORT_FILEAPP          256
#define EXPORT_FILETRUNC        512

/* TRIPxml-specific constants for the filter_options
 * parameter in the filter_data structure.
 */
#define FOXML_NEWREC            1
#define FOXML_REPLACE           2
#define FOXML_VALIDATE          4
#define FOXML_GETBYID           8
#define FOXML_REMAKE            16
#define FOXML_HILIGHT           32
#define FOXML_NOBLOB            64
#define FOXML_STREAM            256 ///< for server from client download!

/* Manifest constants for use with TdbBaseInfo() */
#define BASEINFO_XML            1
#define BASEINFO_CHARFOLD       2
#define BASEINFO_RNAMEFLD       3
#define BASEINFO_MAXFLD         4
#define BASEINFO_PNAMEFLD       5
#define BASEINFO_LANGUAGE       6
#define BASEINFO_CHARSET        7
#define BASEINFO_RNUMFLD        8
#define BASEINFO_MAXRNUM        9
#define BASEINFO_OWNER          10
#define BASEINFO_GRAPH          11
#define BASEINFO_CLUSTERBASES   12
#define BASEINFO_DBNUMBER       13
#define BASEINFO_DBDESC         14
#define BASEINFO_CONNECTOR      15
#define BASEINFO_RECORDCOUNT    16
#define BASEINFO_DESIGN_CREDATE 17
#define BASEINFO_DESIGN_CRETIME 18
#define BASEINFO_DESIGN_MODDATE 19
#define BASEINFO_DESIGN_MODTIME 20
#define BASEINFO_THESAURUS      21
#define BASEINFO_CONTAINER      22
#define BASEINFO_UTC            23

/* Manifest constants for use with TdbClusterInfo() */
#define CLUSTERINFO_RECORDCOUNT     1
#define CLUSTERINFO_BASECOUNT       2
#define CLUSTERINFO_MEMBERCOUNT     3
#define CLUSTERINFO_DATABASES       4
#define CLUSTERINFO_MEMBERS         5
#define CLUSTERINFO_UPDATED         6

/* Manifest constants for use with TdbFieldInfo() */
#define FIELDINFO_SIZE          1
#define FIELDINFO_TYPE          2
#define FIELDINFO_MAX_SUBFIELDS 3
#define FIELDINFO_IS_PARTFIELD  4

/* Manifest constants for use with TdbRecordInfo() */
#define RECORDINFO_RID          1
#define RECORDINFO_DBNAME       2
#define RECORDINFO_RNAME        3
#define RECORDINFO_TSTAMP_CODED 4
#define RECORDINFO_TSTAMP_ASCII 5
#define RECORDINFO_CLASS_NAME   6
#define RECORDINFO_SIZE         7
#define RECORDINFO_HITS         8
#define RECORDINFO_TSTAMP_UTC   9
#define RECORDINFO_TSTAMP_UTCDAY 10

/* Manifest constants for use with TdbCursorInfo() */
#define CURSORINFO_DB           1
#define CURSORINFO_RID          2
#define CURSORINFO_PARTID       3
#define CURSORINFO_FIELDNR      4
#define CURSORINFO_PARANR       5
#define CURSORINFO_ITEMNR       6
#define CURSORINFO_HITS         7

/* Manifest constants for use with TdbSessionInfo() */
#define SESSIONINFO_RUNTIME         1
#define SESSIONINFO_DATEFORM        2
#define SESSIONINFO_OPENBASES       5
#define SESSIONINFO_CURRENTBASE     6
#define SESSIONINFO_CLUSTER         7
#define SESSIONINFO_DBS_IN_SEARCH   8
#define SESSIONINFO_SIF_NAME        9
#define SESSIONINFO_CHARSET        10
#define SESSIONINFO_SESSION_STATUS 11
#define SESSIONINFO_CJK_MODE       12
#define SESSIONINFO_SUPERMAN       13
#define SESSIONINFO_COMMERCIAL     14
#define SESSIONINFO_DBS_INITIATED  15
#define SESSIONINFO_DB_IN_SEARCH   16
#define SESSIONINFO_HIGHEST_SEARCH 17
#define SESSIONINFO_VERSION        18
#define SESSIONINFO_VERSION_MAJOR  19
#define SESSIONINFO_VERSION_MINOR  20
#define SESSIONINFO_VERSION_SERVICE 21
#define SESSIONINFO_VERSION_PATCH  22
#define SESSIONINFO_CURRENTTHES    23
#define SESSIONINFO_THESDEF        24
#define SESSIONINFO_AUTOLIC        25

/* Manifest constants for use with TdbApiKeyInfo() */
#define APIKEYINFO_COMMENT          1
#define APIKEYINFO_OWNER            2
#define APIKEYINFO_CREATED          3
#define APIKEYINFO_CREATED_DATE     4
#define APIKEYINFO_CREATED_TIME     5

/* Manifest constants for use with TdbUserInfo() */
#define USERINFO_FM             1
#define USERINFO_UM             2
#define USERINFO_SM             3
#define USERINFO_STARTPROC      4
#define USERINFO_NAME           5
#define USERINFO_EXT_NAME       6
#define USERINFO_GROUPS         7
#define USERINFO_EXT_GROUPS     8

/* Manifest constants for use with TdbSearchInfo() */
#define SEARCHINFO_SET_NUMBER   1
#define SEARCHINFO_RECORD_COUNT 2
#define SEARCHINFO_HIT_COUNT    3
#define SEARCHINFO_COMMAND      4
#define SEARCHINFO_DATABASE     5
#define SEARCHINFO_DATABASES    6

/* Manifest constants for use with TdbLicenseInfo() */
#define LICINFO_PRODUCT         1
#define LICINFO_PRODUCT_SYSTEM  2
#define LICINFO_PRODUCT_HIGHWAY 3
#define LICINFO_PRODUCT_NXP     4
#define LICINFO_PRODUCT_COF     5
#define LICINFO_PRODUCT_XML     6
#define LICINFO_PRODUCT_JXP     7
#define LICINFO_PRODUCT_SQL     8
#define LICINFO_PRODUCT_CLIENT  9
#define LICINFO_PRODUCT_JTK     10
#define LICINFO_EXPIRATION_DATE 20
#define LICINFO_MAX_CONCURRENT_USERS 21
#define LICINFO_MAX_REGISTERED_USERS 22
#define LICINFO_MAX_OPEN_DATABASES 23
#define LICINFO_MAX_CLUSTER_SIZE 24
#define LICINFO_MAX_SEARCH_SETS 25
#define LICINFO_MAX_DBRECORDS 26
#define LICINFO_INCLUDES_GRAPH_SEARCH 50
#define LICINFO_INCLUDES_XPATH_SEARCH 51

/* Manifest constants for use with TdbChangeUser */
#define CHANGEUSER_KEEPALL           1
#define CHANGEUSER_CLEAR_SETS        2
#define CHANGEUSER_CLOSE_DATABASES   4
#define CHANGEUSER_RESTORE_DEFAULTS  8
#define CHANGEUSER_CLEARALL (CHANGEUSER_CLEAR_SETS|CHANGEUSER_CLOSE_DATABASES|CHANGEUSER_RESTORE_DEFAULTS)
#define CHANGEUSER_APIKEY            0x20

/* Constants for comforter mode */
#define COMFORT_SEARCH  1
#define COMFORT_SORT    2
#define COMFORT_WAIT    3

/* Constants for use with the graph API functions. */
#define GRAPH_DIRECTION_FOLLOW 1
#define GRAPH_DIRECTION_BACK   2
#define GRAPH_RESOLVE_SOURCE 1
#define GRAPH_RESOLVE_TARGET 2
#define GRAPH_SET_UNIVERSAL  0
#define GRAPH_PATH_ALL       1
#define GRAPH_PATH_SHORTEST  2
#define GRAPH_SETTYPE_REGULAR 0
#define GRAPH_SETTYPE_GRAPH   1

/* Structures and constants related to referential integrity */
#define INTEGRITY_RESTRICT              1
#define INTEGRITY_CASCADE               2
#define INTEGRITY_NOACTION              3
#define INTEGRITY_SETNULL               4
#define INTEGRITY_SETDEFAULT            5

/* Modes for opening containers with TdbSetClassificationContainer() */
#define MODE_VIEW     1 /* Read mode, view classes, etc. */
#define MODE_TRAIN    2 /* Training mode */
#define MODE_CLASSIFY 4 /* Classification mode */


/* Generic macros for testing success / failure of TRIPapi routines */
#if !defined(IsError)
#   define IsError(x)           ( !( (x) & 1) )
#endif

#if !defined(IsSuccess)
#   define IsSuccess(x)         ( (x) & 1 )
#endif


/* Macros for interpreting flags word from TdbCheckField() */
#define IsIndexed(x)            ( (x) & 1 )
#define IsWriteable(x)          ( (x) & 2 )
#define IsReadable(x)           ( (x) & 64 )
#define HasAccess(x)            ( (x) & (64+2))
#define IsPartField(x)          ( (x) & 4 )
#define IsMandatory(x)          ( (x) & 16 )
#define IsLayoutRetained(x)     ( (x) & 32 )

/* Macros for interpreting the return code from TdbGetRecordInBase(),
 * TdbGetRecordInSearch() and TdbGetRecordInSearchRis()
 */
#define IsNewBase(x)            ( (x) & 2 )
#define IsDeleted(x)            ( (x) & 4 )
#define IsLocked(x)             ( (x) & 8 )
#define IsAtEnd(x)              ( (x) & 16 )

/* Macros for interpreting flags word from TdbGetWindowInfo() */
#define IsAtTop(x)              ( (x) & 1 )     /* Top of output */
#define IsAtBottom(x)           ( (x) & 2 )     /* Bottom of output */
#define IsWithFF(x)             ( (x) & 4 )     /* Output format uses <FF> */


/* Manifest constants for use with TdbGetRecord() */
#define RECORD_FROM_BASE          0
#define RECORD_FROM_SEARCH        1
#define RECORD_FROM_SORTED_SEARCH 2


/* Database property ids for use with
 * TdbGetBaseProperty and TdbPutBaseProperty
 */
#define BASE_PROPERTY_NAME             1
#define BASE_PROPERTY_COMMENT          2
#define BASE_PROPERTY_LOCATION         3
#define BASE_PROPERTY_BAF_FILE         4
#define BASE_PROPERTY_BIF_FILE         5
#define BASE_PROPERTY_VIF_FILE         6
#define BASE_PROPERTY_LOG_FILE         7
#define BASE_PROPERTY_WORD_CHARS       8
#define BASE_PROPERTY_BAFFIT_ASE1      9
#define BASE_PROPERTY_BAFFIT_ASE2      10
#define BASE_PROPERTY_CONTAINER        11
#define BASE_PROPERTY_IS_XML           12
#define BASE_PROPERTY_IS_GRAPH         13
#define BASE_PROPERTY_CHI_WORDS        14
#define BASE_PROPERTY_USE_AUTO_REORG   15
#define BASE_PROPERTY_USE_AUDIT        16
#define BASE_PROPERTY_USE_LOG_DELETE   17
#define BASE_PROPERTY_PID_FIELD        18
#define BASE_PROPERTY_CHARSET          19
#define BASE_PROPERTY_LANGUAGE         20
#define BASE_PROPERTY_APPL_ID          21
#define BASE_PROPERTY_OUTPUT_FORMAT    22
#define BASE_PROPERTY_ENTRY_FORM       23
#define BASE_PROPERTY_RID_FIELD        24
#define BASE_PROPERTY_COUNTER_FIELD    25
#define BASE_PROPERTY_CHAR_FOLDING     26
#define BASE_PROPERTY_ENCRYPTED        27
#define BASE_PROPERTY_IS_CONNECTOR     28

/* Field property ids for use with
 * TdbGetFieldProperty and TdbPutFieldProperty
 */
 #define FIELD_PROPERTY_NAME               1
 #define FIELD_PROPERTY_NUMBER             2
 #define FIELD_PROPERTY_TYPE_NAME          3
 #define FIELD_PROPERTY_TYPE_NUMBER        4
 #define FIELD_PROPERTY_TYPE               5
 #define FIELD_PROPERTY_COPYRIGHT_FIELD    6
 #define FIELD_PROPERTY_INDEX_MODE         7
 #define FIELD_PROPERTY_IS_ORIG            8
 #define FIELD_PROPERTY_IS_RID_FIELD       9
 #define FIELD_PROPERTY_HAS_PATTERN        10
 #define FIELD_PROPERTY_HAS_FILE_REF       11
 #define FIELD_PROPERTY_IS_PID_FIELD       12
 #define FIELD_PROPERTY_HAS_DEFAULT        13
 #define FIELD_PROPERTY_IS_UNIQUE          14
 #define FIELD_PROPERTY_HAS_CONCORDANCE    15
 #define FIELD_PROPERTY_IS_COUNTER_FIELD   16
 #define FIELD_PROPERTY_IS_PART_FIELD      17
 #define FIELD_PROPERTY_COST               18
 #define FIELD_PROPERTY_MIN_ITEMS          19
 #define FIELD_PROPERTY_MAX_ITEMS          20
 #define FIELD_PROPERTY_COMMENT            21
 #define FIELD_PROPERTY_RESTRICTIONS       22
 #define FIELD_PROPERTY_LOAD_ASE           23
 #define FIELD_PROPERTY_INDEX_ASE          24
 #define FIELD_PROPERTY_ERROR_AT           25


/* Field group property ids for use with
 * TdbGetFieldGroupProperty and TdbPutFieldGroupProperty
 */
#define FGROUP_PROPERTY_NAME     1
#define FGROUP_PROPERTY_NUMBER   2
#define FGROUP_PROPERTY_TYPE     3
#define FGROUP_PROPERTY_TYPENAME 4
#define FGROUP_PROPERTY_COMMENT  5
#define FGROUP_PROPERTY_FIELDS   6
#define FGROUP_PROPERTY_SIZE     7


/* Field group types */
#define FGROUP_TYPE_STRUCTURAL 0
#define FGROUP_TYPE_TUPLE      1


/* Cluster design properties  for use with
 * TdbGetClusterProperty and TdbPutClusterProperty
 */
#define CLUSTER_PROPERTY_NAME          1
#define CLUSTER_PROPERTY_COMMENT       2
#define CLUSTER_PROPERTY_MEMBERCOUNT   3


 /* Template design properties for use with
  * TdbGetTemplateProperty and TdbPutTemplateProperty
  */
#define TEMPLATE_PROPERTY_TYPE    100

/* Template design types */
#define TEMPLATE_TYPE_DATABASE    1
#define TEMPLATE_TYPE_FIELDGROUP  2

/* Manifest constants for use with TdbGetConf */
#define CONF_PRIVILEGED    1
#define CONF_GROUP         2
#define CONF_NONPRIVILEGED 4
#define CONF_ALL ( CONF_PRIVILEGED | CONF_GROUP | CONF_NONPRIVILEGED )

/* Manifest constants for use with TdbSortRecordsEx */
#define SORT_MERGE         1
#define SORT_NO_MERGE      2
#define SORT_PARTS         4

/* Manifest constants for use with TdbPurgeTokens */
#define TOKEN_PURGE_ALL       0
#define TOKEN_PURGE_REFRESH     1
#define TOKEN_PURGE_ACCESS    2
#define TOKEN_PURGE_CREATED   4     // NB: not valid by itself!
#define TOKEN_PURGE_TIMERANGE 128   // NB: not valid by itself!
#define TOKEN_PURGE_REFRESH_BY_TIMERANGE (TOKEN_PURGE_REFRESH|TOKEN_PURGE_TIMERANGE)
#define TOKEN_PURGE_ACCESS_BY_TIMERANGE (TOKEN_PURGE_ACCESS|TOKEN_PURGE_TIMERANGE)
#define TOKEN_PURGE_CREATED_BY_TIMERANGE (TOKEN_PURGE_CREATED|TOKEN_PURGE_TIMERANGE)

#endif

