/*******************************************************************************
 *
 * Filename : tripapi_types.h
 * Copyright: infinIT Services GmbH
 * Purpose  : Type definitions for the TRIP API
 *
 */

#ifndef __TRIPAPI_TYPES_H__
#define __TRIPAPI_TYPES_H__

#include <time.h>
#include "tripapi_defs.h"

#if !defined(__RPCNDR_H__)
    /* This type is defined in rpcndr.h */
    typedef unsigned char boolean;
#endif


/**
 * @brief Handle used for record controls, cursors, etc.
 * @deprecated Deprecated by TdbHandle
 */
typedef void* tdb_handle;

/**
 * @brief Handle used for record controls, cursors, etc.
 */
typedef void* TdbHandle;

/**
 * Unsigned character
 */
typedef unsigned char Char;

/**
 * For 64-bit integer types
 */
#if defined(_MSC_VER) && _MSC_VER < 1600
	/* stdint.h missing in older versions of Microsoft Visual C++ */
	typedef signed __int8 int8_t;
	typedef signed __int16 int16_t;
	typedef signed __int32 int32_t;
	typedef unsigned __int8 uint8_t;
	typedef unsigned __int16 uint16_t;
	typedef unsigned __int16 char16_t;
	typedef unsigned __int32 uint32_t;
	typedef signed __int64 int64_t;
	typedef unsigned __int64 uint64_t;
#else
	#include <stdint.h>
#endif


/**
 * @brief Shell/DAO interface record
 *
 * @note New programs should not use this data type but rather use
 * TdbStartTrip() / TdbStopTrip() / TdbExecuteCcl().
 */
typedef struct shell_dao_rec
{
    char    shin[400],      /*!< CCL command input */
            shout[400],     /*!< Parsed CCL output */
            dao[400];       /*!< DAO equivalent of shout */
    int     shinlen,   /*!< Length of CCL command input */
            shoutlen,  /*!< Length of parsed CCL output */
            daolen,    /*!< Length DAO equivalent */
            shdind;    /*!< Shell/DAO indicator */
} shell_dao_rec, ShellDaoRecord;


/**
 * @brief Structure definition for a database
 */
typedef struct base_spec_rec
{
    char    baf_file[132],    /*!< Base file name */
            bif_file[132],    /*!< Primary index filename */
            vif_file[132],    /*!< Secondary index filename */
            log_file[132],    /*!< Optional transaction log filename */
            word_chars[132],  /*!< Any "special" searchable characters */
            baffit_ase1[32],  /*!< ASE called by Baffit before storing record */
            baffit_ase2[32],  /*!< ASE called by Baffit after storing record */
            container[16],    /*!< Name of classification container used by DB */
            _pad1[6];         /*!< RESERVED for future use */
    boolean graph;            /*!< Is this a graph database? */
    boolean xml;              /*!< Is this an XML-enabled database? */
    char    chi_words;        /*!< Chinese word segmentation option (0,1,2) */
    boolean auto_reorg,       /*!< Does this database allow automatic reorganization of indexes? */
            audit,            /*!< Does this database have an audit trail? */
            log_delete;       /*!< Should deletes be written to the transaction log? */
    char    pid_field[16],      /*!< Name of the part name field */
            char_set[4],        /*!< Character set used by the database */
            language[3];        /*!< Language used by the database */
    char    _pad2,              /*!< RESERVED for future use */
            appl_id[16],        /*!< Application-specific ID information */
            output_format[16],  /*!< Name of default output format */
            entry_form[16],     /*!< Name of default TRIPclassic entry form */
            rid_field[16],      /*!< Name of the record name field */
            counter_field[16],  /*!< Name of the record number field */
            char_folding[3];    /*!< Character folding class */
    char    encrypted;          /*!< Encryption state mask (reserved for future use) */
    char    base_desc[256],     /*!< Description of the database */
            submit_que[256];    /*!< DEPRECATED. Queue used for batch job submissions */
    int     submit_opt;         /*!< DEPRECATED. Batch job submission options */
    char    submit_ase1[32],    /*!< DEPRECATED. ASE called before batch job submission */
            submit_ase2[32];    /*!< DEPRECATED. ASE called after batch job submission */
} base_spec_rec, BaseSpecRecord;


/**
 * @brief Definition of sentence / paragraph boundary markers
 */
typedef struct text_sepa_rec
{
    boolean sent_check,         /*!< Are sentence numbers to be generated? */
            para_check,         /*!< Are paragraph numbers to be generated? */
            sent_end_req,       /*!< Are sentence end markers required for end of paragraph? */
            sent_beg_req;       /*!< Are sentence begin markers required for begin of paragraph? */
    char    sent_beg_spec[8],   /*!< Sentence begin specification: UB */
            sent_sep_spec[8],   /*!< Sentence end specification: SN */
            para_sep_spec[8],   /*!< Paragraph end specification: 2N */
            sent_end_chars[16], /*!< Sentence end characters, class E */
            spec_beg_chars[32], /*!< Sentence begin characters, class B */
            ignore_chars[64];   /*!< Ignore characters, class I */
} text_sepa_rec, TextSepaRecord;


/**
 * @brief Definition of a field's design properties
 */
typedef struct field_spec_rec
{
    char    field_name[16],     /*!< Name of the field */
            field_type[16],     /*!< Type of the field */
            copyright_field[16];/*!< Name of the copyright holder field */
    int     index;              /*!< Index mode for field */
    boolean orig,               /*!< Original layout? */
            rid_field,          /*!< Record name field? */
            pattern,            /*!< Is there a pattern-based restriction? */
            file_ref,           /*!< Is there a database-based restriction? */
            pid_field,          /*!< Part name field? */
            default_value,      /*!< Is there a default value for this field? */
            unique,             /*!< Field values must be unique */
            concordance,        /*!< Used for classification */
            counter_field,      /*!< Record number field? */
            part_field;         /*!< Part field? */
    int     field_number,       /*!< Field number */
            cost,               /*!< Cost of viewing the field */
            min_items,          /*!< Minimum number of allowable subfields / paragraphs */
            max_items;          /*!< Maximum number of allowable subfields / paragraphs */
    char    comment[256],       /*!< Comment, description */
            restrictions[256],  /*!< Any restrictions, as specified by flags above */
            baffit_ase[32],     /*!< ASE called by Baffit when loading field */
            scanit_ase[32];     /*!< ASE called by Sciffit when indexing field */
    int     error_at;           /*!< Error flag */
} field_spec_rec, FieldSpecRecord;


/**
 * @brief Database access record
 *
 * Each element of the database access record ( except the scope fields ) are
 * made up using a bit mask as follows :
 *
 * | Bit Nr | Description |
 * | ------ | ----------- |
 * | bit 0 | read access |
 * | bit 1 | write access |
 *
 * Thus, to give full read only access, set \c all_access to 1. To give full
 * read and write access, set \c all_access" to 3. If any bits in \c sel_access
 * are set, the \c field_access array is read, each element of the array
 * corresponding to the access rights to a particular field in the database.
 */
typedef struct base_access_rec
{
    char    all_access,         /*!< Unrestricted access to all fields */
            sel_access,         /*!< Restrict access to certain fields */
            field_access[1000], /*!< Fields to be given access */
            read_scope[400],    /*!< CCL command defining read scope */
            write_scope[400];   /*!< CCL command defining write scope */
} base_access_rec, BaseAccessRecord;


/**
 * @brief Structure used for defining clusters of databases
 */
typedef struct cluster_spec_rec
{
    int     db_count;                   /*!< Number of databases in the cluster */
    char    db_names[30][16],           /*!< Names of databases in the cluster */
            cluster_desc[256];          /*!< Comment, description */
} cluster_spec_rec, ClusterSpecRecord;


/**
 * @brief Structure used for defining user properties
 */
typedef struct user_profile_rec
{
    char    user_name[60],          /*!< TRIP username */
            company[60],            /*!< Company name */
            address1[60],           /*!< Address line 1 */
            address2[60],           /*!< Address line 12 */
            phone[20],              /*!< Telephone number */
            start_record[33],       /*!< Procedure automatically run upon login */
            dateform_sep[2];        /*!< Default date separators */
    int     dateform_nr,            /*!< ID of default date format */
            start_module;           /*!< Start module -- TRIPclassic only */
    boolean user_manager,           /*!< Is user a UM? */
            file_manager,           /*!< Is used a FM? */
            os_entry;               /*!< Bypass password checks? */
    int     group_count,            /*!< Nr of groups to which user belongs */
            group_ids[100];         /*!< CONTROL record IDs for group records */
} user_profile_rec, UserProfileRecord;


/**
 * @brief Structure used to define hit points in TdbGetFieldInfo()
 */
typedef struct hit_point
{
    int     iLineNr,                /*!< Line number of field on which hit occurs */
            iHitStart,              /*!< Column offset in line */
            iHitLength;             /*!< Length of hit */
} hit_point, HitPoint, *TdbHitVector;


/* Obsolete data structure definitions */
typedef int qbe_vector_entry, QbeVectorEntry, *TdbQbeVector;
typedef int tom_spec_rec;


/**
 * @brief Structure used with TdbImport and TdbExport
 */
typedef struct
{
   TdbHandle    record_control;     /*!< Record to operate on */
   TdbHandle    cursor;             /*!< Field to operate on */
   TdbHandle    filter_address;     /*!< Filter ASE routine adresss */
   char         filter_name[32];    /*!< Filter ASE routine name */
   char         filter_lib_env[32]; /*!< OBSOLETED/UNUSED */
   void*        buffer;             /*!< Data buffer, ASE-specific */
   int          buffer_length;      /*!< Data buffer length, ASE-specific */
   char*        filter_arguments;   /*!< Arguments string, ASE-specific */
   int          arg_length;         /*!< Arguments string length, ASE-specific */
   int          filter_options;     /*!< Filter ASE options */
   int          tdb_options;        /*!< Data transfer options */
   int          blockno;            /*!< Block number, ASE-specific */
   int          errorcode;          /*!< Filter-ASE specific return code */
   char         errortext[256];     /*!< Filter-ASE specific error message */
} filter_data, FilterData;


/**
 * @brief Database integrity rule definition for a field.
 *
 * The \c on_delete and \c on_update values can have the following values:
 *
 * | Symbol | Value | Description |
 * | ------ | ------| ----------- |
 * | INTEGRITY_RESTRICT | 1 | Prohibit the action as long as there are dependent records |
 * | INTEGRITY_CASCADE | 2 | For \c on_update, update also the dependent record. For \c on_delete, delete also dependent record. |
 * | INTEGRITY_NOACTION | 3 | Allow the operation to proceed |
 * | INTEGRITY_SETNULL | 4 | Set the dependent record's field to NULL |
 * | INTEGRITY_SETDEFAULT | 5 | Set the dependent record's field to its defaut value |
 *
 */
typedef struct integrity_rule
{
   int     type;            /*!< 0 for source of link, 1 for target of link */
   int     on_delete;       /*!< Behavior when master records are deleted */
   int     on_update;       /*!< Behavior when master records are updated */
   char*   key_field;       /*!< Field name in dependent database */
   char*   link_database;   /*!< Master database */
   char*   link_field;      /*!< Field in master database */
} integrity_rule, IntegrityRule;


/**
 * @brief Database integrity rule specification
 *
 * When returned from TdbGetIntegrityRules(), instances of this struct
 * are allocated to fit the actual number of integrity rules in the database.
 * The minimum size is always \c sizeof(base_integrity_spec), but if there are
 * two integrity rules, the size is:
 \code{.cpp} sizeof(int) + sizeof(integrity_rule)*2 \endcode
 */
typedef struct base_integrity_spec
{
    int nr;                  /*!< Number of rules in \p rules */
    integrity_rule rules[1]; /*!< Integrity rules array */
} base_integrity_spec, BaseIntegritySpec;

/**
 * @brief Token purge callback function
 * @see TdbPurgeTokens()
 */
typedef void (*FP_TdbPurgeTokensCallback)(void* userdata,
    const char* user, const char* tag, time_t created_at,
    time_t access_expires_at, time_t refresh_expires_at);

#endif

