#!/bin/bash
#
#  TRIPcof uninstallation script
#  Copyright (C) Smaser AG
#
#  This script uninstalls TRIPcof by removing symbolic links and
#  various system configuration done during install. No files will
#  actually be removed during this process.
#

UNINST_ISSUES=0
OS=`uname`

WHOM=`(whoami) 2> /dev/null`
if [ $? -ne 0 ]; then
   if [ -f /usr/ucb/whoami ]; then
      WHOM=`/usr/ucb/whoami`
   fi
fi

# This script needs root privileges unless we're on macOS.
if [ "$OS" != "Darwin" ]; then
   if [ "$WHOM" != root ]; then
      echo ""
      echo "You must be root to run this script!"
      echo "Exiting the uninstallation procedure"
      echo ""
      exit 1
   fi
fi

if [ `echo "X\c"` = X ]; then
   C1=
   C2='\c'
elif [ `echo -n X` = X ]; then
   C1='-n'
   C2=
else
   C1=
   C2=
fi


yesno_valid()
{
   inputval=$1
   if [ "$inputval" == "y" ] || [ "$inputval" == "Y" ] || \
      [ "$inputval" == "n" ] || [ "$inputval" == "N" ]
   then
      return 0
   elif [ "$2" != "" ] && [ "$inputval" = "?" ]; then
      return 0
   else
      return 1
   fi
}


yesno()
{
   yesno_result=-1
   yesno_prompt=$1
   yesno_default=$2
   yesno_help=$3
   if [ "$yesno_default" = "Y" ]; then
      yesno_choice="[Y|n"
   elif [ "$yesno_default" = "N" ]; then
      yesno_choice="[y|N"
   else
      yesno_choice="[y|n"
   fi
   if [ "$yesno_help" != "" ]; then
      yesno_choice="$yesno_choice|?"
   fi
   yesno_choice="$yesno_choice]"
   doloop=0
   while [ "$doloop" = "0" ]
   do
      echo $C1 "$yesno_prompt $yesno_choice $C2"
      read answ
      if [ "$answ" != "" ] ; then
         if yesno_valid $answ "$yesno_help"; then
            printf "\n****** Answer: $answ\n"
         else
            continue
         fi
      elif [ "$answ" = "" ] && [ "$yesno_default" != "" ]; then
         printf "\n****** Answer: default [= $yesno_default]\n"
         answ=$yesno_default
      else
         continue
      fi
      case $answ in
         \?)
            if [ "$yesno_help" != "" ]; then
               eval $yesno_help
           fi
            ;;
         Y|y)
            yesno_result=0
            doloop=1
            ;;
         N|n)
            yesno_result=1
            doloop=1
            ;;
         *)
            ;;
      esac
   done
   return $yesno_result
}


CORRECT_INSTALL_DIR=0
echo ""
if [ ! -d bin ]; then
   if [ -f uninstall_cof.sh ]; then
      cd ..
      if [ ! -d bin ]; then
         CORRECT_INSTALL_DIR=1
      fi
   else
      CORRECT_INSTALL_DIR=1
   fi
fi
if [ $CORRECT_INSTALL_DIR -ne 0 ]; then
   echo "Please run this script from one of the following directories:"
   echo ""
   echo "  <extract_dir>"
   echo "  <extract_dir>/$VERS_DIR"
   echo "  <extract_dir>/$VERS_DIR/bin"
   echo ""
   echo "where <extract_dir> is the directory from which you issued the"
   echo "tar command to read the distribution media."
   echo "Exiting the installation procedure"
   echo ""
   exit 1
fi

echo "This script will uninstall TRIPcof from your machine. Your data will be"
echo "left intact as will the TRIPcof files themselves."
echo ""
if ! yesno "Do you wish to continue?" Y ; then
   echo ""
   echo "Exiting the uninstallation procedure."
   echo ""
   exit 1
fi
echo ""
COF_INSTALLDIR=`pwd`


echo "Removing symbolic links"
LINK_DIR=/usr/local/trip/cof
if [ -e $LINK_DIR/lib ]; then
   rm -fv $LINK_DIR/lib 
   if [ $? -ne 0 ]; then
      UNINST_ISSUES=`expr $UNINST_ISSUES + 1`
   fi
fi
if [ -e $LINK_DIR/conf ]; then
   rm -fv $LINK_DIR/conf 
   if [ $? -ne 0 ]; then
      UNINST_ISSUES=`expr $UNINST_ISSUES + 1`
   fi
fi
echo ""

listRemoveItem()
{
   echo $1 | tr ',' '\n'| while read sym
   do
      if [ "x$sym" != "x$2" ]; then
         echo ,$sym | tr -d '\n'
      fi;
   done | cut -d, -f2-
} 

TDBS_CONF=`readlink -f /usr/local/trip/sys/conf/tdbs.conf 2> /dev/null`
if [ "$TDBS_CONF" = "" ] || [ ! -f "$TDBS_CONF" ]; then
   echo "WARNING: Could not find the TRIPsystem tdbs.conf configuration file."
   echo "         No TRIPsystem deconfiguration needed."
else

   echo "Backing up the TRIPsystem tdbs.conf configuration file"
   CBKPID=`date '+%Y%m%d%H%M%S'`
   cp $TDBS_CONF $TDBS_CONF.cof.$BKPID.bklp
   if [ $? -ne 0 ]; then
      echo "ERROR: Backup of tdbs.conf failed."
      echo "       TRIPcof uninstallation is incomplete."
      echo ""
      exit 1
   fi
   echo ""

   echo "Removing TRIPcof entries from the TRIPsystem tdbs.conf configuration file"
   TMPCFG=`mktemp`
   
   cat $TDBS_CONF | (

      IFS=\\
      while read LINE; do

         RCSSYM=`echo $LINE|cut -d= -f1`
         RCSVAL=`echo $LINE|cut -d= -f2-`

         if [ "$RCSSYM" = "COF_ASE_FIFI" ]; then
           echo "  Removed COF_ASE_FIFI"
           continue 
         fi

         if [ "$RCSSYM" = "COF_CONFIG_DIR" ]; then
           echo "  Removed COF_CONFIG_DIR"
           continue 
         fi

         if [ "$RCSSYM" = "TDBS_ASELIBS" ]; then
            ASELIBS2=`listRemoveItem "$RCSVAL" COF_ASE_FIFI | tr -d '[[:space:]]'`
            if [ "$ASELIBS2" != "" ]; then
               echo $RCSSYM=$ASELIBS2 >> $TMPCFG
               echo "   Edited TDBS_ASELIBS"
            else
               echo "  Removed TDBS_ASELIBS"
            fi
            continue
         fi

         echo $LINE >> $TMPCFG

      done
   )


   # Introduce changes
   cp $TMPCFG $TDBS_CONF
   if [ $? -ne 0 ]; then
      echo ""
      echo "WARNING: Unable to save changes to the TRIPsystem tdbs.conf configuration file"
      echo "         TRIPcof uninstallation is incomplete."      
      UNINST_ISSUES=`expr $UNINST_ISSUES + 1`
   fi
   rm $TMPCFG

fi


if [ "$OS" = "Linux" ]; then
   $COF_INSTALLDIR/bin/chfapolicy_cof.sh -r -p $COF_INSTALLDIR
fi

echo ""
if [ $UNINST_ISSUES -eq 0 ]; then
   echo "TRIPcof uninstalled"
fi

exit $UNINST_ISSUES

