/** \brief Main include file for the File Filter API in TRIPviewX.
 *  \file fifi.h tvxapi.h
 *  \copyright Copyright 2018 infinIT Services GmbH
 */

#ifndef __FILE_FILTER_API_H__
#define __FILE_FILTER_API_H__


/*! \mainpage File Filter Adapter API for C
 * \par About this manual
 *
 * The File Filter adapter API enables TRIPcof to use third party file
 * filters such as Microsoft IFilter and Apache Tika via a single API. The 
 * functionality of these filters are exposed as a file conversion 
 * paradigm. 
 *
 * This API also allows authoring of adapters for any third party file
 * filter techology. This reference manual describes the API with which
 * this can be done using the C programming language.
 *
 * See the \link fifi.h \endlink documentation for a reference of the main
 * part of the API.
 *
 * Please refer to the TRIPcof Operating Manual for more detailed 
 * information.
 */


#define FIFI_API_INTERFACE_VERSION 1       //!< The version of the FIFI API

#define FIFI_ERR_SUCCESS 0                 //!< Return code indicating success

/** @name File Filter Error Codes
 *  These macros declare standardized file filter API errors that the 
 *  file filter API itself may return.
 */
//@{

#define FIFI_ERR_BASE            52000                 //!< Base for file filter API errors
#define FIFI_ERR_INVALID_ARG     (FIFI_ERR_BASE+1)     //!< An invalid argument was passed to a function
#define FIFI_ERR_NOT_FOUND       (FIFI_ERR_BASE+2)     //!< Specified item could not be found
#define FIFI_ERR_LOAD_LIBRARY    (FIFI_ERR_BASE+3)     //!< The library for the specified adapter could not be loaded
#define FIFI_ERR_LOAD_FUNCTION   (FIFI_ERR_BASE+4)     //!< An adapter function could not be loaded
#define FIFI_ERR_PROXY_FAILURE   (FIFI_ERR_BASE+5)     //!< Adapter proxy error
#define FIFI_ERR_NOT_SUPPORTED   (FIFI_ERR_BASE+6)     //!< The call is not supported by the adapter according to its manifest
#define FIFI_ERR_NOT_IMPLEMENTED (FIFI_ERR_BASE+7)     //!< The function is not implemented by the adapter
#define FIFI_ERR_DISABLED        (FIFI_ERR_BASE+8)     //!< The adapter has been disabled in its configuration file
#define FIFI_ERR_BUFFER_TOO_SMALL (FIFI_ERR_BASE+10)   //!< Supplied buffer was too small to hold the requested value
#define FIFI_ERR_INCOMPATIBLE    (FIFI_ERR_BASE+11)    //!< An adapter library is of an incompatible version
#define FIFI_ERR_UNKNOWN         (FIFI_ERR_BASE+999)   //!< Unknown error

//@}


/** @name Adapter Error Codes
 *  These macros declare standardized adapter errors that any adapter
 *  may return.
 */
//@{
 
#define FIFI_AERR_BASE          2000                   //!< Base for standard adapter errors
#define FIFI_AERR_INVALID_ARG   (FIFI_AERR_BASE+1)     //!< An invalid argument was passed to an adapter function
#define FIFI_AERR_INITIALIZE    (FIFI_AERR_BASE+3)     //!< An adapter failed to initialize
#define FIFI_AERR_NOT_SUPPORTED (FIFI_AERR_BASE+6)     //!< The requested operation was not supported by the adapter
#define FIFI_AERR_OUTPUT_TYPE   (FIFI_AERR_BASE+10)    //!< The requested output type is not supported by the adapter
#define FIFI_AERR_OPEN          (FIFI_AERR_BASE+11)    //!< The adapter failed to open the source document for conversion
#define FIFI_AERR_COMOBJECT     (FIFI_AERR_BASE+12)    //!< An error occurred in an underlying COM object
#define FIFI_AERR_CONVERT       (FIFI_AERR_BASE+20)    //!< Conversion to requested target format failed
#define FIFI_AERR_HIGHLIGHT     (FIFI_AERR_BASE+21)    //!< Error while generating hit highlighting
#define FIFI_AERR_THREAD_UNABLE (FIFI_AERR_BASE+30)    //!< Current thread unable to perform requested operation
#define FIFI_AERR_UNKNOWN       (FIFI_AERR_BASE+999)   //!< Unknown error

//@}


#ifdef __cplusplus
extern "C" {
#endif

/** @brief File filter adapter handle
 */
typedef void* HADAPTER;

	
/** @brief Enumeration of the kinds of information it is possible
 *         to request from an adapter
 *  @see FIFIAdapterInfo()
 */
typedef enum
{
   FIFI_ADAPTER_LABEL=1,      //!< Use with FIFIAdapterInfo to get the full name of the adapter (string)
   FIFI_ADAPTER_MANUFACTURER=2, //!< Use with FIFIAdapterInfo to get the manufacturer of the adapter (string)
   FIFI_ADAPTER_ISENABLED=3,    //!< Use with FIFIAdapterInfo to check if the adapter is enabled (int)
   FIFI_ADAPTER_PRIORITY=4,     //!< Use with FIFIAdapterInfo to check the priority of the adapter (int)
   FIFI_ADAPTER_INTERFACE_VERSION=5  //!< Use with FIFIAdapterInfo to check the FIFI API interface version the adapter was built for (current is FIFI_API_INTERFACE_VERSION)
}
EFifiAdapterInfo;


/** @brief Enumeration of the standard types of conversion output formats 
 *         that the File Filter API supports.
 */
typedef enum
{
   FIFI_OUTPUT_ORIG=0,		//!< The adapter can output original format
   FIFI_OUTPUT_TEXT=1,		//!< The adapter can convert files into plain text 
   FIFI_OUTPUT_HTML=2,		//!< The adapter can render files as HTML
   FIFI_OUTPUT_MHT =3,		//!< The adapter can render files as web archive files
   FIFI_OUTPUT_PDF =4,		//!< The adapter can render files as PDF
   FIFI_OUTPUT_XML =5		//!< The adapter can convert files into XML
}
EAdapterOutputType;

/** @brief The highest number of the EAdapterOutputType enumeration
 */
#define FIFI_MAX_ADAPTER_OUTPUT_TYPE 5

/** @brief Represents a single hit location.
 *  @see struct HitInfo
 *
 * Hit locations may be used with conversion target formats such as HTML, MHT
 * and PDF, e.g. for the purpose of visualizing search engine hit locations.
 *
 * Note that a correctly placed hit highlighting will most likely require that
 * the text from which hit locations are taken has been extracted with the same
 * technology that is used to render the conversion (to HTML, etc). This is
 * because different technologies may extract the same text in slightly 
 * different manners, making the hit word offset technology-dependent.
 */
typedef struct
{
	int offset;		//!< Offset to hit location
	int length;		//!< Length of hit
}
HitLocation;

typedef enum
{
	HITUNIT_BYTE = 0,   //<! Hit offset and length is counted in bytes
	HITUNIT_CHAR = 1    //<! Hit offset and length is counted in characters
}
EHitUnit;

/** @brief Represents a word hit within a document.
 * 
 * Hit words may be used instead of hit locations by the adapter that
 * renders output with hit highlighting. It will not match exactly the
 * locations in TRIP, but will use the same words (and word forms) that 
 * were found.
 * 
 * Words are always encoded in UTF-8 when running in Unicode mode. Otherwise
 * the current TRIP session character set is used.
 */
typedef struct HitWord
{
	char* word;	//!< Hit word to highlight
	int length;	//!< Length of the word in bytes
}
HitWord;


/** @brief Represents all hit locations within a document.
 *  @see struct HitLocation
 *
 * Hit locations may be used with conversion target formats such as HTML, MHT
 * and PDF, e.g. for the purpose of visualizing search engine hit locations.
 */
	typedef struct
{
	EHitUnit hitUnit;    //!< The unit in which hits are specified (byte or character)
	int hitCount;        //!< Number of hits
	int hitsSize;        //!< Allocated length of the hits array
	HitLocation* hits;   //!< Array of offset and length for each hit
	int wordCount;       //!< Number of words hit
	int wordsSize;       //!< Allocated length of the words array
	HitWord* words;     //!< Array of words hit
}
HitInfo;

typedef enum 
{
	FIFI_CONVERT_NO = -1,   //!< Adapter cannot convert specified file to requested target format.
	FIFI_CONVERT_YES = 0,   //!< Adapter can specified file to requested target format.
	FIFI_CONVERT_MAYBE = 1  //!< Adapter can possibly convert specified file to requested target format.
}
EConvertAbility;

	
/** @name Callback Functions
 */
//@{
	
/** @brief Function pointer type for a function to receive in-memory conversion results.
 *  @see FIFISetConversionCallback()
 */
typedef void (* FP_FIFIConversionCallback ) ( const char* mimetype, const char* filename, unsigned char* data, int size, void* userdata );
	
/** @brief Function pointer type for a function to receieve extracted file and document properties.
 *  @see
 *
 *  This function type is typically only used together with a conversion into text.
 */
typedef void (* FP_FIFIPropertyCallback ) ( unsigned char* name, int nameSize, unsigned char* value, int valueSize, void* userdata );
	
/** @brief Function pointer type for a callback function used to read file data.
 *  @see FIFIOpenData()
 *
 * To reset the read to the start of the file data, this function may be called
 * with the size parmeter set to zero and the data parameter set to NULL:
 */
typedef int (* FP_FIFIRead ) ( unsigned char* data, int size, void* userdata );

#ifndef __TRIPCOF_ERROR_H__
/** @brief Function pointer type for a logging callback function.
 *  @param userdata Opaque data pointer
 *  @param loglevel Log level 1=fatal, 2=error, 3=warning, 4=info, 5=debug
 *  @param errorcode Error code; use for errors and warnings
 *  @param msg Log message
 */
typedef int (* FP_FIFILog ) ( void* userdata, int loglevel, int errorcode, const char* msg );
#endif

//@}



/** @name API Initialization and Information Functions
 *
 *  These functions are used for the initialization of the File Filter
 *  API and to get information about what adapters are available.
 */
//@{

/** @brief Initialize the file filter adapter API.
    @param confFileName Name of configuration file to load.
    @returns 0 on success and -1 on failure.
 */
int FIFIInitialize(const char* confFileName);


/** @brief Uninitialize all loaded adapters and release all other held resources.
 *  @returns 0 on success and -1 on failure.
 * 
 *  After this function has been called, none of the FIFI-functions
 *  will be valid to call in the current process for the duration of
 *  its lifetime.
 */
int FIFIShutdown();

/** @brief Enumerate the known adapters
 *  @param iter Iteration handle. Pointer to integer assigned to 0 (zero) for the first item.
 *  @param adapterId Buffer to receieve the id string of a connector
 *  @param bufferSize The size of the 'adapterId' buffer.
 *  @return 0 (zero) on success and non-zero on failure.
 *
 *  If the supplied adapterId buffer is too small, this function will
 *  return FIFI_ERR_BUFFER_TOO_SMALL. When there are no more adapter
 *  id strings to be fetched, this function returns FIFI_ERR_NOT_FOUND.
 */
int FIFIEnumerateAdapters(int* iter, char* adapterId, unsigned int bufferSize);

/*! \brief Set callback function to receieve log output from file filter adapters.
    \param cbFunc Callback function
    \param loglevel The current log level of the process.
    \param userdata A pointer to any data the application wishes to be passed along to the callback function.
    \returns 0 (zero) on success and non-zero on failure.

    If logging is to be used, this function should be called immediately after 
    or immediately before FIFIInitialize.
 */
int FIFISetLogCallback ( FP_FIFILog cbFunc, int loglevel, void* userdata);


//@}


/** @name Error Handling
 * 
 * These functions are used to get and set error information.
 *
 */
//@{

/** @brief Retrieves the last error information.
 *  @param errBuf Buffer to receive the error message.
 *  @param bufSize The total allocated size of the memory area that errBuf points to.
 *  @returns The last error code.
 */
int FIFIGetLastError ( char* errBuf, int bufSize );


/** @brief Assigns error information.
 *  @param rc The error code to set.
 *  @param msg The error message to set.
 *
 *  This function should only be used by file filter adapters.
 */
int FIFISetLastError ( int rc, const char* msg, ... );

//@}


/** @name Adapter Loading and Information 
 * 
 * These functions are used load adapters and request information about them.
 */
//@{
	
/** @brief Loads an adapter with a certain ID.
    @param adapterId The ID of the adapter to load
    @param preinit Pass non-zero to pre-initialize the adapter for use
    @returns 0 on success and -1 on failure.

    If the adapter is already loaded and initialized, this function 
    does not do anything. If the adapter is loaded but not initialized
    and the preinit parameter is non-zero, the adapter will be told to
    initialize.
 */
int FIFILoadAdapter(const char* adapterId, int preinit);

/** @brief Request information about an adapter
    @param adapterId The ID of the adapter to query
    @param infotype The kind of information to retrieve
    @param s String buffer to receieve the information (for string-type info
    @param i Pointer to 32-bit signed integer to receieve the information (for int-type info)
    @return 0 on success and -1 on failure.
 */
int FIFIAdapterInfo(const char* adapterId, EFifiAdapterInfo infotype, char* s, int* i);

//@}


/** @name Hit information management
 */
//@{

/** @brief Allocates a hit info structure for a specified number of hits
    @param hitUnit The unit in which hit offset and length will be specified.
	@returns A pointer to a HitInfo structure
 */
HitInfo* FIFIAllocateHits(EHitUnit hitUnit);

/** @brief Add information about a hit to a HitInfo structure.
    @param hitInfo Pointer to a HitInfo structure pointer returned by FIFIAllocateHits
    @param offset Offset to hit
    @param length Length of hit

    The HitInfo area may be reallocated by this function, so that is
    why the first argument is a pointer-pointer and not a regular pointer
    to the HitInfo structure.
 */
void FIFIAddHitLocation(HitInfo** hitInfo, int offset, int length);

/** @brief Add information about a hit to a HitInfo structure.
    @param hitInfo Pointer to a HitInfo structure pointer returned by FIFIAllocateHits
    @param word Hit word
    @param length Length of the word or -1 to calculate using strlen

    The HitInfo area may be reallocated by this function, so that is
    why the first argument is a pointer-pointer and not a regular pointer
    to the HitInfo structure.
 */
void FIFIAddHitWord(HitInfo** hitInfo, const char* word, int length);

/** @brief Deallocates a hit info structure allocated with FIFIAllocateHits
 *  @param hitInfo Pointer to a HitInfo structure returned by FIFIAllocateHits
 */
void FIFIFreeHits(HitInfo* hitInfo);

//@}


/** @name Adapter Proxy Functions
 *
 * These are proxies for functions that individual 
 * file filter adapter can implement. Using the File Filter API for
 * conversion (e.g. to extract text) requires use of these functions.
 */
//@{

/** @brief Initialize the adapter for use. Call once per process.
	@param adapterId The ID of the adapter to initialize.
    @returns 0 (zero) on success and non-zero on failure.

	 The initialization function is optional for adapters to implement. If the adapter
	 does not implement an initialization function, this function will return the
	 FIFI_ERR_NOT_IMPLEMENTED error code. Applications must not regard this return code
	 as an error.

	 Best practice for checking if an error really has occurred when calling this 
	 function is to check the return code from the FIFIGetLastError function. If it
	 is FIFI_ERR_SUCCESS, then the application may continue its processing.
 */
int FIFIInitializeAdapter (const char* adapterId );

/** @brief Uninitialize the adapter.
	@param adapterId The ID of the adapter to uninitialize.
    @returns 0 (zero) on success and non-zero on failure.

	 Do not call this function unless FIFIInitializeAdapter has succeeded
	 for the specified adapter.

	 The uninitialization function is optional for adapters to implement. If the adapter
	 does not implement an uninitialization function, this function will return the
	 FIFI_ERR_NOT_IMPLEMENTED error code. Applications must not regard this return code
	 as an error.

	 Best practice for checking if an error really has occurred when calling this 
	 function is to check the return code from the FIFIGetLastError function. If it
	 is FIFI_ERR_SUCCESS, then the application may continue its processing.
 */
int FIFIUninitializeAdapter (const char* adapterId );

/** @brief Assign a runtime setting (a.k.a. "property") to the adapter
	@param adapterId The ID of the adapter to assign a property for
	@param name Name of property
	@param value Property value
    @returns 0 (zero) on success and non-zero on failure.

	 This function is optional for adapters to implement. If the adapter does not
	 implement a function for assigning an adapter property, this function will return the
	 FIFI_ERR_NOT_IMPLEMENTED error code. Applications must not regard this return code
	 as an error.

	 Best practice for checking if an error really has occurred when calling this 
	 function is to check the return code from the FIFIGetLastError function. If it
	 is FIFI_ERR_SUCCESS, then the application may continue its processing.
 */
int FIFISetAdapterProperty( const char* adapterId, const char* name, const char* value );

/** @brief Set the desired encoding for the text produced by the adapter.
	@param adapterId The ID of the adapter to set the encoding for.
	@param encoding The name of the encoding to set.
    @returns 0 (zero) on success and non-zero on failure.

	 The default encoding is UTF8. Adapter implementors must
	 take measures to assure this.
 */
int FIFISetTextEncoding( const char* adapterId, const char* encoding );

/** @brief Set callback function to receive conversion results.
	@param adapterId The ID of the adapter for which to set callback function for.
	@param cbFunc Callback function
	@param userdata A pointer to any data the application wishes to be passed along to the callback function.
    @returns 0 (zero) on success and non-zero on failure.
 */
int FIFISetConversionCallback ( const char* adapterId, FP_FIFIConversionCallback cbFunc, void* userdata );

/*! \brief Set callback function to receive properties extracted by the conversion process.
	 \param adapterId The ID of the adapter for which to set callback function for.
	 \param cbFunc Callback function
	 \param userdata A pointer to any data the application wishes to be passed along to the callback function.
    \returns 0 (zero) on success and non-zero on failure.
 */
int FIFISetPropertyCallback ( const char* adapterId, FP_FIFIPropertyCallback cbFunc, void* userdata );

/*! \brief Check if a file can be converted by the specified adapter.
    \param adapterId The ID of an adapter
    \param targetFormat The output format 
    \param filename Name of file to convert
    \returns 0 (zero) on success, -1 on failure, and 1 to indicate "maybe".

    The provided file name does not necessarily correspond to a
    physical file. 
 */
EConvertAbility FIFICheckConversion ( const char* adapterId, EAdapterOutputType targetFormat, const char* filename );

/*! \brief Open a local file for conversion.
    \param adapterId The ID of the adapter to use for processing the file.
    \param targetFormat The desired output format of the conversion.
    \param filename Name of file to open.
    \returns A conversion handle for the file, or NULL on failure.
 */
HADAPTER FIFIOpenFile( const char* adapterId, EAdapterOutputType targetFormat, const char* filename );

/*! \brief Assign file data from memory for conversion.
    \param adapterId The ID of the adapter to use for processing the file.
    \param targetFormat The desired output format of the conversion.
    \param filename The file name associated with the file data to convert.
    \param fpRead An application-provided function for reading the file data.
    \param userdata A pointer to any data the application wishes to be passed along to the read function.
    \returns A conversion handle for the file, or NULL on failure.
 */
HADAPTER FIFIOpenData ( const char* adapterId, EAdapterOutputType targetFormat, const char* filename, FP_FIFIRead fpRead, void* userdata );

/*! \brief Assigns hit information to apply to the conversion.
    \param adapterId The ID of the adapter useed for processing the file.
    \param hFileConvHandle Handle return by successful call to FIFIOpenFile or FIFIOpenData.
    \param hits A pointer to a HitInfo structure returned by FIFIAllocateHits
    \returns 0 (zero) on success and non-zero on failure.

    This function is optional for adapters to implement. If the adapter does not
    implement a function for assigning hit information, this function will return the
    FIFI_ERR_NOT_IMPLEMENTED error code. Applications must not regard this return code
    as an error.

    Best practice for checking if an error really has occurred when calling this 
    function is to check the return code from the FIFIGetLastError function. If it
    is FIFI_ERR_SUCCESS, then the application may continue its processing.
 */
int FIFISetHitInfo( const char* adapterId, HADAPTER hFileConvHandle, HitInfo* hits);

/*! \brief End the conversion process.
    \param adapterId The ID of the adapter useed for processing the file.
    \param hFileConvHandle Handle return by successful call to FIFIOpenFile or FIFIOpenData.
    \returns 0 (zero) on success and non-zero on failure.
 */
int FIFIClose ( const char* adapterId, HADAPTER hFileConvHandle );

/*! \brief Converts a file opened by FIFIOpenFile or FIFIOpenData.
    \param adapterId The ID of the adapter with which to perform the conversion. Must be the same as used to open the file.
    \param hFileConvHandle Handle return by successful call to FIFIOpenFile or FIFIOpenData.
    \param filename Target filename (pass NULL to retrieve the conversion results via callbacks)
    \returns 0 (zero) on success and non-zero on failure.
 */
int FIFIConvert ( const char* adapterId, HADAPTER hFileConvHandle, const char* filename );

//@}




/** @name Adapter API
 * These function pointer types declare the interface that file filter
 * adapters must implement.
 */
//@{
	
/*! \brief Initialize the adapter library
    \returns 0 (zero) on success and non-zero on failure.

    This OPTIONAL function must (if present) be exported from the adapter as "InitializeAdapter".
 */
typedef int (* FP_FIFIInitializeAdapter ) ( );

/*! \brief Uninitialize the adapter library
    \returns 0 (zero) on success and non-zero on failure.

    This OPTIONAL function must (if present) be exported from the adapter as "UninitializeAdapter".
 */
typedef int (* FP_FIFIUninitializeAdapter ) ( );

/*! \brief Query the adapter for the API version it was built for
 *  \returns The value of the FIFI_API_INTERFACE_VERSION macro as it was when the adapter was built.
 *
 *  This REQUIRED function must be exported from the adapter as "GetAdapterApiVersion".
 */
typedef int (* FP_FIFIGetAdapterApiVersion) ();

/*! \brief Assign a runtime setting (a.k.a. "property") to the adapter
    \returns 0 (zero) on success and non-zero on failure.

    This OPTIONAL function must (if present) be exported from the adapter as "SetAdapterProperty".
 */
typedef int (* FP_FIFISetAdapterProperty)( const char* name, const char* value );

/*! \brief Requests the adapter to use the specified encoding in its output.
    \param encoding The name of the encoding to produce (e.g. "UTF-8" or "ISO-8859-1").
    \returns 0 (zero) on success and non-zero on failure.

    The function must be exported from the adapter as "SetConvertTextEncoding".
 */
typedef int (* FP_FIFISetTextEncoding ) ( const char* encoding );

/*! \brief Requests the adapter to call the specified function when there is data output of the specified type.
    \param cbFunc Callback function pointer
    \param userdata User data to send to the callback function
    \returns 0 (zero) on success and non-zero on failure.

    The function must be exported from the adapter as "SetConversionCallback".
 */
typedef int (* FP_FIFISetConversionCallback ) ( FP_FIFIConversionCallback cbFunc, void* userdata );

/*! \brief Requests the adapter to call the specified function when there is data output of the specified type.
    \param cbFunc Callback function pointer
    \param userdata User data to send to the callback function
    \returns 0 (zero) on success and non-zero on failure.

    The function must be exported from the adapter as "SetPropertyCallback".
 */
typedef int (* FP_FIFISetPropertyCallback ) ( FP_FIFIPropertyCallback cbFunc, void* userdata );

/*! \brief Requests the adapter to call the specified function for logging.
    \param cbFunc Callback function
    \param loglevel The current log level. Avoid sending log messages with a level higher than this.
    \param userdata A pointer to any data the application wishes to be passed along to the callback function.
    \returns 0 (zero) on success and non-zero on failure.

    This OPTIONAL function must (if present) be exported from the adapter as "SetLogCallback".
 */
typedef int (* FP_FIFISetLogCallback ) ( FP_FIFILog cbFunc, int loglevel, void* userdata );

/*! \brief Check if a file can be converted by the adapter
    \param targetFormat The output format 
    \param filename Name of file to convert
    \returns 0 (zero) on success, -1 on failure, and 1 to indicate "maybe".

	 The provided file name does not necessarily correspond to a
	 physical file. 
 */
typedef EConvertAbility (* FP_FIFICheckConversion)(EAdapterOutputType targetFormat, const char* filename);

/*! \brief Opens a file on the file system for conversion to the specified target format.
    \param targetFormat The output format 
    \param filename Name of file to convert
    \return A file conversion handle or NULL if the call fails

    The function must be exported from the adapter as "ConversionOpenFile".
 */
typedef HADAPTER (* FP_FIFIOpenFile) ( EAdapterOutputType targetFormat, const char* filename );

/*! \brief Opens a file via a user-specified read callback function.
    \param targetFormat The output format 
    \param filename Name of the file (if known)
    \param fpRead A file pointer to a read function 
    \param userdata User data to send to the callback function
    \return A file conversion handle or NULL if the call fails

    The caller (not the adapter) is responsible for that the file data is read strictly
    sequentially from the start of the file. The logic for hopping back and forth in the 
    file (if required) is the responsibility of the adapter.

    The function must be exported from the adapter as "ConversionOpenData".
 */
typedef HADAPTER (* FP_FIFIOpenData) ( EAdapterOutputType targetFormat, const char* filename, FP_FIFIRead fpRead, void* userdata );

/*! \brief Assigns hit information to apply to the conversion.
	 \param hFileConvHandle An open file conversion handle.
    \param hits A pointer to a HitInfo structure.
    \returns 0 (zero) on success and non-zero on failure.

    This OPTIONAL function must be exported from the adapter as "SetHitInfo".

	 If hit information is provided, the adapter must apply hit
	 highlighting output of the types FIFI_OUTPUT_HTML, FIFI_OUTPUT_MHT and
	 FIFI_OUTPUT_PDF.

	 If the adapter does not support highlighting, this method will fail.
 */
typedef int (* FP_FIFISetHitInfo)( HADAPTER hFileConvHandle, HitInfo* hits);

/*! \brief Close down a file conversion operation.
    \param hFileConvHandle A file conversion handle
    \returns 0 (zero) on success and non-zero on failure.

    The function must be exported from the adapter as "ConversionClose".
 */
typedef int (* FP_FIFIClose) ( HADAPTER hFileConvHandle );

/*! \brief Perform a file conversion
    \param hFileConvHandle An open file conversion handle.
	 \param filename Target filename (pass NULL to retrieve the conversion results via callbacks)

    The function must be exported from the adapter as "Convert".
 */
typedef int (* FP_FIFIConvert) ( HADAPTER hFileConvHandle, const char* filename );

/*! \brief Return the last error on the current thread from the adapter .
    \param errBuf Buffer to receieve the error message.
    \param bufSize The size of the errBuf
    \returns The adapter-specific last error code

    The function must be exported from the adapter as "GetLastConversionError".
 */
typedef int (* FP_FIFIGetLastError) ( char* errBuf, int bufSize );
	
//@}


#ifdef __cplusplus
}
#endif

#endif
