#!/bin/sh

# TRIPcof installer script for Unix platforms.
#
#  Created : 2015-07-20

#  Modified:
#
#  How to install TRIPcof by running this script:
#       1 - create a directory where TRIPcof will be installed
#       2 - change directory to the directory just created
#       3 - copy the tar file off the distribution media to .
#       4 - untar the TRIPcof tar file
#	4 - change directory to the <version>/bin directory
#	5 - run install: ./install_cof.sh as root

VER_MAJOR=1
VER_MINOR=7
VER_SERVICE=1
VER_PATCH=0
VER_SPECIAL=

VERS_DIR=v171

# Minimum required TRIPsystem version
REQ_TDBS_VER_MAJ=8
REQ_TDBS_VER_MIN=3
REQ_TDBS_VER_SVC=0

# Recommended Java version
REQ_JAVA_VERSION=17


VERSION=v$VER_MAJOR.$VER_MINOR-$VER_SERVICE:$VER_PATCH$VER_SPECIAL
LINK_DIR=/usr/local/trip/cof
OS=`uname`

echo ""
echo " *************  TRIPcof Installation script $VERSION  *************"
echo ""
echo ""

WHOM=`(whoami) 2> /dev/null`
if [ $? != "0" ] ; then
  if [ -f /usr/ucb/whoami ] ; then
    WHOM=`/usr/ucb/whoami`
  fi
fi

# This script needs root privileges unless we're on macOS.
if [ "$OS" != "Darwin" ]; then
   if [ "$WHOM" != root ]; then
      echo "You must be root to run this script..."
      echo ""
      exit 1
   fi
fi

if [ ! -f `pwd`/install_cof.sh ]; then
   echo Error: Current directory must be the one in which this script is located.
   echo ""
   exit 1
fi

# The name and location of the TRIPsystem configuration file
if [ -f /usr/local/trip/sys/conf/tdbs.conf ]; then
    TDBS_CONF=/usr/local/trip/sys/conf/tdbs.conf
fi

# Standalone install (w/o TRIPsystem) ?
if [ "$TDBS_CONF" = "" ] || [ ! -f $TDBS_CONF ]; then
   echo "Error: Unable to find a TRIPsystem installation on this machine."
   echo ""
   exit 1
fi

# Suitable version of TRIPsystem ?
TDBS_EXE=`grep ^TDBS_EXE $TDBS_CONF| cut -d= -f2 | head -1`
if [ -d "$TDBS_EXE" ] && [ -f "$TDBS_EXE/trip" ]; then
	TDBS_VERSION=`$TDBS_EXE/trip --version`
	TDBS_VER_MAJ=`echo $TDBS_VERSION | cut -d. -f1`
	TDBS_VER_MIN=`echo $TDBS_VERSION | cut -d. -f2- | cut -d- -f1`
	TDBS_VER_SVC=`echo $TDBS_VERSION | cut -d- -f2- | cut -d: -f1 | tr -d [a-zA-Z]+`
else
	TDBS_VERSION="not found"
fi
if [ "$TDBS_VER_MAJ" = "" ]; then
	TDBS_VER_MAJ=0
fi
if [ "$TDBS_VER_MIN" = "" ]; then
	TDBS_VER_MIN=0
fi
if [ "$TDBS_VER_SVC" = "" ]; then
	TDBS_VER_SVC=0
fi
TRIPSYSTEM_IS_COMPATIBLE=0
if [ $TDBS_VER_MAJ -lt $REQ_TDBS_VER_MAJ ]; then
   # The current major TRIPsystem version is less than the one required
   TRIPSYSTEM_IS_COMPATIBLE=1
fi
if [ $TDBS_VER_MAJ -eq $REQ_TDBS_VER_MAJ ] && [ $TDBS_VER_MIN -lt $REQ_TDBS_VER_MIN ]; then
   # The current main TRIPsystem version is less than the one required
   TRIPSYSTEM_IS_COMPATIBLE=1
fi
if [ $TDBS_VER_MAJ -eq $REQ_TDBS_VER_MAJ ] && [ $TDBS_VER_MIN -eq $REQ_TDBS_VER_MIN ] && [ $TDBS_VER_SVC -lt $REQ_TDBS_VER_SVC ]; then
   # The current TRIPsystem service version is less than the one required
   TRIPSYSTEM_IS_COMPATIBLE=1
fi

if [ $TRIPSYSTEM_IS_COMPATIBLE -ne 0 ]; then
   echo "Error: Your TRIPsystem version ($TDBS_VERSION) is incompatible."
   echo "This version of TRIPcof requires TRIPsystem $REQ_TDBS_VER_MAJ.$REQ_TDBS_VER_MIN"
   if [ $REQ_TDBS_VER_SVC -gt 0 ]; then
      echo "with service release $REQ_TDBS_VER_SVC or later"
   fi
   echo ""
   exit 1
fi


CORRECT_INSTALL_DIR=0
echo ""
if [ ! -d bin ]; then
   if [ -f install_cof.sh ]; then
      cd ..
      if [ ! -d bin ]; then
         CORRECT_INSTALL_DIR=1
      fi
   else
      CORRECT_INSTALL_DIR=1
   fi
fi
if [ $CORRECT_INSTALL_DIR -ne 0 ]; then
   echo "Please run this script from one of the following directories:"
   echo ""
   echo "  <extract_dir>"
   echo "  <extract_dir>/$VERS_DIR"
   echo "  <extract_dir>/$VERS_DIR/bin"
   echo ""
   echo "where <extract_dir> is the directory from which you issued the"
   echo "tar command to read the distribution media."
   echo "Exiting the installation procedure"
   echo ""
   exit 1
fi

COF_INSTALLDIR=`pwd`
COF_CONFDIR=$COF_INSTALLDIR/conf


##########################################################################
# If we're running on Linux, adjust the fapolicyd database as needed.
# This is required in order to avoid "operation not permitted" errors
# when using the libraries provided in the installation's lib directory
# and when TRIPcof programs and scripts run other TRIPcof programs.
#

if [ "$OS" = "Linux" ]; then
   $COF_INSTALLDIR/bin/chfapolicy_cof.sh -a -p $COF_INSTALLDIR
   echo ""
fi


# Make sure we have write access the config directory
if [ ! -w $TDBS_CONF ]; then
   echo Error: $TDBS_CONF must be writable.
   echo ""
   exit 1
fi

if [ `echo "X\c"` = X ]; then
   C1=
   C2='\c'
elif [ `echo -n X` = X ]; then
   C1='-n'
   C2=
else
   C1=
   C2=
fi

yesno_valid()
{
   inputval=$1
   if [ "$inputval" == "y" ] || [ "$inputval" == "Y" ] || \
      [ "$inputval" == "n" ] || [ "$inputval" == "N" ]
   then
      return 0
   elif [ "$2" != "" ] && [ "$inputval" = "?" ]; then
      return 0
   else
      return 1
   fi
}

yesno()
{
   yesno_result=-1
   yesno_prompt=$1
   yesno_default=$2
   yesno_help=$3
   if [ "$yesno_default" = "Y" ]; then
      yesno_choice="[Y|n"
   elif [ "$yesno_default" = "N" ]; then
      yesno_choice="[y|N"
   else
      yesno_choice="[y|n"
   fi
   if [ "$yesno_help" != "" ]; then
      yesno_choice="$yesno_choice|?"
   fi
   yesno_choice="$yesno_choice]"
   doloop=0
   while [ "$doloop" = "0" ]
   do
      echo $C1 "$yesno_prompt $yesno_choice $C2"
      read answ
      if [ "$answ" != "" ] ; then
         if yesno_valid $answ "$yesno_help"; then
            printf "\n****** Answer: $answ\n"
         else
            continue
         fi
      elif [ "$answ" = "" ] && [ "$yesno_default" != "" ]; then
         printf "\n****** Answer: default [= $yesno_default]\n"
         answ=$yesno_default
      else
         continue
      fi
      case $answ in
         \?)
            if [ "$yesno_help" != "" ]; then
               eval $yesno_help
            fi
            ;;
         Y|y)
            yesno_result=0
            doloop=1
            ;;
         N|n)
            yesno_result=1
            doloop=1
            ;;
         *)
            ;;
      esac
   done
   return $yesno_result
}


# Check and optionally install dependencies
bin/dependencies_cof.sh --check --from-installer
DEPRC=$?
if [ $DEPRC -ne 0 ]; then
   if [ "$OS" = "Linux" ]; then
      echo ""
      if yesno "Do you wish to attempt to install the missing dependences?" Y ; then
         bin/dependencies_cof.sh --install --from-installer | tee -a $LOGFILE
         DEPRC=$?
      fi
   else
      # Dependency installation on non-Linux operating systems must
      # be done manually.
      DEPRC=1
   fi
fi
if [ $DEPRC -ne 0 ]; then
   echo ""
   echo "Exiting the installation procedure" | tee -a $LOGFILE
   echo ""
   exit 1
fi
echo ""


# Path to the 'bin' directory of TRIPsystem
TDBS_EXE=`grep "^TDBS_EXE" $TDBS_CONF | tr -d '\r' | ( read r; echo $r | cut -d= -f2 )`

# Detect if TRIPview or TRIPview-C are installed and ask
# if it is OK to proceed.
HAS_OLD_VIEW=0
OLD_VIEW_SYM=`grep viewsfilters.so $TDBS_CONF | grep -v \^# | ( read r; echo $r | cut -d= -f1 )`
if [ "x$OLD_VIEW_SYM" != "x" ]; then
   OLD_VIEW_LIB=`grep viewsfilters.so $TDBS_CONF | grep -v \^# | ( read r; echo $r | cut -d= -f1 )`
   $TDBS_EXE/queryrcs TDBS_ASELIBS | cut -d\" -f2 | grep $OLD_VIEW_SYM 2> /dev/null > /dev/null
   if [ $? -eq 0 ]; then
      HAS_OLD_VIEW=1
   fi
fi
if [ $HAS_OLD_VIEW -eq 1 ]; then
   echo An installation of TRIPview or TRIPview-C has been detected.
   echo It will be disabled in the TRIPsystem configuration if you proceed.
   if ! yesno "Do you wish to proceed?" Y ; then
      echo ABORT
      exit 1
   fi
fi

COF_LINE=`grep "COF_ASE_FIFI" $TDBS_CONF | tr -d '\r' | grep -v \^# | cut -d= -f1`
if [ "x$COF_LINE" != "x" ]; then
   IS_UPDATE=1
else
   IS_UPDATE=0
fi

OLD_COF_HOME=

if [ $IS_UPDATE -eq 1 ]; then
   COF_LIBPATH=`$TDBS_EXE/queryrcs COF_ASE_FIFI | cut -d\" -f2`
   COF_LIBDIR=`dirname $COF_LIBPATH`
   
   OLD_COF_HOME=`dirname $COF_LIBDIR 2> /dev/null`
   if [ "$OLD_COF_HOME" = "$COF_INSTALLDIR" ]; then
      OLD_COF_HOME=
   fi
   
   # If old installation is broken, there is no point in asking
   # for permission to disable it.
   if [ -f $COF_LIBPATH ]; then

      if [ "$COF_LIBDIR" = "`cd $COF_INSTALLDIR/lib; pwd`" ]; then
         echo This module has already been installed.
         if ! yesno "Do you want to reinstall TRIPcof?" N ; then
            echo ABORT
            exit 1
         fi
      fi

      echo Existing TRIPcof installation in `cd $COF_LIBDIR/..; pwd` will be disabled.
      if ! yesno "OK to proceed?" Y ; then
         echo ABORT
         exit 1
      fi

      if [ -f "$COF_LIBDIR/../bin/tikamond" ]; then
         echo Terminating running tikamond processes
         $COF_LIBDIR/../bin/tikamond -kill
      fi
      
   else
      echo New installation
   fi
else
   echo New installation
fi

if [ "`uname`" = "SunOS" ]; then
   FINDDEPTH=
else
   FINDDEPTH="-maxdepth 1"
fi

# Set the umask for the duration of the script
OUMASK=`umask`
umask 022


# change user ownership

doloop=0
while [ "$doloop" = "0" ]
do
  echo ""
  echo "Which user should own the TRIPcof files [root]? $C"
  read ans
  case $ans in
  \?)
     echo ""
     echo ""
     echo "   Enter the name of the user who should own all files"
     echo "   and directories in the TRIPcof directory structure."
     echo ""
     echo "   The default is root."
     echo ""
     ;;
  *)
     if [ "$ans" = "" ]; then
       owner=root
     else
       owner=$ans
     fi
     chown -R $owner $COF_INSTALLDIR
     if [ $? -eq 0 ]; then
        doloop=1
     fi
  esac
done
echo "User $owner given ownership of the TRIPcof files"

# change group ownership

doloop=0
while [ "$doloop" = "0" ]
do
  echo ""
  echo "Which group should own the TRIPcof files? $C"
  read ans
  case $ans in
  \?)
     echo ""
     echo ""
     echo "   Enter the name of the group that should own all files"
     echo "   and directories in the TRIPcof directory structure."
     echo ""
     ;;
  "")
     ;;
  *)
     owngroup=$ans
     chgrp -R $owngroup $COF_INSTALLDIR
     if [ $? -eq 0 ]; then
        doloop=1
     fi
  esac
done
echo "Group $owngroup given group ownership of the TRIPcof files"

if [ -d $COF_INSTALLDIR/bin ]; then
   chmod 755 $COF_INSTALLDIR/bin/*
fi

if [ -d $COF_INSTALLDIR/lib ]; then
   chmod 755 $COF_INSTALLDIR/lib/*
fi

if [ -f /tmp/cof2rcs ]; then
   rm -f /tmp/cof2rcs
   touch /tmp/cof2rcs
fi

if [ -f $COF_INSTALLDIR/TRIPrcs.mods ]; then
   rm -f $COF_INSTALLDIR/TRIPrcs.mods
   touch $COF_INSTALLDIR/TRIPrcs.mods
fi

COF_LIB=`find $COF_INSTALLDIR/lib -name '*fifiase*' -print`
if [ "$COF_LIB" = "" ]; then
   echo "Error: Could not find FIFI ASE library!"
   echo ""
   exit 2
fi
COF_LIBPATH1=$COF_LIB


echo ""
echo Modifying $TDBS_CONF ...

# Utility function to check if a symbol is set in TRIPrcs
findSymbol()
{
   cat $1 | cut -d= -f1 | grep $2 | grep -v \^# > /dev/null
   return $?
}

# Add an item to a comma-separated list if it does not already exist in it.
listAddUnique()
{
   if [ "$2" = "" ]; then
      echo $1
   else
      echo $1 | grep $2 2> /dev/null 1> /dev/null
      if [ $? -eq 0 ]; then
         echo $1
      else
         if [ "" = "$1" ]; then
            echo $2
         else
            echo $2,$1
         fi
      fi
   fi
}

# Remove all instances of a symbol from a comma separated list
listRemoveItem()
{
   echo $1 | tr ',' '\n'| while read sym
   do 
      if [ "x$sym" != "x$2" ]; then 
         echo ,$sym | tr -d '\n'
      fi; 
   done | cut -d, -f2-
}

cat $TDBS_CONF | (

   IFS=\\
   while read LINE; do

      RCSSYM=`echo $LINE|cut -d= -f1`

      # Modify existing FIFI ASE library symbol
      if [ "$RCSSYM" = "COF_ASE_FIFI" ]; then
         echo "COF_ASE_FIFI=$COF_LIBPATH1" >> /tmp/cof2rcs
         echo "COF_ASE_FIFI=$COF_LIBPATH1" >> $COF_INSTALLDIR/TRIPrcs.mods
         continue
      fi

      # Modify existing config dir path
      if [ "$RCSSYM" = "COF_CONFIG_DIR" ]; then
         echo "COF_CONFIG_DIR=$COF_CONFDIR" >> /tmp/cof2rcs
         echo "COF_CONFIG_DIR=$COF_CONFDIR" >> $COF_INSTALLDIR/TRIPrcs.mods
         continue
      fi
 
      # Modify existing TDBS_ASELIBS setting
      if [ "$RCSSYM" = "TDBS_ASELIBS" ]; then
         ASELIBS1=`$TDBS_EXE/queryrcs TDBS_ASELIBS|awk -F\" '{print $2}'`
         if [ "x$OLD_VIEW_SYM" != "x" ]; then
            # Old TRIPview/-C symbol exists - remove it
            ASELIBS2=`listRemoveItem $ASELIBS1 $OLD_VIEW_SYM | tr -d '[[:space:]]'`
         else
            ASELIBS2=$ASELIBS1
         fi
	 ASELIBS=`listAddUnique $ASELIBS2 COF_ASE_FIFI`
       
         echo "# $LINE" >> /tmp/cof2rcs
         echo "TDBS_ASELIBS=$ASELIBS" >> /tmp/cof2rcs
         echo "TDBS_ASELIBS=$ASELIBS" >> $COF_INSTALLDIR/TRIPrcs.mods
         continue
      fi

      # Comment out the old TRIPview/TRIPview-C library symbol
      if [ "x$OLD_VIEW_SYM" != "x" ]; then
         if [ "$RCSSYM" = "$OLD_VIEW_SYM" ]; then
            echo "# $LINE" >> /tmp/cof2rcs
            echo "# $LINE" >> $COF_INSTALLDIR/TRIPrcs.mods
            continue
         fi
      fi

      # Just perform a copy of all other lines 
      echo $LINE >> /tmp/cof2rcs
   done
)

CONFIG_ADDED=0

# If ASELIBS setting is not found - add it.
findSymbol /tmp/cof2rcs TDBS_ASELIBS
if [ $? -ne 0 ]; then
   echo "TDBS_ASELIBS=COF_ASE_FIFI" >> /tmp/cof2rcs
   echo "TDBS_ASELIBS=COF_ASE_FIFI" >> $COF_INSTALLDIR/TRIPrcs.mods
   CONFIG_ADDED=1
fi

# If COF_CONFIG_DIR setting is not found - add it.
findSymbol /tmp/cof2rcs COF_CONFIG_DIR
if [ $? -ne 0 ]; then
   echo "COF_CONFIG_DIR=$COF_CONFDIR" >> /tmp/cof2rcs
   echo "COF_CONFIG_DIR=$COF_CONFDIR" >> $COF_INSTALLDIR/TRIPrcs.mods
   CONFIG_ADDED=1
fi

# If FIFI ASE library symbol setting is not found - add it.
findSymbol /tmp/cof2rcs COF_ASE_FIFI
if [ $? -ne 0 ]; then
   echo "COF_ASE_FIFI=$COF_LIBPATH1" >> /tmp/cof2rcs
   echo "COF_ASE_FIFI=$COF_LIBPATH1" >> $COF_INSTALLDIR/TRIPrcs.mods
   CONFIG_ADDED=1
fi

if [ $CONFIG_ADDED -eq 1 ]; then
   echo "" >> /tmp/cof2rcs
fi

# Introduce changes
cp $TDBS_CONF $TDBS_CONF.cof$VER_MAJOR$VER_MINOR$VER_PATCH.bkp
cp /tmp/cof2rcs $TDBS_CONF
rm /tmp/cof2rcs

echo ""
echo "Initializing TRIPcof configuration files"

# Copy configuration files from previous installation
if [ "$$OLD_COF_HOME" != "" ]; then

  if [ -d "$OLD_COF_HOME/conf/cfw.importds.d" ]; then
    mkdir -p $COF_INSTALLDIR/conf/cfw.importds.d
    find $OLD_COF_HOME/conf/cfw.importds.d -type f -name "*.conf" | while read ff; do
      if [ -f $COF_INSTALLDIR/conf/cfw.importds.d/$ff ]; then
        echo "  keeping existing $ff"
      else
        echo "  migrating $ff"
        cp $ff $COF_INSTALLDIR/conf/cfw.importds.d/
      fi
    done
  fi
  chmod o+rx $COF_INSTALLDIR/conf/cfw.importds.d
  find $COF_INSTALLDIR/conf/cfw.importds.d -name "*.conf" | while read cfg; do chmod o+r $COF_INSTALLDIR/conf/cfw.importds.d/*.conf; done

  if [ -f "$OLD_COF_HOME/conf/std_properties.conf" ]; then
    diff -w $OLD_COF_HOME/conf/std_properties.conf $COF_INSTALLDIR/conf/std_properties.conf.tmpl 2>&1 > /dev/null
    if [ $? -ne 0 ] && [ ! -f $COF_INSTALLDIR/conf/std_properties.conf ]; then
      echo "  migrating $OLD_COF_HOME/conf/std_properties.conf"
      cp $OLD_COF_HOME/conf/std_properties.conf $COF_INSTALLDIR/conf
    fi
  fi
  if [ -f "$OLD_COF_HOME/conf/std_propmap.conf" ]; then
    diff -w $OLD_COF_HOME/conf/std_propmap.conf $COF_INSTALLDIR/conf/std_propmap.conf.tmpl 2>&1 > /dev/null
    if [ $? -ne 0 ] && [ ! -f $COF_INSTALLDIR/conf/std_propmap.conf ]; then
      echo "  migrating $OLD_COF_HOME/conf/std_propmap.conf"
      cp $OLD_COF_HOME/conf/std_propmap.conf $COF_INSTALLDIR/conf
    fi
  fi
fi

# Initialize configuration files from templates
find $COF_INSTALLDIR/conf -type f -name "*.conf.tmpl" | while read ff; do
   ffl=`echo $ff|wc -c`
   ffl=`expr $ffl - 6`
   ffc=`echo $ff|cut -c1-$ffl`
   if [ ! -f $ffc ]; then
      cp -p $ff $ffc
      if [ $? -eq 0 ]; then
         echo "  $ffc"
      fi
   fi
done

# Look for available Java VMs. First scan for highest avalable using the
# alternatives feature, and second check the default version (in case it
# is not installed using the system's package manager).

JVM_VERSION=0
JVM=""
JVMINFO=`$COF_INSTALLDIR/bin/showjvm.sh | head -n 1`
if [ "$JVMINFO" != "" ]; then
   JVM=`echo $JVMINFO | cut -d" " -f2-`
   JVM_VERSION=`echo $JVMINFO | cut -d\. -f1`
   if [ "$JVM_VERSION" = "" ]; then
      JVM_VERSION=0
   fi
fi


if [ -f "$JVM" ]; then
   echo ""
   echo "Found JVM: $JVM"
   if [ $JVM_VERSION -lt $REQ_JAVA_VERSION ]; then
      echo "WARNING: Recommended Java version is $REQ_JAVA_VERSION, but you have $JVM_VERSION"
   fi

   # Edit JVM property in fifi.conf
   if [ -f /tmp/fificonfedit.tmp ]; then rm -f /tmp/fificonfedit.tmp; fi
   touch /tmp/fificonfedit.tmp
   cat $COF_INSTALLDIR/conf/fifi.conf | (
      IFS=\\
      while read LINE; do

         RCSSYM=`echo $LINE|cut -d= -f1`

         # Modify existing JavaVM symbol
         if [ "$RCSSYM" = "JavaVM" ]; then
            echo "JavaVM=$JVM" >> /tmp/fificonfedit.tmp
            echo "Configured TRIPcof to use found JVM for Java based file filter adapters"
            continue
         fi

         # Just perform a copy of all other lines 
         echo $LINE >> /tmp/fificonfedit.tmp
      done
   )
   cp /tmp/fificonfedit.tmp $COF_INSTALLDIR/conf/fifi.conf
   rm -f /tmp/fificonfedit.tmp

   # Edit JVM property in cfw.conf
   if [ -f /tmp/cfwconfedit.tmp ]; then rm -f /tmp/cfwconfedit.tmp; fi
   touch /tmp/cfwconfedit.tmp
   cat $COF_INSTALLDIR/conf/cfw.conf | (
      IFS=\\
      while read LINE; do

         RCSSYM=`echo $LINE|cut -d= -f1`

         # Modify existing JavaVM symbol
         if [ "$RCSSYM" = "JavaVM" ]; then
            echo "JavaVM=$JVM" >> /tmp/cfwconfedit.tmp
            echo "Configured TRIPcof to use found JVM for Java based connectors"
            continue
         fi

         # Just perform a copy of all other lines 
         echo $LINE >> /tmp/cfwconfedit.tmp
      done
   )
   cp /tmp/cfwconfedit.tmp $COF_INSTALLDIR/conf/cfw.conf
   rm -f /tmp/cfwconfedit.tmp
   
else
   echo "WARNING: No JVM found. TRIPcof will not work properly!"
fi
echo ""

# Create link directory if not already present
if [ ! -d $LINK_DIR ]; then
   mkdir -p $LINK_DIR
   echo "Creating link directory: $LINK_DIR"
else
   chmod o+rx $LINK_DIR
   if [ 0 -ne `ls -1 $LINK_DIR/lib 2> /dev/null | wc -l` ]; then
      rm -f $LINK_DIR/lib
      echo "Current lib directory link removed from $LINK_DIR"
   fi
fi

# Make links (used by add-on adapters)
echo "Creating new lib directory link: $LINK_DIR/lib -> $COF_INSTALLDIR/lib"
if [ -d $LINK_DIR/lib ]; then rm $LINK_DIR/lib; fi
ln -s -f $COF_INSTALLDIR/lib $LINK_DIR/lib
echo "Creating new conf directory link: $LINK_DIR/conf -> $COF_INSTALLDIR/conf"
if [ -d $LINK_DIR/conf ]; then rm $LINK_DIR/conf; fi
ln -s -f $COF_INSTALLDIR/conf $LINK_DIR/conf

# Reset umask
umask $OUMASK

echo ""
echo "Installation of TRIPcof $VERSION complete."
echo ""

# For MacOS installations, check if the JRE/JDK is configured for JNI
# use. If not, any use of Java adapters (e.g. icepdf for HTML conversion
# of PDFs) will abort procesing with the rather misleading error message
# "No Java runtime present, requesting install".
if [ "`uname`" = "Darwin" ] && [ -f /usr/libexec/java_home ]; then
JHOME=`/usr/libexec/java_home 2> /dev/null`
   if [ "$JHOME" != "" ]; then
      JPLIST=`dirname $JHOME`/Info.plist
   fi
   if [ -f "$JPLIST" ]; then
      xmllint --xpath '//dict/key[text()="JVMCapabilities"]/following-sibling::*[position()=1]' $JPLIST | xmllint --xpath "//*[not(comment())]" - | grep -q JNI 2> /dev/null
      if [ $? -ne 0 ]; then
         echo "WARNING"
         echo "------------------------------------------------------------------------------"
         echo "Your Java environment is not configured for JNI use."
         echo "This is likely to cause the Java-based functionality in TRIPcof not to"
         echo "function properly."
         echo ""
         echo "You should manually enable JNI by editing the file:"
         echo "$JPLIST"
         echo "and add JNI as an option for JVMCapabilities."
         echo "------------------------------------------------------------------------------"
         echo ""
      fi
   fi
fi

