#!/bin/sh
#
# USAGE: chfapolicy_cof.sh [OPTIONS]
#
# OPTIONS:
#    -h|--help       Show this help text
#    -r|--remove     Remove TRIPcof fapolicyd settings
#    -a|--add        Install TRIPcof fapolicyd settings
#
# This script add and removes TRIPcof related entries for fapolicyd
# trust database on Linux systems on which fapolicyd is available. This
# script is normally run via the install and uninstall script.
#
# On systems where fapolicyd is in use, we must adjust the fapolicyd database.
# This is required in order to avoid "operation not permitted" errors
# when using the libraries provided in the installation's lib directory, and
# when TRIP programs and scripts run other TRIP programs (e.g. for batch
# job execution).
#

OPMODE=add
OPVERB="Adding to"
TRUSTFILE=tripcof

while [ "$1" != "" ]; do
   case "$1" in
      -h|--help|"")
         head -19 $0 | tail -18 | cut -c3-
         exit 0
         ;;
      -a|--add)
         OPMODE=add
         OPVERB="Adding to"
         ;;
      -r|--remove|"")
         OPMODE=delete
         OPVERB="Removing from"
         ;;
      -p|--install-dir)
         shift
         TPATH=$1
         ;;
      *)
         echo "ERROR: Invalid argument: $1"
         echo ""
         head -9 $0 | tail -8 | cut -c3-
         exit 99
         ;;
   esac
   shift
done


##########################################################################
# Can this script be run?
#

OSNAME=`uname`
if [ "$OSNAME" != "Linux" ]; then
   echo "The chfapolicy_cof.sh script is not supported on this operating system."
   exit 1
fi

WHOM=`(whoami) 2> /dev/null`
if [ "$WHOM" != root ]; then
   echo ""
   echo "You must be root to run the chfapolicy_cof.sh script!"
   echo "Exiting the uninstallation procedure"
   echo ""
   exit 1
fi

if [ "$TPATH" = "" ]; then
   TPATH=$(dirname `readlink -f /usr/local/trip/cof/lib` 2> /dev/null)
   if [ "$TPATH" = "" ] || [ ! -d "$TPATH" ]; then
      echo "ERROR: TRIPcof is not installed on this system."
      exit 2
   fi
fi

##########################################################################
# Is fapolicyd present?
#

echo "Configuring fapolicyd for TRIPcof ($OPMODE)"

FAPCLI=`which fapolicyd-cli 2>/dev/null`
if [ "$FAPCLI" = "" ]; then
   echo "No action required - fapolicyd is not installed on this system."
   exit 0
fi


##########################################################################
# Add or remove the fapolicyd settings as requested
#

FAPCNT=0

echo "$OPVERB fapolicyd: $TPATH/lib"
$FAPCLI -f $OPMODE $TPATH/lib --trust-file $TRUSTFILE 2>&1
if [ $? -eq 0 ]; then
   FAPCNT=`expr $FAPCNT + 1`
fi

echo "$OPVERB fapolicyd: $TPATH/bin"
$FAPCLI -f $OPMODE $TPATH/bin --trust-file $TRUSTFILE  2>&1
if [ $? -eq 0 ]; then
   FAPCNT=`expr $FAPCNT + 1`
fi

if [ $FAPCNT -gt 0 ]; then
   $FAPCLI --update 2>&1
   if [ $? -ne 0 ]; then
      echo "WARNING: Could not notify fapolicyd of changes. It may not be running."
      exit 0
   fi
   echo "Reconfiguration of fapolicyd complete."
else
   echo "No reconfiguration of fapolicyd needed."
fi

